% Myers et al. "Dynamically Linking Influenza Virus Infection Kinetics, Lung Injury, Inflammation, and Disease Severity"
% Figures 4B-E, 5A-D, S4A-F
% Author, copyright 2019: Amber M Smith, UTHSC
% amber.smith@uthsc.edu

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all;
close all;
warning off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

gray = [0.52 0.52 0.52];
IAcolor = [0 0.8 0.1];
IAtextcolor = [0 0.75 0.1];
tspan = 0:0.01:15;
msize = 22;
fsize = 28;
tscat = [1/6,1:12];
exl_col = [0.75 0.75 0.75];
ylim1 = 57;
ylim2 = 47;

CD8yticks = [5.5 6.0 6.5];
CD8_time = 0:1:12;

% Virus
Vdat = xlsread('Myersetal_SourceData', 1, 'B3:N12');
meanVdat = nanmean(Vdat);
sdVdat = nanstd(Vdat);

% CD8
Edat = xlsread('Myersetal_SourceData', 1, 'B15:N24');
meanEdat_Lin = nanmean(10.^Edat);
meanEdat = nanmean(Edat);
sdEdat = nanstd(Edat);

% Lung-specific CD8
Lung_Edat = xlsread('Myersetal_SourceData',1, 'B27:N31');
meanLungEdat_Lin=nanmean(10.^Lung_Edat);
meanLungEdat=nanmean(Lung_Edat);
sdLungEdat=nanstd(Lung_Edat);

% CD25-CD43+
% Censored: 0:3
Lung_EffPhen = xlsread('Myersetal_SourceData',1, 'B34:N38');
Lung_EffPhen_Lin_Mean=nanmean(10.^Lung_EffPhen);
Lung_EffPhen_Mean=nanmean(Lung_EffPhen);
Lung_EffPhen_SD=nanstd(Lung_EffPhen);

% CD43-CD69+CD62L-CD127+CD103+
% Censored: 0:5
Lung_MemPhen = xlsread('Myersetal_SourceData',1, 'B41:N45');
Lung_MemPhen_Mean=nanmean(Lung_MemPhen);
Lung_MemPhen_SD=nanstd(Lung_MemPhen);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Weights
WL_d0dX = xlsread('Myersetal_SourceData', 1, 'B75:L75');
WL_d0dX_SD = xlsread('Myersetal_SourceData', 1, 'B76:L76');

WL_CD8depl = xlsread('Myersetal_SourceData', 2, 'B13:L13');
WL_CD8depl_SD = xlsread('Myersetal_SourceData', 2, 'B14:L14');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Pathology
H_time = [0:1:10]';

H_active_all = xlsread('Myersetal_SourceData', 1, 'B48:L52');
H_active = nanmean(H_active_all(1:5,:))';
H_activeSD = nanstd(H_active_all(1:5,:))';

H_inactive_all = xlsread('Myersetal_SourceData', 1, 'B55:L59');
H_inactive = nanmean(H_inactive_all(1:5,:))';
H_inactiveSD = nanstd(H_inactive_all(1:5,:))';

H_total_all = H_active_all+H_inactive_all;
H_total=nanmean(H_total_all)';
H_totalSD=nanstd(H_total_all)';

predict_histol = dlmread('95CI_Lesions.txt');
CAUCmin = predict_histol(1,:);
CAUCmax =predict_histol(2,:);

H_interstial_all = xlsread('Myersetal_SourceData', 1, 'B62:L66');
H_interstial = nanmean(H_interstial_all(1:5,:))';
H_interstialSD = nanstd(H_interstial_all(1:5,:))';

H_alveolar_all = xlsread('Myersetal_SourceData', 1, 'B69:L73');
H_alveolar = nanmean(H_alveolar_all(1:5,:))';
H_alveolarSD = nanstd(H_alveolar_all(1:5,:))';

CIpreds = dlmread('Histol_predicted_minmax.txt');

CAUCmin = CIpreds(1,:);
CAUCmax = CIpreds(2,:);
IALmin = CIpreds(3,:);
IALmax = CIpreds(4,:);


AL_j = 600;
ALmin = zeros(size(CAUCmin)); ALmax = zeros(size(CAUCmin));
ALmin(:, 1:AL_j) = CAUCmin(:, 1:AL_j);  ALmax(:, 1:AL_j) = CAUCmax(:, 1:AL_j); 

flag = 1;
AL_j = 600;
while flag == 1
    ALmin(:, AL_j+1) = ALmin(:,AL_j) - 0.287/14.2;
    if ALmin(:,AL_j+1) <= 0
        ALmin(:,AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end
      
flag = 1;
AL_j = 600;
while flag == 1
    ALmax(:, AL_j+1) = ALmax(:,AL_j) - 0.287/14.2;
    if ALmax(:,AL_j+1) <= 0
        ALmax(:,AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end

TotLmin(1,:) = 14.2.*ALmin(1,:)  + 0.458.*IALmin(1,:);
TotLmax(1,:) = 14.2.*ALmax(1,:) +  0.458.*IALmax(1,:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Model 

% Parameters
bp0008 = importdata('params.0518.txt');
bestparams =bp0008(1,2:end-6);
for i = [1,2,3,7,8,9,10,13]
    bestparams(i) = 10.^bestparams(i);
end
lags=[bestparams(:,end-1),bestparams(:,end)];

% Model
tmax = 12.5;
tspan = 0:0.01:tmax; 
tscat = [1/6,1:tmax];
tscat2 = [0,1:tmax];
sol=dde23(@DDE_Tcell,lags,[1e7 75 0 0 0 0],[0 tmax],[],bestparams);
Estar=bestparams(end-2);
Ys = deval(sol, tspan)';
Vbest = Ys(:,4);
I2best = Ys(:,3);
Ebest = Ys(:,5);
Mbest = Ys(:,6);
CD8best = Ebest+Mbest+Estar;
Ebest_scaled = Ebest(:)/max(Ebest);

% CAUC
int_virus=cumtrapz(sol.x,sol.y(4,:));
int_virus_ = cumtrapz(tspan,Ys(:,4))./1e7;

int_infcell=cumtrapz(sol.x,sol.y(3,:));
int_infcell_ = cumtrapz(tspan,Ys(:,3))./1e7;

% adding on linear end to active lesion prediction 
ALbest = 14.2.*int_infcell_;
ALbest(:,600:end) = 0;
AL_j = 600;
flag = 1;
while flag == 1
    ALbest(AL_j+1) = ALbest(AL_j) - 0.287;
    if ALbest(AL_j+1) <= 0
        ALbest(AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end

%total 
TotLbest(:,1) = 0.458.*100*Ebest_scaled(:,1) + ALbest(:,1);

CIpreds = dlmread('Histol_predicted_minmax.txt');
CAUCmin = CIpreds(1,:);
CAUCmax = CIpreds(2,:);
IALmin = CIpreds(3,:);
IALmax = CIpreds(4,:);

AL_j = 600;
ALmin = zeros(size(CAUCmin)); ALmax = zeros(size(CAUCmin));
ALmin(:, 1:AL_j) = CAUCmin(:, 1:AL_j);  ALmax(:, 1:AL_j) = CAUCmax(:, 1:AL_j); 

flag = 1;
AL_j = 600;
while flag == 1
    ALmin(:, AL_j+1) = ALmin(:,AL_j) - 0.287/14.2;
    if ALmin(:,AL_j+1) <= 0
        ALmin(:,AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end
      
flag = 1;
AL_j = 600;
while flag == 1
    ALmax(:, AL_j+1) = ALmax(:,AL_j) - 0.287/14.2;
    if ALmax(:,AL_j+1) <= 0
        ALmax(:,AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end

TotLmin(1,:) = 14.2.*ALmin(1,:)  + 0.458.*IALmin(1,:);
TotLmax(1,:) = 14.2.*ALmax(1,:) +  0.458.*IALmax(1,:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Model - Effector + Memory Phenotypes

bp_eff = importdata('params.0618.txt')';

CD8_s =  bp_eff(end-1); 
CDM_s =  bp_eff(end); 
tau_Es = bp_eff(end-3);
tau_Ms = bp_eff(end-2);
lags_s=[tau_Es,tau_Ms];

sol_s=dde23(@DDE_Tcell,lags_s,[1e7 75 0 0 0 0],[0 tmax],[],bp_eff);

ts_s = sol_s.x';
Ys_s = deval(sol_s, tspan)';
Vs_s = Ys_s(:,4);
Es_s = Ys_s(:,5);
Ms_s = Ys_s(:,6);
As_s = Ys_s+CD8_s;
Bs_s = Ys_s+CDM_s;

% CAUC 
int_infcell_s = cumtrapz(sol_s.x,sol_s.y(3,:));
int_infcell_s_ = cumtrapz(tspan,Ys_s(:,3))./1e7;

Ebest_scaled_s = Es_s(:)/max(Es_s);

clear AL_j
ALbest_s = 14.2.*int_infcell_s_;
ALbest_s(:,600:end) = 0;
AL_j = 600;
flag = 1;
while flag == 1
    ALbest_s(AL_j+1) = ALbest_s(AL_j) - 0.287;
    if ALbest_s(AL_j+1) <= 0
        ALbest_s(AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end

%total 
TotLbest_s(:,1) = 0.458.*100*Ebest_scaled_s(:,1) + ALbest_s(:,1);

days=1:100:1100;
TotLbest_s_days(1:11,1)=[0:1:10]';
TotLbest_s_days(:,2)=TotLbest_s(days,:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Model - CD8 Depletion
CD8Depl_Time=[0, 2, 4, 6, 8, 10];
CD8Depl_Time2=[2, 4, 6, 8, 10];

Depl_Virus = xlsread('Myersetal_SourceData',2, 'B3:G8');
Depl_Virus_Mean=nanmean(Depl_Virus);
Depl_Virus_SD=nanstd(Depl_Virus);

Depl_CD8 = xlsread('Myersetal_SourceData',2, 'B8:G11');
Depl_CD8_Mean=nanmean(Depl_CD8);
Depl_CD8_SD=nanstd(Depl_CD8);

[H,P,CI,STATS]=ttest2(10.^Vdat(:,3),10.^Depl_Virus(:,2));
[H1,P1,CI1,STATS1]=ttest2(10.^Vdat(:,5),10.^Depl_Virus(:,3));
[H2,P2,CI2,STATS2]=ttest2(10.^Vdat(:,7),10.^Depl_Virus(:,4));
[H3,P3,CI3,STATS3]=ttest2(10.^Vdat(:,9),10.^Depl_Virus(:,5));

E_star_deplete=10.^Depl_CD8_Mean(1);
frac_decline=E_star_deplete./Estar;

% Depletion parameters
bestparams_depl=bestparams;
T0_depl=3980795;
bestparams_depl(8)=13299.21;
bestparams_depl(10)=1.01e-6;

%Run model from d0-d3
sol_deplete = dde23(@DDE_Tcell, lags, [T0_depl 75 0 0 0 0], [0 3], [], bestparams_depl); 
sol_deplete.y(5,end)=sol_deplete.y(5,end)*frac_decline;
sol_deplete.y(6,end)=sol_deplete.y(6,end)*frac_decline;
historysol=sol_deplete;

%Run model from d3-d6
sol_deplete2 = dde23(@DDE_Tcell, lags, historysol, [3 7], [], bestparams_depl); 
sol_deplete2.y(5,end)=sol_deplete2.y(5,end)*frac_decline;
sol_deplete2.y(6,end)=sol_deplete2.y(6,end)*frac_decline;

historysol2=sol_deplete2;

%Run model from d6-d10
sol_deplete3 = dde23(@DDE_Tcell, lags, historysol2, [7 20], [], bestparams_depl); 
Ys_depl = deval(sol_deplete3, tspan)';
Vbest_depl = Ys_depl(:,4);
I2best_depl = Ys_depl(:,3);
Ebest_depl = Ys_depl(:,5);
Mbest_depl = Ys_depl(:,6);
CD8best_depl = Ebest_depl+Mbest_depl+E_star_deplete;
Ebest_scaled_depl = Ebest_depl(:)/max(Ebest);

int_infcell_depl=cumtrapz(sol_deplete3.x,sol_deplete3.y(3,:));
int_infcell_depl_ = cumtrapz(tspan,Ys_depl(:,3))./1e7;

clear AL_j
ALbest_depl = 14.2.*int_infcell_depl_;
ALbest_depl(:,871:end) = 0;
AL_j = 871;
flag = 1;
while flag == 1
    ALbest_depl(AL_j+1) = ALbest_depl(AL_j) - 0.287;
    if ALbest_depl(AL_j+1) <= 0
        ALbest_depl(AL_j+1:end) = 0;
        flag = 0;
    else
        AL_j = AL_j + 1;
    end
end

%total 
TotLbest_depl(:,1) = 0.458.*100*Ebest_scaled_depl(:,1) + ALbest_depl(:,1);

days=1:100:1100;
TotLbest_depl_days(1:11,1)=[0:1:10]';
TotLbest_depl_days(:,2)=TotLbest_depl(days,:);


%%%%%%%%%%%%%%%%%%%

WtLoss=0:0.01:15;

n = 5.2437;
K = 2.5772;
alpha = 39.7019;

wl_fn=alpha*WtLoss.^n/(WtLoss.^n+K.^n);

for i=1:length(WtLoss)
    myesty(i)=alpha*(WtLoss(i).^n/(K^n + WtLoss(i).^n));
end

alpha_depl = 17.7316;

for i=1:length(WtLoss)
    myesty_depl(i)=alpha_depl*(WtLoss(i).^n/(K^n + WtLoss(i).^n));
end

% plotting params
tspan = 0:0.01:12.5;
CAUC_xlim = 1000;
new_CAUClim = CAUCmax(CAUC_xlim) + 0.02;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 4B-D

figure; 
xSize = 30; X=xSize; ySize = 7;
xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

g1 = subplot(1,3,1); 
fill([tspan fliplr(tspan)], [14.2.*ALmin 14.2.*fliplr(ALmax)], exl_col); hold on;
plot(tspan, ALbest, 'k', 'LineWidth', 2.5);  hold on; 
plot(tspan, ALbest_depl, 'm', 'LineWidth', 2.5);  hold on;
plot(tspan, ALbest_s, 'k--', 'LineWidth', 2.5);  hold on;
errorbar(H_time, 100*H_active, 100*H_activeSD,'ks','markersize',msize,'markerfacecolor','r','LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 10.8],'YLim',[-2.3 55],'YTick',0:15:100,...
    'Fontsize',fsize, 'ycolor', 'k','linewidth',2);
ylabel('% Active Lesion','Fontsize',fsize,'color','r');
xlabel('Time (d)', 'Fontsize', fsize);

g2 = subplot(1,3,2);
fill([tspan fliplr(tspan)], [0.458.*IALmin 0.458.*fliplr(IALmax)], exl_col); hold on; 
plot(tspan, 0.458.*100*Ebest_scaled, 'k', 'LineWidth', 2.5);  hold on;
plot(tspan, 0.458.*100*Ebest_scaled_depl, 'm', 'LineWidth', 2.5);  hold on;
plot(tspan, 0.458.*100*Ebest_scaled_s, 'k--', 'LineWidth', 2.5);  hold on;
errorbar(H_time, 100*H_inactive, 100*H_inactiveSD,'ks','markersize',msize,'markerfacecolor',IAcolor,'LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 10.8],'YLim',[-2.8 60],'YTick',0:20:100,...
    'Fontsize',fsize, 'ycolor', 'k','linewidth',2); hold on;
ylabel('% Inactive Lesion','Fontsize',fsize,'color',IAtextcolor); hold on;
axx = gca; xlabel('Time (d)', 'Fontsize', fsize);

g3 = subplot(1,3,3);
fill([tspan fliplr(tspan)], [TotLmin fliplr(TotLmax)], exl_col); hold on;
plot(tspan, TotLbest,'k-', 'LineWidth', 2.5);hold on;
plot(tspan, TotLbest_depl,'m-', 'LineWidth', 2.5);hold on;
plot(tspan, TotLbest_s,'k--', 'LineWidth', 2.5);hold on;

errorbar(H_time, 100.*H_total, 100.*H_totalSD, 'ks', 'LineWidth', 2, 'markersize',msize,'markerfacecolor','b');  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 10.8], 'YLim',[-3.5 77],'YTick',0:25:100,'Fontsize', fsize, 'ycolor', 'k','linewidth',2); 
ylabel('% Total Lesion','Fontsize',fsize,'color','b'); hold on;
axx = gca; xlabel('Time (d)', 'Fontsize', fsize);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 4E - Inflammation
tspan=0:0.01:12;
MyAI=[];
finalsol=dde23(@DDE_Tcell_inflam,lags,[1e7 75 0 0 0 0 0],tspan,[],[bestparams 4.2651e-07 8.6536e-08]);
 
AIsmin = importdata('AIsmin_CIpred.txt');
AIsmax = importdata('AIsmax_CIpred.txt');

sol_inflam=dde23(@DDE_Tcell_inflam,lags,[1e7 75 0 0 0 0 0],tspan,[],[bestparams 4.2651e-07 8.6536e-08]);
Ys_inflam = deval(sol_inflam, tspan)';
Inflambest = Ys_inflam(:,7);

% Inflammation - CD8 depletion
%Run model from d0-d3
sol_deplete_inflam = dde23(@DDE_Tcell_inflam, lags, [T0_depl 75 0 0 0 0 0], [0 3], [], [bestparams_depl 4.2651e-07 8.6536e-08]); 
sol_deplete_inflam.y(5,end)=sol_deplete_inflam.y(5,end)*frac_decline;
sol_deplete_inflam.y(6,end)=sol_deplete_inflam.y(6,end)*frac_decline;
historysol_inflam=sol_deplete_inflam;

%Run model from d3-d6
sol_deplete2_inflam = dde23(@DDE_Tcell_inflam, lags, historysol_inflam, [3 7], [], [bestparams_depl 4.2651e-07 8.6536e-08]); 
sol_deplete2_inflam.y(5,end)=sol_deplete2_inflam.y(5,end)*frac_decline;
sol_deplete2_inflam.y(6,end)=sol_deplete2_inflam.y(6,end)*frac_decline;

historysol2_inflam=sol_deplete2_inflam;

%Run model from d6-d10
sol_deplete3_inflam = dde23(@DDE_Tcell_inflam, lags, historysol2_inflam, [7 20], [], [bestparams_depl 4.2651e-07 8.6536e-08]); 
Ys_depl_inflam = deval(sol_deplete3_inflam, tspan)';
Inflambest_depl = Ys_depl_inflam(:,7);

Inflambest_depl_days(1:11,1)=[0:1:10]';
Inflambest_depl_days(:,2)=Inflambest_depl(days,:);

figure; 
xSize = 30; X=xSize; ySize = 7;
xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;
subplot(1,3,1);box on; hold on;
    fill([tspan fliplr(tspan)], [AIsmin fliplr(AIsmax)], exl_col); hold on;
plot(tspan,Inflambest_depl,'m', 'LineWidth', 2.5);hold on;
plot(tspan,Inflambest,'k', 'LineWidth', 2.5);hold on;
errorbar(H_time,H_alveolar,H_alveolarSD,'k^','Markersize',18','Markerfacecolor','w','Linewidth',2); hold on;
errorbar(H_time,H_interstial,H_interstialSD,'k^','Markersize',18','Markerfacecolor','y','Linewidth',2); hold on;
set(gca,'Xlim',[-0.5 10.5],'XTick',0:2:10,'Ylim',[-0.5 5.25],'YTick',0:1:5,'Fontsize', fsize, 'ycolor', 'k','linewidth',2);
xlabel('Time (d)'); ylabel('Inflammation');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 5A, C

figure;xSize = 26; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;
subplot(1,3,1);
[ax h1 h2] = plotyy(H_time, 100.*H_total, H_time, -WL_d0dX);
set(h1,'color', 'b'); hold on;
set(h2,'color', 'k'); hold on;
hold(ax(1), 'on'); box off;
h1 = errorbar(ax(1), H_time, 100.*H_total, 100.*H_totalSD, 'bs-', 'LineWidth', 2, 'markersize',msize,'markerfacecolor','b','markeredgecolor','k');  hold on;
plot(ax(1), H_time, 100.*H_total, 'b-', 'LineWidth', 2);  hold on;
plot([-1 11],[ylim1 ylim1],'k-','linewidth',2);
plot([-0.5 -0.5],[-5 ylim1],'b-','linewidth',2.2);
hold(ax(2), 'on');
h2= errorbar(ax(2), H_time, -WL_d0dX, WL_d0dX_SD, 'kd-', 'LineWidth', 2, 'markersize',msize,'markerfacecolor','k');  hold on;
errorbar(ax(2), H_time, -WL_CD8depl,WL_CD8depl_SD, 'kd-', 'LineWidth', 2, 'markersize',msize,'markerfacecolor','m');  hold on;
plot(ax(2), H_time, -WL_d0dX, 'kd-', 'LineWidth', 2, 'markersize',msize,'markerfacecolor','k');  hold on;
set(ax(1), 'XTick', 0:2:12, 'XLim', [-0.5 10.8], 'YLim',[-5 57],'YTick',0:10:50,'Fontsize', fsize, 'ycolor', 'k','linewidth',2); 
ylabel(ax(1), '% Total Lesioned Area','Fontsize',fsize,'color','b'); 
axx = ax(1);
for i = 1:6
    axx.YTickLabel{i} = ['\color{blue}' axx.YTickLabel{i}];hold on;
end
ylabel(ax(2), '% Weight Loss','Fontsize',fsize,'Rotation',270,'VerticalAlignment','middle','HorizontalAlignment','center'); 
set(ax(2), 'XTick', 0:2:12, 'XLim', [-0.5 10.8],'YLim',[-1 20.1],'YTickLabel',[0,5,10,15],'Fontsize',fsize, 'ycolor', 'k','linewidth',2); hold on;
yh = get(ax(2),'YLabel'); set(yh, 'Units', 'Normalized'); pos = get(yh, 'Position');
xlabel('Time (d)', 'Fontsize', fsize);
set(yh, 'Position',pos.*[1.12,1,1],'Rotation',270); hold off;


subplot(1,3,3);
[ax h1 h2]=plotyy(H_time,H_interstial,H_time,-WL_d0dX);
set(ax(1),'ycolor', 'k'); hold on;
set(ax(2),'ycolor', 'k'); hold on;
hold(ax(1), 'on'); box on;
h1=errorbar(ax(1),H_time,H_interstial,H_interstialSD,'k-^','Markersize',msize,'Markerfacecolor','y','Linewidth',2); hold on;
errorbar(ax(1),H_time,H_alveolar,H_alveolarSD,'k-^','Markersize',msize,'Markerfacecolor','w','Linewidth',2); hold on;
hold(ax(1), 'off');
hold(ax(2), 'on');
h2=errorbar(ax(2),H_time,-WL_d0dX,-WL_d0dX_SD,'k-d','Markersize',msize,'Markerfacecolor','k','Linewidth',2); hold on;
set(ax(1),'XTick', 0:2:12, 'XLim', [-0.5 10.5],'YLim',[-0.5 5],'YTick',0:1:5,'Fontsize',fsize,'linewidth',2); hold on;
set(ax(2),'XTick', 0:2:12, 'XLim', [-0.5 10.5],'YLim',[-1,20],'YTick',0:5:15,'Fontsize',fsize,'linewidth',2); hold on;
ylabel(ax(2), '% Weight Loss','Fontsize',fsize); 

ylabel(ax(1), 'Inflammation','Fontsize',fsize,'color','k');
yh = get(ax(2),'YLabel'); set(yh, 'Units', 'Normalized'); pos = get(yh, 'Position');
xlabel('Time (d)', 'Fontsize', fsize);
set(yh, 'Position',pos.*[1.12,1,1],'Rotation',270); hold off;


%%%%%%%%%%%%%%%%
% Figure 5B,D

WLstep=0:0.01:20;
na = 3.6527;
Ka = 1.9493;
aa = 3.6298;

aa_depl = Inflambest_depl_days(end,2);

for i=1:length(WLstep)
    Hill_alv(i)=aa*(WLstep(i).^na/(Ka^na+WLstep(i).^na));
    Hill_alv_depl(i)=aa_depl*(WLstep(i).^na/(Ka^na+WLstep(i).^na));
end

ni= 3.152;
Ki = 1.9642;
ai = 3.4006;

ai_depl = Inflambest_depl_days(end,2);

for i=1:length(WLstep)
    Hill_interst(i)=ai*(WLstep(i).^ni/(Ki^ni+WLstep(i).^ni));
    Hill_interst_depl(i)=ai_depl*(WLstep(i).^ni/(Ki^ni+WLstep(i).^ni));
end

nt = 5.2437;
Kt = 2.5772;
at = 39.7019;

for i=1:length(WLstep)
    Hill_tot(i)=at*(WLstep(i).^nt/(Kt^nt+WLstep(i).^nt));
end

figure;xSize = 26; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;
subplot(1,3,1);
plot(-WL_CD8depl,TotLbest_depl_days(:,2),'mo','Markerfacecolor','m','Markersize',18,'Linewidth',2);hold on;
plot(WtLoss,myesty_depl,'m','Linewidth',3);hold on;
plot(-WL_d0dX,100*H_total,'ko','Markersize',18','Markerfacecolor','k','Linewidth',2);hold on;
plot(WLstep,Hill_tot,'k','Linewidth',2);hold on;
set(gca,'Xlim',[-0.25 15],'XTick',0:5:15,'Ylim',[-5 55],'YTick',0:10:50,'Fontsize',28,'linewidth',2);
ylabel('% Total Lesion');xlabel('% Weight Loss');

subplot(1,3,2);
plot(-WL_CD8depl(1:end-1),Inflambest_depl_days(1:end-1,2),'mo','Markerfacecolor','w','Markersize',18,'Linewidth',2);hold on;
plot(WLstep,Hill_alv_depl,'m','Linewidth',2);hold on;
plot(-WL_d0dX,H_alveolar,'ko','Markersize',18','Markerfacecolor','w','Linewidth',2);hold on;
plot(WLstep,Hill_alv,'k','Linewidth',2);hold on;

xlabel('% Weight Loss'); ylabel('Alveolar Inflammation');
set(gca,'Xlim',[-0.5 15],'XTick',0:5:15,'Ylim',[-0.5 5],'YTick',0:1:5,'Fontsize',fsize);

subplot(1,3,3);
plot(-WL_CD8depl(1:end-1),Inflambest_depl_days(1:end-1,2),'mo','Markerfacecolor','w','Markersize',18,'Linewidth',2);hold on;
plot(WLstep,Hill_interst_depl,'m','Linewidth',2);hold on;
plot(-WL_d0dX,H_interstial,'ko','Markersize',18','Markerfacecolor','y','Linewidth',2);hold on;
plot(WLstep,Hill_interst,'k','Linewidth',2);hold on;
xlabel('% Weight Loss'); ylabel('Interstial Inflammation');
set(gca,'Xlim',[-0.5 15],'XTick',0:5:15,'Ylim',[-0.5 5],'YTick',0:1:5,'Fontsize',fsize,'linewidth',2); hold off;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure S4A-B

linup=4.5:0.1:8.5;
lindown=8.5:0.1:10.5;
lindown2=8.5:0.1:12.5;
lindown3=8.5:0.1:11.5;
lindown4=5.5:0.1:8.5;

disp(['------------']);
pfit_alesdown = polyfit([6:7],100*H_active(7:8),1);
fline_alesdown = polyval(pfit_alesdown,lindown4);
disp(['Active Lesion - down: ',num2str(pfit_alesdown(1))]);

disp(['------------']);

pfit_lesup = polyfit([5:8],100*H_inactive(6:9),1);
fline_lesup = polyval(pfit_lesup,linup);
disp(['Inactive Lesion - up: ',num2str(pfit_lesup(1))]);

pfit_lesdown = polyfit([9:10],100*H_inactive(10:11),1);
fline_lesdown = polyval(pfit_lesdown,lindown);
disp(['Inactive Lesion - down: ',num2str(pfit_lesdown(1))]);

disp(['------------']);

pfit_totup_lin = polyfit([5:8],10.^meanEdat(6:9),1);
disp(['Total CD8 - up: ',num2str(pfit_totup_lin(1))]);

pfit_totdown_lin = polyfit([9:10],10.^meanEdat(10:11),1);
disp(['Total CD8 - down: ',num2str(pfit_totdown_lin(1))]);

disp(['------------']);

pfit_lungup_lin = polyfit([5:8],10.^meanLungEdat(6:9),1);
disp(['Lung CD8 - up: ',num2str(pfit_lungup_lin(1))]);

pfit_lungdown_lin2 = polyfit([9:12],10.^meanLungEdat(10:13),1);
disp(['Lung CD8 - down: ',num2str(pfit_lungdown_lin2(1))]);

disp(['------------']);

pfit_effup_lin = polyfit([5:8],10.^Lung_EffPhen_Mean(6:9),1);
disp(['Effector CD8 - up: ',num2str(pfit_effup_lin(1))]);

pfit_effdown_lin2 = polyfit([9:11],10.^Lung_EffPhen_Mean(10:12),1);
disp(['Effector CD8 - down: ',num2str(pfit_effdown_lin2(1))]);

disp(['------------']);

pfit_totup = polyfit([5:8],meanEdat(6:9),1);
fline_totup = polyval(pfit_totup,linup);

pfit_totdown = polyfit([9:10],meanEdat(10:11),1);
fline_totdown = polyval(pfit_totdown,lindown);

pfit_lungup = polyfit([5:8],meanLungEdat(6:9),1);
fline_lungup = polyval(pfit_lungup,linup);

pfit_lungdown = polyfit([9:12],meanLungEdat(10:13),1);
fline_lungdown = polyval(pfit_lungdown,lindown2);

pfit_effup = polyfit([5:8],Lung_EffPhen_Mean(6:9),1);
fline_effup = polyval(pfit_effup,linup);

pfit_effdown = polyfit([9:11],Lung_EffPhen_Mean(10:12),1);
fline_effdown = polyval(pfit_effdown,lindown3);

%%%%%%%%%%%%%%%%
figure; 
xSize = 30; X=xSize; ySize = 7;
xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

g1 = subplot(1,3,1); 
plot(lindown4,fline_alesdown,'k-','linewidth',2); hold on;
errorbar(H_time, 100*H_active, 100*H_activeSD,'ks','markersize',msize,'markerfacecolor','r','LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 10.8],'YLim',[-3 47],'YTick',0:15:100,...
    'Fontsize',fsize, 'ycolor', 'k','linewidth',2); hold on;
axx = gca;
for i = 1:4
    axx.YTickLabel{i} = ['\color{red}' axx.YTickLabel{i}];
end
ylabel('% Active Lesion','Fontsize',fsize,'color','r');
xlabel('Time (d)', 'Fontsize', fsize);

g2 = subplot(1,3,2);
plot(linup,fline_lesup,'k-','linewidth',2); hold on;
plot(lindown,fline_lesdown,'k-','linewidth',2); hold on;
errorbar(0:1:10, 100*H_inactive, H_inactiveSD,'ks','markersize',msize,'markerfacecolor',IAcolor,'LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 10.8],'YLim',[-3 57],'YTick',0:10:100,'Fontsize',fsize, 'ycolor', 'k','linewidth',2); hold on;
axx = gca;
for i = 1:6
    axx.YTickLabel{i} = ['\color[rgb]{0,0.75,0.1}' axx.YTickLabel{i}];
end
ylabel('% Inactive Lesion','Fontsize',fsize,'color',IAtextcolor); hold on;
xlabel('Time (d)', 'Fontsize', fsize);


%%%%%%%%%%%%%%%%
% Figure S4C-E

figure; 
xSize = 30; X=xSize; ySize = 7;
xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

g1 = subplot(1,3,1); 
plot(linup,fline_totup,'k-','linewidth',2); hold on;
plot(lindown,fline_totdown,'k-','linewidth',2); hold on;
errorbar(0:1:12, meanEdat, sdEdat,'ks','markersize',msize,'markerfacecolor','k','LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 12.5],'YLim',[5.47 6.53],'YTick',5.5:0.5:6.5,'YTickLabel',{'5.5','6.0','6.5'},'Fontsize',26, 'ycolor', 'k','linewidth',2); hold on;
ylabel('Log_{10} CD8^+ T cells','Fontsize',fsize); hold on;
xlabel('Time (d)', 'Fontsize', fsize);

g2 = subplot(1,3,2);
plot(linup,fline_lungup,'k-','linewidth',2); hold on;
plot(lindown2,fline_lungdown,'k-','linewidth',2); hold on;
errorbar(0:1:12, meanLungEdat, sdLungEdat,'ks','markersize',msize,'markerfacecolor',gray,'LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 12.5],'YLim',[4.25 6.53],'YTick',4.5:0.5:6.5,'YTickLabel',{'4.5','5.0','5.5','6.0','6.5'},'Fontsize',26, 'ycolor', 'k','linewidth',2); hold on;
ylabel('Log_{10} Lung CD8^+ T cells','Fontsize',fsize); hold on;
xlabel('Time (d)', 'Fontsize', fsize);

g3 = subplot(1,3,3);
plot(linup,fline_effup,'k-','linewidth',2); hold on;
plot(lindown3,fline_effdown,'k-','linewidth',2); hold on;
errorbar(0:1:12, Lung_EffPhen_Mean, Lung_EffPhen_SD,'ks','markersize',msize,'markerfacecolor','c','LineWidth', 2);  hold on;
set(gca, 'XTick', 0:2:12, 'XLim', [-0.5 12.5],'YLim',[2.5 6.53],'YTick',3.0:1:6.0,'YTickLabel',{'3.0','4.0','5.0','6.0'},'Fontsize',26, 'ycolor', 'k','linewidth',2); hold on;
ylabel('Log_{10} CD25^-CD43^+CD8^+ T cells','Fontsize',fsize); hold on;
xlabel('Time (d)', 'Fontsize', fsize);

%%%%%%%%%%%%%%%%
% Figure S4F

LogCD8s=0:0.01:8;

%Total:
n_tot = 56.319;
K_tot = 6.1236;
alpha_tot = 52.8799;

%Lung-Specific:
n_lung = 25.1574;
K_lung = 6.0207;
alpha_lung = 64.6542;

%CD25-CD43+:
n_eff = 24.0305;
K_eff = 5.2303;
alpha_eff = 48.0441;

for i=1:length(LogCD8s)
    my_lung(i)=alpha_lung*(LogCD8s(i).^n_lung/(K_lung^n_lung+LogCD8s(i).^n_lung));
    my_tot(i)=alpha_tot*(LogCD8s(i).^n_tot/(K_tot^n_tot+LogCD8s(i).^n_tot));
    my_eff(i)=alpha_eff*(LogCD8s(i).^n_eff/(K_eff^n_eff+LogCD8s(i).^n_eff));
end

figure; 
xSize = 30; X=xSize; ySize = 7;
xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

g1 = subplot(1,3,1);
plot(meanEdat(1:end-2),100*H_inactive,'ks','Markersize',18','Markerfacecolor','k','Linewidth',2);hold on;
plot(LogCD8s(400:end),my_tot(400:end),'k','Linewidth',3);hold on;
set(gca,'Xlim',[5.5 6.5],'XTick',5.6:0.2:6.4,'XTickLabel',{'5.6','5.8','6.0','6.2','6.4'},'Ylim',[-5 60],'YTick',0:10:50,'Fontsize',26,'linewidth',2);
ylabel('% Inactive Lesion');xlabel('Log_{10} Total CD8^+ T cells');

g2 = subplot(1,3,2);
plot(meanLungEdat(1:end-2),100*H_inactive,'ks','Markersize',18','Markerfacecolor',gray,'Linewidth',2);hold on;
plot(LogCD8s,my_lung,'k','Linewidth',3);hold on;
set(gca,'Xlim',[4.4 6.6],'XTick',4.5:0.5:6.5,'XTickLabel',{'4.5','5.0','5.5','6.0','6.5'},'Ylim',[-5 60],'YTick',0:10:50,'Fontsize',26,'linewidth',2);
ylabel('% Inactive Lesion');xlabel('Log_{10} Lung CD8^+ T cells');

g3 = subplot(1,3,3);
plot(Lung_EffPhen_Mean(1:end-2),100*H_inactive,'ks','Markersize',18','Markerfacecolor','c','Linewidth',2);hold on;
plot(LogCD8s,my_eff,'k','Linewidth',3);hold on;
set(gca,'Xlim',[3 6.75],'XTick',3:1:6,'Ylim',[-5 60],'YTick',0:10:50,'Fontsize',26,'linewidth',2);
ylabel('% Inactive Lesion');xlabel('Log_{10} CD25^-CD43^+CD8^+ T cells');

