% Myers et al. "Dynamically Linking Influenza Virus Infection Kinetics, Lung Injury, Inflammation, and Disease Severity"
% Figure 1B,C,D, 3F, A1
% Author, copyright 2019: Amber M Smith, UTHSC
% amber.smith@uthsc.edu

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all;
close all;
warning off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

gray = [0.52 0.52 0.52];
IAcolor = [0 0.8 0.1];
IAtextcolor = [0 0.75 0.1];
tspan = 0:0.01:15;
tspan2 = 0:0.01:12.5;
msize = 22;
fsize = 28;
tscat = [1/6,1:12];
tscat2 = [0:12];

CD8yticks = [5.5 6.0 6.5];
CD8_time = 0:1:12;

% Virus
Vdat = xlsread('Myersetal_SourceData', 1, 'B3:N12');
meanVdat = nanmean(Vdat);
sdVdat = nanstd(Vdat);

% CD8
Edat = xlsread('Myersetal_SourceData', 1, 'B15:N24');
meanEdat = nanmean(Edat);
sdEdat = nanstd(Edat);

% Lung-specific CD8
Lung_Edat = xlsread('Myersetal_SourceData',1, 'B27:N31');
meanLungEdat=nanmean(Lung_Edat);
sdLungEdat=nanstd(Lung_Edat);

% CD25-CD43+
% Censored: 0:3
Lung_EffPhen = xlsread('Myersetal_SourceData',1, 'B34:N38');
Lung_EffPhen_Mean=nanmean(Lung_EffPhen);
Lung_EffPhen_SD=nanstd(Lung_EffPhen);

% CD43-CD69+CD62L-CD127+CD103+
% Censored: 0:5
Lung_MemPhen = xlsread('Myersetal_SourceData',1, 'B41:N45');
Lung_MemPhen_Mean=nanmean(Lung_MemPhen);
Lung_MemPhen_SD=nanstd(Lung_MemPhen);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

CIpreds = dlmread('Histol_predicted_minmax.txt');

Vsmin = CIpreds(5,:);
Vsmax = CIpreds(6,:);
Esmin = CIpreds(7,:);
Esmax = CIpreds(8,:);

CIpreds2 = dlmread('EMsmin_CIpred.txt');

EEsmin = CIpreds2(1,:);
EEsmax = CIpreds2(2,:);
EMsmin = CIpreds2(3,:);
EMsmax = CIpreds2(4,:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Model - CD8 T cell
bp0008 = importdata('params.0518.txt');
bestparams =bp0008(1,2:end-6);
for i = [1,2,3,7,8,9,10,13]
    bestparams(i) = 10.^bestparams(i);
end
lags=[bestparams(:,end-1),bestparams(:,end)];

tmax = 12.5;
tspan = 0:0.01:tmax; 
tscat = [1/6,1:tmax];
tscat2 = [0,1:tmax];
sol=dde23(@DDE_Tcell,lags,[1e7 75 0 0 0 0],[0 tmax],[],bestparams);
Estar=bestparams(end-2);
Ys = deval(sol, tspan)';
Vbest = Ys(:,4);
I2best = Ys(:,3);
Ebest = Ys(:,5);
Mbest = Ys(:,6);
CD8best = Ebest+Mbest+Estar;
Ebest_scaled = Ebest(:)/max(Ebest);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Model - Effector + Memory Phenotypes

bp_eff = importdata('params.0618.txt')';
bp_eff;

CD8_s =  bp_eff(end-1); 
CDM_s =  bp_eff(end); 
tau_Es = bp_eff(end-3);
tau_Ms = bp_eff(end-2);

lags_s=[tau_Es,tau_Ms];

sol_s=dde23(@DDE_Tcell,lags_s,[1e7 75 0 0 0 0],[0 12],[],bp_eff);

ts_s = sol_s.x';
Ys_s = deval(sol_s, 0:0.1:12)';
Vs_s = Ys_s(:,4);
Es_s = Ys_s(:,5);
Ms_s = Ys_s(:,6);
As_s = Es_s+CD8_s;
Bs_s = Ms_s+CDM_s;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 1

figure;
xSize = 20; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

subplot(1,2,1); hold on;set(gca,'Fontsize',26);box on;
plot(tspan, log10(Vbest)', 'k', 'LineWidth', 2);   hold on;
    fill([tspan fliplr(tspan)], [Vsmin fliplr(Vsmax)], [0.8  0.8  0.8]); hold on;
plot(tspan, log10(Vbest)', 'k', 'LineWidth', 2);   
errorbar(tscat,meanVdat,sdVdat,'s','Color','black','MarkerSize',15,'MarkerFaceColor','black','LineStyle','none','LineWidth',2); hold on;
ylabel('Virus (log_{10} TCID_{50})');xlabel('Time (d)');
set(gca,'XTick',[0:2:10],'XLim',[-0.25 10.5],'YLim',[-0.6 8],'Fontsize',26,'linewidth',2);
hold on;

subplot(1,2,2); hold on;set(gca,'Fontsize',26);box on;
    fill([tspan fliplr(tspan)], [Esmin fliplr(Esmax)], [0.8  0.8  0.8]); hold on;
plot(tspan, log10(CD8best)', 'k', 'LineWidth', 2);  
errorbar(tscat2,meanEdat,sdEdat,'ks','MarkerSize',15,'MarkerFaceColor','black','LineStyle','none','LineWidth',2); hold on;
xlabel('Time (d)');ylabel('Log_{10} CD8+ T cells');
set(gca,'XTick',[0:2:tmax],'XLim',[-0.2 12.5],'YLim',[5.45 6.55],'YTick',5.5:0.5:6.5,...
    'YTickLabel',sprintfc('%1.1f',CD8yticks),'Fontsize',26,'linewidth',2);
hold off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure;
xSize = 20; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;
subplot(1,2,1); hold on;set(gca,'Fontsize',26);box on;
plot(tspan, log10(Ebest+Mbest)', 'k', 'LineWidth', 2);  hold on;
errorbar(tscat2,meanLungEdat,sdLungEdat,'ko','MarkerSize',15,'MarkerFaceColor',gray,'LineStyle','none','LineWidth',2); hold on;
xlabel('Time (d)');ylabel('Log_{10} CD8+ T cells');
set(gca,'XTick',0:2:tmax,'XLim',[-0.2 12.5],'YLim',[4.15 6.55],'YTick',4.5:0.5:6.5,...
    'YTickLabel',{'4.5','5.0','5.5','6.0','6.5'},'Fontsize',26,'linewidth',2);
hold off;

subplot(1,2,2); hold on;set(gca,'Fontsize',26);box on; hold on;
    fill([tspan fliplr(tspan)], [EMsmin fliplr(EMsmax)], [0.8  0.8  0.8]); hold on;
    fill([tspan fliplr(tspan)], [EEsmin fliplr(EEsmax)], [0.8  0.8  0.8]); hold on;
errorbar(tscat2,Lung_EffPhen_Mean,Lung_EffPhen_SD,'kd','MarkerSize',16,'MarkerFaceColor','c','LineStyle','none','LineWidth',2); hold on;
errorbar(tscat2(4:end),Lung_MemPhen_Mean(4:end),Lung_MemPhen_SD(4:end),'kd','MarkerSize',16,'MarkerFaceColor','b','LineStyle','none','LineWidth',2); hold on;
plot(0:0.1:12, log10(Bs_s), 'k--', 'LineWidth', 3);  hold on;
plot(0:0.1:12, log10(As_s), 'k', 'LineWidth', 3);  hold on;
xlabel('Time (d)');ylabel('Log_{10} CD8+ T cells');
set(gca,'XTick',0:2:tmax,'XLim',[-0.2 12.5],'YLim',[2.15 6.55],'YTick',3:1:6,'Fontsize',26,'linewidth',2);
hold off;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Model - CD8 Depletion
CD8Depl_Time=[0, 2, 4, 6, 8, 10];
CD8Depl_Time2=[2, 4, 6, 8, 10];

Depl_Virus = xlsread('Myersetal_SourceData',2, 'C3:G6');
Depl_Virus_Mean=nanmean(Depl_Virus);
Depl_Virus_SD=nanstd(Depl_Virus);

Depl_CD8 = xlsread('Myersetal_SourceData',2, 'B8:G11');
Depl_CD8_Mean=nanmean(Depl_CD8);
Depl_CD8_SD=nanstd(Depl_CD8);

[H,P,CI,STATS]=ttest2(10.^Vdat(:,3),10.^Depl_Virus(:,1));
[H1,P1,CI1,STATS1]=ttest2(10.^Vdat(:,5),10.^Depl_Virus(:,2));
[H2,P2,CI2,STATS2]=ttest2(10.^Vdat(:,7),10.^Depl_Virus(:,3));
[H3,P3,CI3,STATS3]=ttest2(10.^Vdat(:,9),10.^Depl_Virus(:,4));

disp(['    ']);
disp(['P(d2)= ',num2str(P)]);
disp(['P(d4)= ',num2str(P1)]);
disp(['P(d6)= ',num2str(P2)]);
disp(['P(d8)= ',num2str(P3)]);
disp(['    ']);

E_star_deplete=10.^Depl_CD8_Mean(1);
frac_decline=E_star_deplete./Estar;

% Depletion parameters
bestparams_depl=bestparams;
T0_depl=3980795;
bestparams_depl(8)=13299.21;
bestparams_depl(10)=1.01e-6;

%Run model from d0-d3
sol_deplete = dde23(@DDE_Tcell, lags, [T0_depl 75 0 0 0 0], [0 3], [], bestparams_depl); 
sol_deplete.y(5,end)=sol_deplete.y(5,end)*frac_decline;
sol_deplete.y(6,end)=sol_deplete.y(6,end)*frac_decline;
historysol=sol_deplete;

%Run model from d3-d6
sol_deplete2 = dde23(@DDE_Tcell, lags, historysol, [3 7], [], bestparams_depl); 
sol_deplete2.y(5,end)=sol_deplete2.y(5,end)*frac_decline;
sol_deplete2.y(6,end)=sol_deplete2.y(6,end)*frac_decline;

historysol2=sol_deplete2;

%Run model from d6-d10
sol_deplete3 = dde23(@DDE_Tcell, lags, historysol2, [7 20], [], bestparams_depl); 
Ys_depl = deval(sol_deplete3, tspan)';
Vbest_depl = Ys_depl(:,4);
I2best_depl = Ys_depl(:,3);
Ebest_depl = Ys_depl(:,5);
Mbest_depl = Ys_depl(:,6);
CD8best_depl = Ebest_depl+Mbest_depl+E_star_deplete;
Ebest_scaled_depl = Ebest_depl(:)/max(Ebest);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 3F

figure;
xSize = 20; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

% Virus
subplot(1,2,1); hold on;set(gca,'Fontsize',26);box on;
errorbar(tscat,meanVdat,sdVdat,'sk','MarkerSize',15,'MarkerFaceColor','k','LineWidth',2); hold on;
errorbar(CD8Depl_Time2,Depl_Virus_Mean,Depl_Virus_SD,'dk','MarkerSize',15,'MarkerFaceColor','m','LineWidth',2); hold on;
plot(tspan, log10(Vbest)', 'k', 'LineWidth', 2);hold on;
plot(sol_deplete3.x, log10(sol_deplete3.y(4,:)),'m','Linewidth',3); hold on;
ylabel('Virus (log_{10} TCID_{50})');xlabel('Time (d)');
set(gca,'XTick',0:2:10,'XLim',[-0.25 10.5],'YLim',[-1 8],'Fontsize',26,'linewidth',2);
hold on;

% T cells
subplot(1,2,2); hold on;set(gca,'Fontsize',26);box on;
errorbar(tscat2,meanEdat,sdEdat,'s','Color','k','MarkerSize',15,'MarkerFaceColor','k','LineStyle','none','LineWidth',2); hold on;
errorbar(CD8Depl_Time, Depl_CD8_Mean, Depl_CD8_SD,'kd','MarkerSize',15,'MarkerFaceColor','m','LineStyle','none','LineWidth',2);
plot(tspan, log10(CD8best)', 'k', 'LineWidth', 2);hold on;
plot(tspan, log10(Ebest)', 'k--', 'LineWidth', 2);hold on;
plot(sol_deplete3.x, log10(sol_deplete3.y(5,:)),'m--','Linewidth',3); hold on;
plot(sol_deplete3.x, log10(sol_deplete3.y(5,:)+sol_deplete3.y(6,:)+E_star_deplete),'m','Linewidth',3); hold on;
xlabel('Time (d)');ylabel('Log_{10} CD8+ T cells');
set(gca,'XTick',0:2:13,'XLim',[-0.5 13],'YLim',[3.25 6.5],'YTick',3.5:0.5:6.5,'YTickLabel',{'3.5','4.0','4.5','5.0','5.5','6.0','6.5'},'Fontsize',26,'linewidth',2);
hold off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure A1

% Alternate Model #1
bestparams_EDD=bestparams;
bestparams_EDD(6)=4.0194e-06; %deltaE
bestparams_EDD(7)=9.5326e+05; %KD
bestparams_EDD(8)=3.1161e+05; %xi

lags_EDD = [3.6,3.5];

sol_EDD = dde23(@DDE_Tcell_EDD, lags_EDD, [1e7 75 0 0 0 0], tspan, [], bestparams_EDD); 
Ys_EDD = deval(sol_EDD, tspan)';
Vbest_EDD = Ys_EDD(:,4);
I2best_EDD = Ys_EDD(:,3);
Ebest_EDD = Ys_EDD(:,5);
Mbest_EDD = Ys_EDD(:,6);
CD8best_EDD = Ebest_EDD+Mbest_EDD+Estar;

% Alternate Model #2
sol_baral = ode15s( @DDE_Tcell_Baral, tspan, [10 10^2], [], []);
Ys_baral = deval(sol_baral, tspan)';
Is_baral = Ys_baral(:,1);
Es_baral = Ys_baral(:,2);


figure;
xSize = 20; X=xSize; ySize = 7;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;

% virus
subplot(1,2,1); hold on;set(gca,'Fontsize',26);box on;
errorbar(tscat,meanVdat,sdVdat,'s','Color','k','MarkerSize',15,'MarkerFaceColor','k','LineStyle','none','LineWidth',2); hold on;
plot(tspan, log10(Vbest)', 'k', 'LineWidth', 2);hold on;
plot(tspan, log10(Vbest_EDD)', 'b', 'LineWidth', 2);hold on;
plot(tspan, log10(Is_baral)', 'b--', 'LineWidth', 2);hold on;
ylabel('Virus (log_{10} TCID_{50})');xlabel('Time (d)');
set(gca,'XTick',0:2:10,'XLim',[-0.25 10.5],'YLim',[-1 8],'Fontsize',26);
hold on;

% T cells
subplot(1,2,2); hold on;set(gca,'Fontsize',26);box on;
errorbar(tscat2,meanEdat,sdEdat,'s','Color','k','MarkerSize',15,'MarkerFaceColor','k','LineStyle','none','LineWidth',2); hold on;
plot(tscat2(4:6),meanEdat(4:6),'sk','MarkerSize',15,'MarkerFaceColor','w','LineStyle','none','LineWidth',2); hold on;
plot(tspan, log10(CD8best_EDD)', 'b', 'LineWidth', 2);hold on;
plot(tspan, log10(Es_baral+Estar)', 'b--', 'LineWidth', 2);hold on;
plot(tspan, log10(CD8best)', 'k', 'LineWidth', 2);hold on;
xlabel('Time (d)');ylabel('Log_{10} CD8+ T cells');
set(gca,'XTick',0:2:13,'XLim',[-0.2 12.5],'YLim',[5.0 6.5],'YTick',5:0.5:6.5,'YTickLabel',{'5.0','5.5','6.0','6.5'},'Fontsize',26);
hold off;


