% Myers et al. "Dynamically Linking Influenza Virus Infection Kinetics, Lung Injury, Inflammation, and Disease Severity"
% Figure 3D-E
% Author, copyright 2019: Amber M Smith, UTHSC
% amber.smith@uthsc.edu

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close all;
clear all;
warning off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Virus
Vdat = xlsread('Myersetal_SourceData', 1, 'B3:N12');
meanVdat = nanmean(Vdat);
sdVdat = nanstd(Vdat);

% CD8
Edat = xlsread('Myersetal_SourceData', 1, 'B15:N24');
meanEdat = nanmean(Edat);
sdEdat = nanstd(Edat);

% Parameters
paramnames = {'p','c','\beta','k','\delta','\delta_E','K_{\delta_E}','\xi','K_E','\eta','d_E','\zeta','E^{*}','\tau_E','\tau_M'};
bp0008 = importdata('params.0518.txt');
bp0008 = bp0008.data;
bestcost =bp0008(3,1);
bestparams =bp0008(3,2:end-6);
params = bestparams;
for i = [1,2,3,7,8,9,10,13]
    params(i) = 10.^params(i);
end

tau_E = params(end-1);
tau_M = params(end);
Estar0 = params(end-2);
nparams = length(params);

T0 = 1e+07;
I10 = 75;
I20 = 0;
V0 = 0;
E0 = 0;
M0 = 0;
x0 = [ T0 I10 I20 V0 E0 M0];

for i=1:nparams
    lb(i) = 0.5*params(i);
    ub(i) = 1.5*params(i);
end

log_ub=log10(ub);
log_lb=log10(lb);

dim = length(lb);

lb_b=0;
ub_b= 1*params(10);
nstep_b = 60;
step(10) = (ub_b - lb_b)/nstep_b;
lb(10) = lb_b;
ub(10) = ub_b;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tmax = 60;
tscat = [1/6,1:tmax];
tspan = 0:0.05:tmax;
options = [];
tspan0 = tspan;

% Model
sol = dde23( @DDE_Tcell, [tau_E, tau_M], x0, [0 tmax], options, params );
Ys = deval(sol, tspan)';
Vbest = Ys(:,4);
Ebest = Ys(:,5);
Mbest = Ys(:,6);
CD8best = Ebest+Mbest+Estar0;
Emax = max(CD8best);
V_clear = TimeVirusCleared(tspan, Vbest, 1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


for i = 10
    ntrials_b = nstep_b+1;
    devs_b(1,:) = lb(i):step(i):ub(i);
    parval = params(i);
   
    simparams_b = zeros(ntrials_b,dim);
    for j = 1:ntrials_b
        simparams_b(j,1:dim) = params;
    end
    
    simparams_b(:,i) = devs_b(1,:);
    Es_b = zeros(ntrials_b, length(tspan));
    Ms_b = zeros(ntrials_b, length(tspan));
    CD8s_b = zeros(ntrials_b, length(tspan));

    tic;

    for itrial = 1:nstep_b%
        parx_b = simparams_b(itrial,:);
        percentval = devs_b(1,itrial)/parval;
        solx = dde23( @DDE_Tcell, [parx_b(end-1), parx_b(end)], x0, [0 tmax], options, parx_b );
        D = length(x0);
        X = zeros(length(tspan),D);
        for i = 1:D
            X(:,i) = deval(solx, tspan, i)';
        end
        X(X < 0) = 0;

        Es_b(itrial,:) = X(:,5);
        Ms_b(itrial,:) = X(:,6);
        CD8s_b(itrial,:) = Es_b(itrial,:)+Ms_b(itrial,:)+parx_b(end-2);

        Emax_x(itrial) = max(CD8s_b(itrial,:));
        percentEmax(itrial) = Emax_x(itrial)/Emax;
        V_clear_x(itrial) = TimeVirusCleared(tspan, X(:,4), 1);
        I2_clear_x(itrial) = TimeVirusCleared(tspan, X(:,3), 1);
        V_delay(itrial) = V_clear_x(itrial) - V_clear;
        I2_Kd_x(itrial) = TimeVirusCleared(tspan, X(:,3), params(1,7));
        t_Kd_Vclear_x(itrial) = V_clear_x(itrial)-I2_Kd_x(itrial);
        tt_Kd_x = ((length(tspan)-1)/tmax) * I2_Kd_x(itrial);
        E_I2Kd_x(itrial) = X(round(tt_Kd_x),5) + X(round(tt_Kd_x),6) + parx_b(end-2); 
        percentE_I2Kd_x(itrial) =  E_I2Kd_x(itrial)/Emax;
    end
    fprintf('\n')
    toc


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 3E

figure;
plot_I2_Kd = plot(I2_Kd_x,100*percentEmax, 'mo','linewidth',2,'markersize',5,'markeredgecolor',[0.6 0.6 0.6],'markerfacecolor',[0.6 0.6 0.6], 'displayname','Time at which I_2(t) = K_{\delta E}'); hold on;
plot_I2_0 = plot(V_clear_x,100*percentEmax,'kd-','linewidth',2,'markersize',7,'markerfacecolor','k','displayname','Time at which I_2(t) = 0'); hold on;
for i=1:length(percentEmax)
	plot([V_clear_x(i),I2_Kd_x(i)],[100*percentEmax(i),100*percentEmax(i)], 'color',[0.6 0.6 0.6],'linewidth',0.6); hold on;
end
set(gca,'Box','on','YLim',[35.5 48],'YTick',36:4:44,'XLim',[5 52],'XTick',10:10:50,'FontName','times','FontSize',28,'LineWidth', 1.5);
      ylabel('% Maximum CD8^+ T Cells','fontsize',28);
      xlabel('Infection Duration (d)','fontsize',28);
      hold off;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

linestyles = {'m:','m--','m-.','m-'};   

for i = 10
    parval = params(i);
    eta_evals = [38, 49.1, 49.6, 59.6];
    nstep = length(eta_evals);
    ntrials = nstep;
    clear devs;
    devs(1,:) = 0.01*eta_evals.*params(i)';
    simparams = zeros(ntrials,dim);

   for j = 1:ntrials,
       simparams(j,1:dim) = params;
   end
   simparams(:,i) = devs(1,:);

   Vs = zeros(ntrials, length(tspan));
   Es = zeros(ntrials, length(tspan));
   Ms = zeros(ntrials, length(tspan));
   CD8s = zeros(ntrials, length(tspan));

    for itrial = 1:nstep
        parx = simparams(itrial,:);
        percentval = devs(1,itrial)/parval;
        sole = dde23( @DDE_Tcell, [parx(end-1), parx(end)], x0, [0 tmax], options, parx );
        D = length(x0);
        X = zeros(length(tspan),D);
        for i = 1:D
            X(:,i) = deval(sole, tspan, i)';
        end
        X(X < 0) = 0;
        Vs(itrial,:) = X(:,4);
        I2s(itrial,:) = X(:,3);
        Es(itrial,:) = X(:,5);
        Ms(itrial,:) = X(:,6);
        CD8s(itrial,:) = Es(itrial,:)+Ms(itrial,:)+parx(end-2);

    end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 3D

figure;
xSize = 18; X=xSize; ySize = 6;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on; 
    subplot(1,2,1);
    for itrial = 1:nstep
        plot(tspan, log10(Vs(itrial,:)),linestyles{itrial}, 'LineWidth', 3); hold on; 
    end
    plot(tspan0, log10(Vbest), 'k', 'LineWidth', 3); hold on;
    xlabel('Time (d)','fontsize',40);
    ylabel('Virus (log_{10} TCID_{50})', 'fontsize',28);
    set(gca,'box','on','XTick',0:4:16,'XLim',[-0.3 18],'YLim',[-0.1 8],'Fontsize',28,'linewidth',1.5, 'fontname','times');

    subplot(1,2,2);
    hold on;
    for itrial = 1:nstep
        plot(tspan, log10(CD8s(itrial,:)),linestyles{itrial}, 'LineWidth', 2); hold on; 
    end
    plot(tspan0, log10(CD8best), 'k', 'LineWidth', 3); hold on;
    xlabel('Time (d)','fontsize',28);
    ylabel('Log_{10} CD8^+ T Cells', 'fontsize',28);
    set(gca,'Box','on',...
          'XLim',[-0.3 18],...
          'XTick',0:4:16,...
          'YLim',[5.55 6.35],...
          'YTick',5.7:0.3:6.3,'YTickLabel',{'5.7','6.0','6.3'},...
          'FontName','times',...
          'FontSize',28,...
          'LineWidth', 1.5);

end 