% Myers et al. "Dynamically Linking Influenza Virus Infection Kinetics, Lung Injury, Inflammation, and Disease Severity"
% Figure 3A,C
% Author, copyright 2019: Amber M Smith, UTHSC
% amber.smith@uthsc.edu

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all;
close all;
warning off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Parameters
bp0008 = importdata('params.0518.txt');
bestcost =bp0008(1,1);
bestparams =bp0008(1,2:end-6);
pars = bestparams;
for i = [1,2,3,7,8,9,10,13]
    pars(i) = 10.^pars(i);
end

deltaE = pars(6);
Kd = pars(7);
Estar = pars(13);

tvec = [5, 6, 7, 7.25, 7.262, 7.263, 7.2635];
tvecplot = 1:(length(tvec));

%Plot colors
mycolors = [[1 0.4 0.4]
    [1 0.8 0.2]
    [1 1 0.4]
    [0.4 1 0.4]
    [0.2 1 1]
    [0.2 0.4 1]
    [0.6 0.2 1]];

cm = colormap('hsv');
ddplot_cm = 'jet';

% Model
T0 = 1e+07;
I10 = 75;
I20 = 0;
V0 = 0;
E0 = 0;
M0 = 0;
x0 = [ T0 I10 I20 V0 E0 M0];
tmax = 16;
tspan = 0:0.0001:tmax;
options = ddeset('Maxstep',5e-3);
tscat = [1/6,1:tmax];
sol = dde23(@DDE_Tcell, [pars(end-1), pars(end)], x0, [0 tmax], options, pars );
Ys = deval(sol, tspan)';
Ys(Ys < 0) = 0;
Vbest = Ys(:,4);
Ebest = Ys(:,5);
Mbest = Ys(:,6);
I2best = Ys(:,3);
CD8best = Ebest+Mbest+Estar;

t_int = (length(tspan)-1)/tmax;
tvec_ = t_int*tvec;
for t = 1:length(tvec_)
    tt=tvec_(t);
    I2_t(t) = Ys(tt,3);
    E_t(t) = Ys(tt,5);
end

E = 5:.01:6.28;
E=10.^(E);
CD8ticks = 5:0.4:6.2;
CD8Eyticks = [5.0 5.4 5.8 6.2];
Itwo=0:0.1:8;
Itwo=10.^(Itwo);
[Ex,I2x]=meshgrid(E,Itwo);
DD = deltaE*Ex./(I2x + Kd);
DDpoints = deltaE*E_t./(I2_t + Kd);
t_int = (length(tspan)-1)/tmax;
tvec_ = t_int*tvec;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 3A-B

figure;
        yscale = -0.15;
        xscale = 0.1;
        pos = get(gcf, 'Position');
        pos(2) = 0.85*pos(2)+yscale*pos(4);
        pos(4) = 0.9*(1-yscale)*pos(4);
        set(gcf, 'Position', pos); box on;
surf(log10(Ex),log10(I2x),DD,'facealpha',0.5,'edgealpha',0.4); hold on;
colormap(ddplot_cm);hold on;
for i = tvecplot
    plot3(log10(E_t(i)),log10(I2_t(i)),DDpoints(i),'wo','markerfacecolor',mycolors(i,:),'markersize',20);hold on;
end
for i = tvecplot
    tvecstr = sprintf('%3.4g d',tvec(i));
    fprintf('rate is %4.3g, I2 = %5.5g on %4.4g d pi \n',DDpoints(i),I2_t(i), tvec(i));hold on;
end
text(4.95,0,10600,'\times10^3',...
  'VerticalAlignment', 'middle', ...
  'HorizontalAlignment', 'left',...
  'FontSize', 28,...
  'FontName','times', 'color','k');
set(gca,'Ylim',[0 8],'YTick',0:2:8,'YDir','reverse',...
    'XLim',[5 6.25],'XTick',CD8ticks,'fontsize',28,'FontName','times','XTickLabel',sprintfc('%1.1f',CD8ticks),...
    'ZLim',[0 9200],'Ztick',0:3000:9000,'Zticklabel',0:3:9,'linewidth',1);
xlabel('Log_{10} CD8_E (E)','Fontsize',26,'FontName','times');
ylabel('Log_{10} Infected Cells (I_2)','Fontsize',26,'FontName','times');
zlabel('\delta_{E}(I_{2}, E) d^{-1}','Fontsize',30,'FontName','times'); 
xh = get(gca,'XLabel');
set(xh, 'Units', 'Normalized')
pos = get(xh, 'Position');
set(xh, 'Position',pos.*[0.75,0.8,0.5],'Rotation',15)
yh = get(gca,'YLabel');
set(yh, 'Units', 'Normalized')
pos = get(yh, 'Position');
set(yh, 'Position',pos.*[0.5,-0.2,0.5],'Rotation',-28); hold off;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure 3C

figure;
xSize = 25; X=xSize; ySize = 6;xLeft = (xSize-xSize)/2; Y=ySize; yTop = (ySize-ySize)/2;
set(gcf,'PaperPosition',[xLeft yTop xSize ySize]);set(gcf,'Position',[X Y xSize*50 ySize*55]);
hold on;
subplot(1,3,1); hold on;set(gca,'Fontsize',26);box on;
plot(tspan, log10(Vbest), 'k', 'LineWidth', 2);   
for i = tvecplot
        plot(tvec(i),log10(Vbest(tvec_(i))),'ko','MarkerFaceColor',mycolors(i,:),'markersize',20);hold on;
end
ylabel('Virus (Log_{10} TCID_{50})');xlabel('Time (d)');
set(gca,'XTick',0:2:12,'XLim',[-0.25 10.8],'YLim',[-1 8],'Fontsize',28,'linewidth',1.5, 'fontname','times');hold on;

subplot(1,3,2); hold on;set(gca,'Fontsize',26);box on;
plot([-0.2 tmax+0.2],[log10(Kd) log10(Kd)],'k--','linewidth',1.3,'color',[0.5 0.5 0.5]); hold on;
plot(tspan, log10(I2best), 'k', 'LineWidth', 2);   
for i = tvecplot
        plot(tvec(i),log10(I2best(tvec_(i))),'ko','MarkerFaceColor',mycolors(i,:),'markersize',20);hold on;
end
ylabel('Log_{10} Infected Cells', 'fontsize',25);xlabel('Time (d)');
text(9.6,log10(Kd)+0.1, '$K_{\delta_E}$',...    
    'interpreter','LaTex','VerticalAlignment', 'bottom', ...
                 'HorizontalAlignment', 'center',...
                 'FontSize', 28,...
                 'FontName','times', 'color',[0.5 0.5 0.5]);
set(gca,'XTick',0:2:12,'XLim',[-0.25 10.8],'YLim',[-1 8],'Fontsize',28,'linewidth',1.5, 'fontname','times');hold on;

subplot(1,3,3); hold on;set(gca,'Fontsize',26);box on;
plot(tspan, log10(Ebest), 'k', 'LineWidth', 2);  hold on;
for i = tvecplot
        plot(tvec(i),log10(Ebest(tvec_(i))),'ko','MarkerFaceColor',mycolors(i,:),'markersize',20);
end
xlabel('Time (d)');ylabel('Log_{10} Eff. CD8^+ T cells');
set(gca,'XTick',0:2:13,'XLim',[-0.2 10.5],'YLim',[4.9 6.3],'YTick',5:0.4:6.2,... % 
    'YTickLabel',sprintfc('%1.1f',CD8Eyticks),'Fontsize',28,'linewidth',1.5, 'fontname','times');hold off;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
