% Myers et al. "Dynamically Linking Influenza Virus Infection Kinetics, Lung Injury, Inflammation, and Disease Severity"
% Figures A2B-A4
% Author, copyright 2019: Amber M Smith, UTHSC
% amber.smith@uthsc.edu

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close all;
clear all;
warning off;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Virus
Vdat = xlsread('Myersetal_SourceData', 1, 'B3:N12');
meanVdat = nanmean(Vdat);
sdVdat = nanstd(Vdat);

% CD8
Edat = xlsread('Myersetal_SourceData', 1, 'B15:N24');
meanEdat = nanmean(Edat);
sdEdat = nanstd(Edat);

% Parameters
bp0008 = importdata('params.0518.txt');
bp0008 = bp0008.data;
bestcost =bp0008(3,1);
bestparams =bp0008(3,2:end-6);

nparams=15;

T0 = 1e+07;
I10 = 75;
I20 = 0;
V0 = 0;
E0 = 0;
M0 = 0;
x0 = [ T0 I10 I20 V0 E0 M0];

tmax = 12;
tscat = [1/6,1:tmax];
tspan = 0:0.01:tmax;
options = [];

pars0=bestparams;
for i = [1,2,3,7,8,9,10,13]
    pars0(i) = 10.^pars0(i);
end
pars0;
initpars0=pars0;

minstep = 0.5;
maxstep = 1.5;
for i=1:nparams
    lb(i) = minstep*pars0(i);
    ub(i) = maxstep*pars0(i);
end
params = pars0;

paramnames = {'p', 'c', '\beta', 'k', '\delta', '\delta_E','K_\delta_E','\xi','K_E','\eta','d_E','\zeta','E^{\ast}','\tau_E','\tau_M'};

% Model
tau_E = params(end-1);
tau_M = params(end);
Estar0 = params(end-2);
sol = dde23( @DDE_Tcell, [tau_E, tau_M], x0, [0 tmax], options, params );
Ys = deval(sol, tspan)';
Vbest = Ys(:,4);
Ebest = Ys(:,5);
Mbest = Ys(:,6);
CD8best = Ebest+Mbest+Estar0;

nstep = 9;
dim = length(lb);
steps=1:(nstep+1);
CD8yticks = [5.5 5.9 6.3 6.7];
CD8yticks = [5.5 6.0 6.5];
for i=1:dim
    step(i)=(ub(i)-lb(i))/nstep;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figures A2B-A4

parsamp = 1:1:15;
for i = parsamp
    ntrials = nstep+1;
    parval = pars0(i);
    devs(1,:) = lb(i):step(i):ub(i);   
    simparams = zeros(ntrials,dim);
    for j = 1:ntrials,
        simparams(j,1:dim) = params;
    end
    
    simparams(:,i) = devs(1,:);    
    V = zeros(ntrials, dim+1);
    V(:,1:dim) = simparams(:,1:dim);
    Vs = zeros(ntrials, length(tspan));
    Es = zeros(ntrials, length(tspan));
    Ms = zeros(ntrials, length(tspan));
    CD8s = zeros(ntrials, length(tspan));
    parstr={simparams(1,i),params(i),simparams(end,i)};
    parname=paramnames{i};
    pstr=sprintf('Perturbation of %s = %g \n [%g,  %g]',parname,params(i),simparams(1,i),simparams(end,i));

    tic;
    fprintf('Evaluating for %s...\t\n',parname);
    for itrial = 1:nstep+1, 
        parx = simparams(itrial,:);
        solx = dde23( @DDE_Tcell, [parx(end-1), parx(end)], x0, [0 tmax], options, parx );
        D = length(x0);
        X = zeros(length(tspan),D);
        for m = 1:D
            X(:,m) = deval(solx, tspan, m)';
        end
        X(X < 0) = 0;
        ts=tspan;
        Ys=X;
        Vs(itrial,:) = Ys(:,4);
        I2s(itrial,:) = Ys(:,3);
        Es(itrial,:) = Ys(:,5);
        Ms(itrial,:) = Ys(:,6);
        CD8s(itrial,:) = Es(itrial,:)+Ms(itrial,:)+parx(end-2);
        fprintf('%g  ',itrial);
    end
    fprintf('\n')
    toc

    figure(i); hold on;
    xSize = 24; X=xSize; ySize = 10;xLeft = (20-xSize)/2; Y=ySize; yTop = (30-ySize)/2;
    set(gcf,'PaperPosition',[xLeft yTop xSize ySize],'Position',[X Y xSize*50 ySize*50]); 
    subplot; hold on;
    axes('Box','on',...
              'XLim',[-0.24 tmax],...
              'XTick',0:2:tmax,...
              'YLim',[-1-0.24 8+0.24],...
              'YTick',0:2:8,...
              'FontSize',36);
    hold on;
    subplot(1,2,1);
    for itrial = 1:nstep+1
        plot(tspan, log10(Vs(itrial,:)), 'LineWidth', 2, 'Color', v2rgb( itrial, steps)); hold on;
    end
    plot(tspan, log10(Vbest), 'k', 'LineWidth', 2); hold on;
    errorbar(tscat,meanVdat,sdVdat,'s','Color','black','MarkerSize',24,'MarkerFaceColor','black','LineStyle','none','LineWidth',2);
    xlabel('Time (d)','fontsize',36);
    ylabel('Virus (log_{10} TCID_{50})', 'fontsize',36);
    set(gca,'Box','on',...
          'XLim',[-0.24 tmax],...
          'XTick',0:2:tmax,...
          'YLim',[-0.5 8],...
          'YTick',0:2:8,...
          'FontName','times',...
          'FontSize',36,...
          'LineWidth', 2.5);
    subplot(1,2,2); hold on;
    for itrial = 1:nstep+1
        plot(tspan, log10(CD8s(itrial,:)), 'LineWidth', 2, 'Color', v2rgb(itrial, steps));hold on;
    end
    plot(tspan, log10(CD8best), 'k', 'LineWidth', 2);hold on;
    errorbar(tscat,meanEdat,sdEdat,'s','Color','black','MarkerSize',24,'MarkerFaceColor','black','LineStyle','none','LineWidth',2);
    xlabel('Time (d)','fontsize',36);
    ylabel('Log_{10} CD8^+ T cells', 'fontsize',36,'FontName','times'); 
    set(gca,'Box','on',...
          'XLim',[-0.24 tmax],...
          'XTick',0:2:tmax,...
          'YLim',[5.35 6.7],...
          'YTick',5.5:0.5:6.5,...
          'YTickLabel',sprintfc('%1.1f',CD8yticks),...
          'FontName','times',...
          'FontSize',36,...
          'LineWidth', 2); hold off;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figures A2B-A4

parsamp = 1:1:15;
for i = parsamp
    ntrials = nstep+1;
    parval = pars0(i);
    devs(1,:) = lb(i):step(i):ub(i);   
    simparams = zeros(ntrials,dim);
    for j = 1:ntrials,
        simparams(j,1:dim) = params;
    end
    
    simparams(:,i) = devs(1,:);    
    V = zeros(ntrials, dim+1);
    V(:,1:dim) = simparams(:,1:dim);
    Vs = zeros(ntrials, length(tspan));
    Es = zeros(ntrials, length(tspan));
    Ms = zeros(ntrials, length(tspan));
    CD8s = zeros(ntrials, length(tspan));
    parstr={simparams(1,i),params(i),simparams(end,i)};
    parname=paramnames{i};
    pstr=sprintf('Perturbation of %s = %g \n [%g,  %g]',parname,params(i),simparams(1,i),simparams(end,i));

    tic;
    fprintf('Evaluating for %s...\t\n',parname);
    for itrial = 1:nstep+1, 
        parx = simparams(itrial,:);
        solx = dde23( @DDE_Tcell, [parx(end-1), parx(end)], x0, [0 tmax], options, parx );
        D = length(x0);
        X = zeros(length(tspan),D);
        for m = 1:D
            X(:,m) = deval(solx, tspan, m)';
        end
        X(X < 0) = 0;
        ts=tspan;
        Ys=X;
        Vs(itrial,:) = Ys(:,4);
        I2s(itrial,:) = Ys(:,3);
        Es(itrial,:) = Ys(:,5);
        Ms(itrial,:) = Ys(:,6);
        CD8s(itrial,:) = Es(itrial,:)+Ms(itrial,:)+parx(end-2);
        fprintf('%g  ',itrial);
    end
    fprintf('\n')
    toc

    figure(i); hold on;
    xSize = 24; X=xSize; ySize = 10;xLeft = (20-xSize)/2; Y=ySize; yTop = (30-ySize)/2;
    set(gcf,'PaperPosition',[xLeft yTop xSize ySize],'Position',[X Y xSize*50 ySize*50]); 
    subplot; hold on;
    axes('Box','on',...
              'XLim',[-0.24 tmax],...
              'XTick',0:2:tmax,...
              'YLim',[-1-0.24 8+0.24],...
              'YTick',0:2:8,...
              'FontSize',36);
    hold on;
    subplot(1,2,1);
    for itrial = 1:nstep+1
        plot(tspan, log10(Vs(itrial,:)), 'LineWidth', 2, 'Color', v2rgb( itrial, steps)); hold on;
    end
    plot(tspan, log10(Vbest), 'k', 'LineWidth', 2); hold on;
    errorbar(tscat,meanVdat,sdVdat,'s','Color','black','MarkerSize',24,'MarkerFaceColor','black','LineStyle','none','LineWidth',2);
    xlabel('Time (d)','fontsize',36);
    ylabel('Virus (log_{10} TCID_{50})', 'fontsize',36);
    set(gca,'Box','on',...
          'XLim',[-0.24 tmax],...
          'XTick',0:2:tmax,...
          'YLim',[-0.5 8],...
          'YTick',0:2:8,...
          'FontName','times',...
          'FontSize',36,...
          'LineWidth', 2.5);
    subplot(1,2,2); hold on;
    for itrial = 1:nstep+1
        plot(tspan, log10(CD8s(itrial,:)), 'LineWidth', 2, 'Color', v2rgb(itrial, steps));hold on;
    end
    plot(tspan, log10(CD8best), 'k', 'LineWidth', 2);hold on;
    errorbar(tscat,meanEdat,sdEdat,'s','Color','black','MarkerSize',24,'MarkerFaceColor','black','LineStyle','none','LineWidth',2);
    xlabel('Time (d)','fontsize',36);
    ylabel('Log_{10} CD8^+ T cells', 'fontsize',36,'FontName','times'); 
    set(gca,'Box','on',...
          'XLim',[-0.24 tmax],...
          'XTick',0:2:tmax,...
          'YLim',[5.35 6.7],...
          'YTick',5.5:0.5:6.5,...
          'YTickLabel',sprintfc('%1.1f',CD8yticks),...
          'FontName','times',...
          'FontSize',36,...
          'LineWidth', 2); hold off;
end
    