
library(dplyr)
library(tidyr)
library(ggplot2)
library(maps)
library(stringr)
library(scico)
library(cowplot)
library(ggpubr)
library(ggtext)
library(UpSetR)

amphiData <- read.csv(file = "./Data/Amphibians_in_trade.csv",
                      stringsAsFactors = FALSE)

amphiData %>% 
  group_by(anyTrade_extra) %>% 
  count()

amphiData$isocc

worldData <- map_data("world")
worldData$region
isoData <- iso3166
names(isoData)

sort(unique(worldData$region))
isoData$mapname

isoData <- isoData %>% 
  mutate(mapname = 
           case_when(str_detect(mapname, "Norway") ~ "Norway",
                     str_detect(mapname, "China") ~ "China",
                     str_detect(mapname, "Finland") ~ "Finland",
                     # mapname == "Finland(?!:Aland)" ~ "Finland",
                     # mapname == "Finland:Aland Islands" ~ ""
                     str_detect(mapname, "Cyprus") ~ "Cyprus",
                     mapname == "UK(?!r)" ~ "UK",
                     TRUE ~ mapname
           )
  )

# "China:Hong Kong"
# "Norway(?!:Bouvet|:Svalbard|:Jan Mayen)"
# "Cyprus:Akrotiri"                       
# "Cyprus:Dhekeli"
# "Norway:Svalbard"
# "Norway:Jan Mayen"

worldData <- worldData %>% 
  left_join(isoData %>% 
              rename("region" = mapname))

countrySpecies <- as.data.frame(table(unlist(str_split(amphiData$isocc, ","))))
names(countrySpecies) <- c("country", "nSpecies")
countryTradedSpecies <- as.data.frame(table(unlist(str_split(amphiData$isocc[amphiData$anyTrade],
                                                             ","))))
names(countryTradedSpecies) <- c("country", "nTradedSpecies")

countryTradeData <- countrySpecies %>% 
  full_join(countryTradedSpecies) %>% 
  mutate(percentTrade = nTradedSpecies / nSpecies *100)

worldDataAll <- worldData %>% 
  rename("country" = a2) %>% 
  left_join(countryTradeData) %>% 
  group_by(region) %>% 
  mutate(
    nSpecies = max(nSpecies, na.rm = TRUE),
    nTradedSpecies = max(nTradedSpecies, na.rm = TRUE),
    percentTrade = max(percentTrade, na.rm = TRUE))

nSpeciesLabelData <- worldDataAll %>% 
  group_by(region) %>%
  ungroup() %>%
  slice_max(nSpecies, n = 1) %>% 
  mutate(arrowEndx = mean(long),
         arrowEndy = mean(lat)) %>% 
  slice(n = 1) %>% 
  mutate(labelx = -42,
         labely = 18)

nTradedSpeciesLabelData <- worldDataAll %>% 
  group_by(region) %>%
  ungroup() %>%
  slice_max(nTradedSpecies, n = 1) %>% 
  mutate(arrowEndx = mean(long),
         arrowEndy = mean(lat)) %>% 
  slice(n = 1) %>% 
  mutate(labelx = -65,
         labely = 30)

percentTradeLabelData <- worldDataAll %>% 
  group_by(region) %>%
  ungroup() %>%
  filter(!percentTrade == -Inf) %>% 
  slice_min(percentTrade, n = 1) %>%
  mutate(arrowEndx = mean(long),
         arrowEndy = mean(lat)) %>% 
  slice(n = 1) %>%
  mutate(labelx = 85,
         labely = -15)


nSpeciesPlot <- worldDataAll %>% 
  ggplot(aes(x = long, y = lat, group = group)) +
  geom_polygon(aes(fill = nSpecies)) +
  geom_text(data = nSpeciesLabelData,
            aes(x = labelx, y = labely+2,
                label = paste0("Max spp.\n", nSpecies)),
            fontface = 2, lineheight = 0.875,
            vjust = 0) +
  geom_segment(data = nSpeciesLabelData,
               aes(x = labelx, y = labely,
                   xend = arrowEndx, yend = arrowEndy),
               arrow = arrow(angle = 30,
                             length = unit(2, "mm"),
                             type = "closed")) +
  coord_map("mollweide", xlim = c(-180, 180)) +
  scale_fill_scico(palette = "roma",
                   breaks = c(1,
                              250, 500, 750,
                              max(worldDataAll$nSpecies, rm.na = TRUE))) +
  labs(fill = "# of amphibian\nspecies") +
  theme_void() +
  theme(legend.position = c(0.15,0.2),
        legend.title = element_text(face = 2)) +
  guides(fill = guide_colourbar(title.position = "bottom",
                                direction = "horizontal",
                                barwidth = unit(40, "mm"),
                                barheight = unit(5, "mm")))

nSpeciesPlot <- ggdraw(nSpeciesPlot) +
  draw_grob(ggplotGrob(ggplot(worldDataAll) +
                         geom_density(aes(nSpecies), fill = "black") +
                         theme_void()),
            x = 0.015, y = 0.348, width = 0.27, height = 0.15)

nTradedSpeciesPlot <- worldDataAll %>% 
  ggplot(aes(x = long, y = lat, group = group)) +
  geom_polygon(aes(fill = nTradedSpecies)) +
  geom_text(data = nTradedSpeciesLabelData,
            aes(x = labelx+23, y = labely-16,
                label = paste0("Max spp.\n", nTradedSpecies)),
            fontface = 2, lineheight = 0.875,
            vjust = 0, hjust = 0.5) +
  geom_segment(data = nTradedSpeciesLabelData,
               aes(x = labelx, y = labely,
                   xend = arrowEndx+20, yend = arrowEndy-10),
               arrow = arrow(angle = 30,
                             length = unit(2, "mm"),
                             type = "closed")) +
  coord_map("mollweide", xlim = c(-180, 180)) +
  scale_fill_scico(palette = "roma",
                   breaks = c(1,
                              25, 50, 75,
                              max(worldDataAll$nTradedSpecies, rm.na = TRUE))) +
  labs(fill = "# of species\ntraded") +
  theme_void() +
  theme(legend.position = c(0.15,0.2),
        legend.title = element_text(face = 2)) +
  guides(fill = guide_colourbar(title.position = "bottom",
                                direction = "horizontal",
                                barwidth = unit(40, "mm"),
                                barheight = unit(5, "mm")))

nTradedSpeciesPlot <- ggdraw(nTradedSpeciesPlot) +
  draw_grob(ggplotGrob(ggplot(worldDataAll) +
                         geom_density(aes(nTradedSpecies), fill = "black") +
                         theme_void()),
            x = 0.015, y = 0.348, width = 0.27, height = 0.15)

percentTradePlot <- worldDataAll %>% 
  ggplot(aes(x = long, y = lat, group = group)) +
  geom_polygon(aes(fill = percentTrade)) +
  geom_text(data = percentTradeLabelData,
            aes(x = labelx-10, y = labely,
                label = paste0("Min %\n", round(percentTrade, digits = 1))),
            fontface = 2, lineheight = 0.875,
            vjust = 0) +
  geom_segment(data = percentTradeLabelData,
               aes(x = labelx, y = labely,
                   xend = arrowEndx, yend = arrowEndy),
               arrow = arrow(angle = 30,
                             length = unit(2, "mm"),
                             type = "closed")) +
  coord_map("mollweide", xlim = c(-180, 180)) +
  scale_fill_scico(palette = "roma",
                   breaks = c(0,
                              seq(20,80,20),
                              max(worldDataAll$percentTrade, rm.na = TRUE))) +
  labs(fill = "% of species\ntraded") +
  theme_void() +
  theme(legend.position = c(0.15, 0.2),
        legend.title = element_text(face = 2)) +
  guides(fill = guide_colourbar(title.position = "bottom",
                                direction = "horizontal",
                                barwidth = unit(40, "mm"),
                                barheight = unit(5, "mm")))

percentTradePlot <- ggdraw(percentTradePlot) +
  draw_grob(ggplotGrob(ggplot(worldDataAll) +
                         geom_density(aes(percentTrade), fill = "black") +
                         theme_void()),
            x = 0.015, y = 0.348, width = 0.27, height = 0.15)

ggarrange(nSpeciesPlot,
          nTradedSpeciesPlot,
          percentTradePlot,
          nrow = 3, align = "h",
          labels = c("A", "B", "C"))

ggsave("./Figures/Trade by country.png", units = "mm",
       width = 160, height = 220, dpi = 300)
ggsave("./Figures/Trade by country.pdf", units = "mm",
       width = 160, height = 220)

# Endangerment status -----------------------------------------------------

iucnData <- amphiData %>% 
  mutate(
    iucnSimp = sub("\\).*", "", sub(".*\\(", "", iucn)), 
    iucn = sub(".{5}$", "", iucn),
    colour = case_when(
      iucn == "Least Concern" ~ "#309706",
      iucn == "Near Threatened" ~ "#A8DB06", 
      iucn == "Vulnerable" ~ "#F5D800",
      iucn == "Endangered" ~ "#DC7000", 
      iucn == "Critically Endangered" ~ "#CB1500", 
      iucn == "Extinct in the Wild" ~ "#701E08", 
      iucn == "Extinct" ~ "#000000",
      iucn == "Data Deficient" ~ "#717171",
      is.na(iucn) ~ "#BFBFBF"),
    iucnCol = glue::glue("{iucn} <b style='color:{colour}'>({iucnSimp})</b>"),
    iucnCol = factor(iucnCol, levels = c(
      "NA <b style='color:#BFBFBF'>(NA)</b>",
      "Data Deficient <b style='color:#717171'>(DD)</b>",
      "Extinct <b style='color:#000000'>(EX)</b>",
      "Extinct in the Wild <b style='color:#701E08'>(EW)</b>",
      "Critically Endangered <b style='color:#CB1500'>(CR)</b>",
      "Endangered <b style='color:#DC7000'>(EN)</b>",
      "Vulnerable <b style='color:#F5D800'>(VU)</b>",
      "Near Threatened <b style='color:#A8DB06'>(NT)</b>",
      "Least Concern <b style='color:#309706'>(LC)</b>"
    )),
    anyTrade = ifelse(anyTrade, "Traded species", "Not traded species")) %>% 
  group_by(anyTrade, iucnCol) %>% 
  summarise(nSpecies = length(amphiName)) %>%
  mutate(nSpecies = ifelse(anyTrade == "Not traded species", -nSpecies, nSpecies)) %>% 
  ungroup() %>%
  tidyr::complete(anyTrade, iucnCol,
           fill = list(nSpecies = 0))

cols <- scico(2, palette = "roma")

tots <- iucnData %>% 
  group_by(anyTrade) %>% 
  summarise(Tot = abs(sum(nSpecies))) %>% 
  pull(Tot)

iucnData %>% 
  ggplot() +
  geom_vline(xintercept = 0, linetype = 2, alpha = 0.5) +
  geom_col(aes(x = nSpecies, y = iucnCol,
                   colour = iucnCol, fill = iucnCol), width = 0.01) +
  geom_point(aes(x = nSpecies, y = iucnCol, colour = iucnCol)) +
  annotate("segment", x = 750, xend = 750, y = 4, yend = 7) +
  annotate("segment", x = 200, xend = 750, y = 7, yend = 7) +
  annotate("segment", x = 100, xend = 750, y = 4, yend = 4) +
  geom_richtext(aes(x = nSpecies,
                y = iucnCol, label = abs(nSpecies),
                hjust = c(rep(1.25, 9), rep(-0.5, 9)),
                fill = iucnCol),
                # fill = NA,
                colour = "white",
                label.color = NA,
                lineheight = 1.6,
                fontface = 2,
                vjust = 0.5,
                label.padding = grid::unit(rep(2, 4), "pt")) +
  annotate("segment", x = -50, xend = -500, y = 10, yend = 10,
           size = 2, arrow = arrow(angle = 30, type = "closed",
                                   length = unit(3, "mm")),
           colour = cols[2]) +
  annotate("text", x = -550, y = 10, label = paste(tots[1], ":Not traded"),
           vjust = 0.5, hjust = 1, fontface = 4) +
  annotate("segment", x = 50, xend = 500, y = 10, yend = 10,
           size = 2, arrow = arrow(angle = 30, type = "closed",
                                   length = unit(3, "mm")),
           colour = cols[1]) +
  annotate("text", x = 550, y = 10, label = paste("Traded:", tots[2]),
           vjust = 0.5, hjust = 0, fontface = 4) +
  annotate("label", x = 750, y = 5.5, label = paste0(sum(iucnData[iucnData$iucnCol %in% c(
    "Extinct in the Wild <b style='color:#701E08'>(EW)</b>",
    "Critically Endangered <b style='color:#CB1500'>(CR)</b>",
    "Endangered <b style='color:#DC7000'>(EN)</b>",
    "Vulnerable <b style='color:#F5D800'>(VU)</b>") &
      iucnData$anyTrade == "Traded species",]$nSpecies), "\nthreatened\nspecies"),
    hjust = 0.5, fontface = 2, lineheight = 0.875, label.size = 0) +
  labs(fill = "IUCN RedList\nStatus",
       alpha = "Traded",
       x = "Number of species",
       y = "") +
  scale_y_discrete(expand = expansion(c(0,0), c(1,1.5))) +
  scale_x_continuous(expand = expansion(c(0,0), c(500,500)), position = "top",
                     sec.axis = dup_axis(),
                     breaks = seq(-3000, 1500, 500),
                     labels = c(seq(3000, 0, -500), seq(500, 1500, 500))) +
  scale_colour_manual(values = str_extract(unique(iucnData$iucnCol), "\\#.{6}"))+
  scale_fill_manual(values = str_extract(unique(iucnData$iucnCol), "\\#.{6}"))+
  theme_bw() +
  theme(axis.title = element_text(face = 2, hjust = 0),
        axis.title.y = element_text(hjust = 0, vjust = 1, angle = 0),
        axis.title.x = element_text(margin = margin(5,0,5,0)),
        axis.title.x.top = element_text(margin = margin(5,0,5,0)),
        legend.title = element_text(face = 2),
        legend.position = "none",
        axis.text.y = element_markdown(),
        axis.ticks.y = element_blank(),
        strip.background = element_blank(),
        strip.text = element_text(face = 4, hjust = 0),
        panel.background = element_blank(),
        panel.border = element_blank(),
        axis.line.x = element_line(),
        panel.grid = element_blank())

ggsave("./Figures/IUCN and traded species.png", width = 180, height = 100,
       dpi = 300, units = "mm")
ggsave("./Figures/IUCN and traded species.pdf", width = 180, height = 100,
       units = "mm")

# Upset plot --------------------------------------------------------------

upsetData <- amphiData %>%
  select(amphiName, onlineTr_2, lemisTrade,
         citesTrade, Mohantytrade, IUCNTrade) %>% 
  mutate(across(.cols = c("onlineTr_2", "lemisTrade",
                   "citesTrade", "Mohantytrade", "IUCNTrade"),
                .fns = as.numeric))

pdf(file = "./Figures/Base upset plot.pdf",
    width = 8, height = 4)
upset(upsetData)
dev.off()

amphiData %>% 
  select(amphiName, onlineTr_2, lemisTrade,
         citesTrade, Mohantytrade, IUCNTrade, citesApp) %>% 
  mutate(across(.cols = c("onlineTr_2", "lemisTrade",
                   "citesTrade", "Mohantytrade", "IUCNTrade"),
                .fns = as.numeric)) %>% 
  reshape2::melt() %>%
  mutate(source = factor(variable,
                         levels = c("lemisTrade", "onlineTr_2",
                                    "IUCNTrade", "Mohantytrade",
                                    "citesTrade"))) %>% 
  filter(value == 1) %>% 
  group_by(source, citesApp) %>% 
  count() %>% 
  ggplot() +
  geom_col(aes(x = source, y = n, fill = citesApp)) +
  coord_flip() +
  scale_y_reverse() +
  scale_fill_grey(start = 0.6, end = 0.1) +
  theme_bw() 

ggsave("./Figures/Figure 1 - extra plot.pdf")

amphiData %>% 
  filter(anyTraded) %>% 
  group_by(order) %>% 
  count()
