library(tidyverse)
library(cowplot)

# Script to produce figure3.
# Source data for the panels  are in ./data/

#### Panel A: Spaghetti Plot ####
d.long <- read_tsv("data/panelA_spaghetti_data.tsv")

p.spaghetti <- d.long %>%
    filter(frag_len>=80) %>%
    filter(frag_len<=400) %>%
    ggplot(aes(x=frag_len,
               y=dens,
               col=class,
               group=sample)) +
    geom_line(lwd = 0.2, alpha=0.2) +
    geom_vline(aes(xintercept=167), lwd=0.5, lty=3) +
    scale_x_continuous(name="Fragment length (bp)") +
    scale_y_continuous(name="Relative frequency") +
    theme_bw() +
    theme(legend.position=c(0.8, 0.6)) + 
    guides(colour = guide_legend(override.aes = list(alpha=1, lwd=1))) +
    theme(legend.background = element_rect(size=0.2, linetype="solid", color="black")) +
    theme(legend.title=element_blank())
#ggsave("delfi_spaghetti.pdf", width = 6, height = 6)


#### Panel B: NMF 2 Sigs samplewise ####
d.long <- read_tsv('data/panelB_sample_wise_signatures_data.tsv')

p.two_sigs_sample <- d.long %>%
  filter(frag_len>=80) %>%
  filter(frag_len<=400) %>%
  ggplot(aes(x=frag_len,
             y=dens,
             col=Signature,
             group=Signature)) +
  geom_line() +
  geom_vline(aes(xintercept=167), lwd=0.5, lty=3) +
  scale_x_continuous(name="Fragment length (bp)") +
  scale_y_continuous(name="Relative frequency") +
  scale_color_brewer(name="Sample-wise NMF", palette="Set2") +
  theme_bw() +
  theme(legend.position=c(0.7, 0.6))  +
  theme(legend.background = element_rect(size=0.2, linetype="solid", color="black")) +
  theme(legend.title=element_text(size = 9, face="bold"))


#### Panel C: NMF 2 AUC samplewise ####

d <- read_tsv("data/panelC_AUC_data.tsv")

p.auc_sample <- d %>%
  ggplot(aes(x=n_components, y=auc, group=n_components)) + 
  geom_boxplot() +
  scale_y_continuous("AUC") + 
  scale_x_continuous("Number of NMF components") +
  theme_bw()

#### Panel D: NMF 2 Sigs binwise ####

d.long <- read_tsv('data/panelD_bin_wise_signatures_data.tsv')

p.two_sigs_bin <- d.long %>%
  filter(frag_len>=80) %>%
  filter(frag_len<=400) %>%
  ggplot(aes(x=frag_len,
             y=dens,
             col=Signature,
             group=Signature)) +
  geom_line() +
  geom_vline(aes(xintercept=167), lwd=0.5, lty=3) +
  scale_x_continuous(name="Fragment length (bp)") +
  scale_y_continuous(name="Relative frequency") +
  scale_color_brewer(name="Bin-wise NMF (controls)", palette="Set2") +
  theme_bw() +
  theme(legend.position=c(0.67, 0.85))  +
  theme(legend.background = element_rect(size=0.2, linetype="solid", color="black")) +
  theme(legend.title=element_text(size = 9, face="bold"))


#### Panel E: ATACseq correlation ####

d <- read_tsv("data/panelE_correlation_data.tsv")

labels <- data.frame(
  r = c(signif(cor(d[["Signature#1 weight"]], d$ATACseq, method = "spearman"),2), signif(cor(d[["DELFI ratio"]], d$ATACseq, method = "spearman"),2)),
  name = c("Signature#1 weight", "DELFI ratio"),
  y = c(0.85, 0.27)
)

p.ATACseq_3 <- d %>% 
  pivot_longer(cols = c('DELFI ratio', 'Signature#1 weight')) %>%
  ggplot(aes(x=ATACseq, y=value)) + 
  geom_point(alpha=0.1)+
  geom_density_2d(contour_var = "ndensity")+
  facet_grid(name~., scale= "free_y") + 
  theme_bw() +
  geom_text(data = labels, aes(label = paste("rho","==", r), y= y), x = 0.1, hjust = 0, parse =T) +
  scale_y_continuous("") +
  scale_color_brewer(palette="Set2")
#p.ATACseq_3

#### Panel F: AUC binwise ####

d <- read_tsv('data/panelF_binwise_AUC.tsv')

p.auc_bins <- d %>%
  ggplot(aes(x=binsize,y=value, ymin=value-se, ymax=value+se, color=input)) + 
  geom_point() + geom_line() +
  geom_errorbar(width=0) + 
  scale_y_continuous("AUC") + 
  scale_x_continuous("Bin Size in kb") +
  scale_color_discrete("SVM input") + 
  theme_bw() + 
  theme(legend.position=c(0.7, 0.8)) +
  theme(legend.background = element_rect(size=0.2, linetype="solid", color="black")) +
  theme(legend.title=element_text(size = 9, face="bold"))

#p.auc_bins


#### Joint Figure ####

p.fig3 <- plot_grid(p.spaghetti, p.two_sigs_sample, p.auc_sample, p.two_sigs_bin, p.ATACseq_3, p.auc_bins, labels = "auto", nrow =2)
ggsave("figure3.pdf", width=12, height=7)
ggsave("figure3.png", p.fig3, width=12, height=7)

