clear all
close all
stylefile
%% PAIRWISE CASE, STATIONARY TIME SERIES
% random number seed and parameters
rng(1);
n_steps = 1000;
n_perm = 10^4-1;
burn = 0;
r = 1;
mu = 0;
sig = 1;
x0 = 0;
% simulate two random walks
x = ar_gauss(n_steps+burn, r, mu, sig, x0);
x = x(burn+1:end);
y = ar_gauss(n_steps+burn, r, mu, sig, x0);
y = y(burn+1:end);
z = ar_gauss(n_steps+burn, r, mu, sig, x0);
z = z(burn+1:end);
w = ar_gauss(n_steps+burn, r, mu, sig, x0);
w = w(burn+1:end);
v = ar_gauss(n_steps+burn, r, mu, sig, x0);
v = v(burn+1:end);
pval = permtest(y,w,n_perm);
rho = pcorr(y,w);

% plot the result
fig1 = figure(1);
plot(x); hold on; plot(y); plot(z); plot(w); plot(v)
xticks([1,500,1000])
yticks([-50, -25, 0, 25, 50, 75, 100])
%ylim([-25, 100])
xlabel('time step')
ylabel('change in biomass')
legend('x','y', 'z', 'w', 'v')
txt = sprintf('\\rho = %.2f, p = %.2e', rho, pval);
text(300,10,txt)
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig1,'raw/rw.svg')
close

%%
fig1 = figure(1);
scatter(z, y, 10, 'k', 'filled')
xlabel('z')
ylabel('y')
yticks([-20, 20, 60, 100])
pval = permtest(z,y,n_perm)
rho = pcorr(z,y)
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 250 250]);
set(findall(gcf,'-property','FontSize'),'FontSize',14)
saveas(fig1,'raw/rw_scatter.svg')
close

%% PLOT SOME TRAJECTORIES
fig1 = figure(1);
scatter(x,y,20,'k','filled')
xlabel('change x in biomass')
ylabel('change y in biomass')
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig1,'raw/scatter1.svg')

fig1 = figure(1);
scatter(x,z,20,'r','filled')
xlabel('change x in biomass')
ylabel('change z in biomass')
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig1,'raw/scatter2.svg')

fig1 = figure(1);
scatter(y,z,20,'b','filled')
xlabel('change y in biomass')
ylabel('change z in biomass')
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig1,'raw/scatter3.svg')
close all
%% 1000 RANDOM WALKS
rng(1);

% perform 1000 random walks and record results
n_trials = 1000;
RHO = zeros(1, n_trials);
PVAL = zeros(1, n_trials);
X = zeros(n_steps, n_trials);
Y = zeros(n_steps, n_trials);
for trial = 1 : n_trials
    x = ar_gauss(n_steps+burn, r, mu, sig, x0);
    x = x(burn+1:end);
    y = ar_gauss(n_steps+burn, r, mu, sig, x0);
    y = y(burn+1:end);
    X(:,trial) = x';
    Y(:,trial) = y';
    RHO(trial) = pcorr(x,y);
    PVAL(trial) = permtest(x,y,n_perm);
end
%% save X and Y data
csvwrite('raw/rw_xdata.csv', X);
csvwrite('raw/rw_ydata.csv', Y);

%% save rho and pval
csvwrite('raw/rw_data.csv', [RHO;PVAL]')

%% make pie chart
fig = figure;
frac_sig = sum(PVAL <= 0.05)/ n_trials;
piedata = [frac_sig, 1-frac_sig];
pie(piedata)
saveas(fig,'raw/rw_piechart.svg')

%% make histogram of significant rho values
fig = figure;
histogram(abs(RHO(PVAL<=0.05)), 'Normalization', 'probability')
xlabel('|\rho|')
ylabel('frequency observed')
title({'magnitudes of','significant correlations'})
xlim([0, 1])
xticks([0 : 0.25 : 1])
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig,'raw/rw_sign_rho_hist.svg')
close

%% boxplot
fig = figure;
boxplot(abs(RHO(PVAL<=0.05)), 'Symbol', 'o')
ylim([0, 1])
yticks([0, 0.5, 1])
xticks([])
ylabel('correlation magnitude |\rho|')
title({'magnitudes of', 'significant', 'correlations'})
set(gcf, 'Position', [10 10 150 300]);
saveas(fig,'raw/rw_boxplot.svg')

%% METHOD DEFINITIONS
function x = randwalk(n_steps)
% random walk with 2 choices at each step:
% go up (50 %)
% go down (50 %)
x = zeros(1,n_steps);
x(1) = 0;
for t = 2 : n_steps
    choice = randi(2);
    if choice == 1
        x(t) = x(t-1) + 1;
    else
        x(t) = x(t-1) - 1;
    end
end
end

function x = ar_gauss(n_steps, r, mu, sig, x0)
% random walk with gaussian steps
x = zeros(1,n_steps);
x(1) = x0;
for t = 2 : n_steps
    x(t) = r * x(t-1) + normrnd(mu,sig);
end
end

function p = pvalPearson(tail, rho, n)
%PVALPEARSON Tail probability for Pearson's linear correlation.
% This function was written by MathWorks and is from corr.m

% default for tail is 'b'
% rho is just the pearson correlation
% n is the number of data points
t = rho.*sqrt((n-2)./(1-rho.^2)); % +/- Inf where rho == 1
switch tail
case 'b' % 'both or 'ne'
    p = 2*tcdf(-abs(t),n-2);
case 'r' % 'right' or 'gt'
    p = tcdf(-t,n-2);
case 'l' % 'left' or 'lt'
    p = tcdf(t,n-2);
end
end

function rho = pcorr(x,y)
cov_ = cov(x,y);
rho = cov_(1,2) ./ sqrt(cov_(1,1) .* cov_(2,2));
end

function p = bstrap_pval(x,y,n_bstraps)
% compute p-value for Pearson correlation
% using the univariate bootstrap procedure of
% Lee & Rodgers. (1998). Psych. Meth. Vol 3, No 1, p 91-103
rho_obs = pcorr(x,y);
RHO_ = zeros(n_bstraps,1);
for i = 1 : n_bstraps
    x_ = randsample(x, length(x), true);
    y_ = randsample(y, length(y), true);
    RHO_(i) = pcorr(x_, y_);
end
p = sum(abs(RHO_) >= abs(rho_obs)) / n_bstraps;
end
