clear all
close all
stylefile
%% PAIRWISE CASE, STATIONARY TIME SERIES
% random number seed and parameters
rng(1);
n_steps = 1000;
n_perm = 10^4-1;
burn = 1000;
r = 0.75;
mu = 10;
sig = 1;
x0 = 40;
% simulate two random walks
x = ar_gauss(n_steps+burn, r, mu, sig, x0);
x = x(burn+1:end);
y = ar_gauss(n_steps+burn, r, mu, sig, x0);
y = y(burn+1:end);
z = ar_gauss(n_steps+burn, r, mu, sig, x0);
z = z(burn+1:end);

% plot the result
fig1 = figure(1);
plot(x); hold on; plot(y); plot(z)
xticks([1,500,1000])
xticklabels([1001, 1500, 2000])
ylim([30, 50])
xlabel('time step')
ylabel('biomass')
legend('x', 'y', 'z')
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig1,'raw/ar2.svg')
close
%%
fig1 = figure(1);
scatter(x, z, 10, 'k', 'filled')
xlabel('x')
ylabel('z')
pval = permtest(x,z,n_perm)
rho = pcorr(x,z)
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 250 250]);
set(findall(gcf,'-property','FontSize'),'FontSize',14)
saveas(fig1,'raw/ar_scatter.svg')
close

%% 1000 AR(1) simulations
rng(1);
% perform 1000 AR(1) simulations and record results
n_trials = 1000;
RHO = zeros(1, n_trials);
PVAL = zeros(1, n_trials);
X = zeros(n_steps, n_trials);
Y = zeros(n_steps, n_trials);
for trial = 1 : n_trials
    x = ar_gauss(n_steps+burn, r, mu, sig, x0);
    x = x(burn+1:end);
    y = ar_gauss(n_steps+burn, r, mu, sig, x0);
    y = y(burn+1:end);
    X(:,trial) = x';
    Y(:,trial) = y';
    RHO(trial) = pcorr(x,y);
    PVAL(trial) = permtest(x,y,n_perm);
end
%% save X and Y data
csvwrite('raw/ar_xdata.csv', X);
csvwrite('raw/ar_ydata.csv', Y);

%% save rho and pval
csvwrite('raw/ar_data.csv', [RHO;PVAL]')

%% make pie chart
fig = figure;
frac_sig = sum(PVAL <= 0.05)/ n_trials;
piedata = [frac_sig, 1-frac_sig];
pie(piedata)
saveas(fig,'raw/ar_piechart.svg')

%% make histogram of significant rho values
fig = figure;
histogram(abs(RHO(PVAL<=0.05)), 'Normalization', 'probability')
xlabel('|\rho|')
ylabel('frequency observed')
title({'magnitudes of','significant correlations'})
xlim([0, 1])
xticks([0 : 0.25 : 1])
set(gca, 'ticklength', [0.020, 0.05])
set(gcf, 'Position', [10 10 300 300]);
saveas(fig,'raw/ar_sign_rho_hist.svg')
close

%% boxplot
fig = figure;
boxplot(abs(RHO(PVAL<=0.05)), 'Symbol', 'o')
ylim([0, 1])
yticks([0, 0.5, 1])
xticks([])
ylabel('correlation magnitude |\rho|')
title({'magnitudes of', 'significant', 'correlations'})
set(gcf, 'Position', [10 10 150 300]);
saveas(fig,'raw/ar_boxplot.svg')

%% METHOD DEFINITIONS
function x = randwalk(n_steps)
% random walk with 2 choices at each step:
% go up (50 %)
% go down (50 %)
x = zeros(1,n_steps);
x(1) = 0;
for t = 2 : n_steps
    choice = randi(2);
    if choice == 1
        x(t) = x(t-1) + 1;
    else
        x(t) = x(t-1) - 1;
    end
end
end

function x = ar_gauss(n_steps, r, mu, sig, x0)
% random walk with gaussian steps
x = zeros(1,n_steps);
x(1) = x0;
for t = 2 : n_steps
    x(t) = r * x(t-1) + normrnd(mu,sig);
end
end

function p = pvalPearson(tail, rho, n)
%PVALPEARSON Tail probability for Pearson's linear correlation.
% This function was written by MathWorks and is from corr.m

% default for tail is 'b'
% rho is just the pearson correlation
% n is the number of data points
t = rho.*sqrt((n-2)./(1-rho.^2)); % +/- Inf where rho == 1
switch tail
case 'b' % 'both or 'ne'
    p = 2*tcdf(-abs(t),n-2);
case 'r' % 'right' or 'gt'
    p = tcdf(-t,n-2);
case 'l' % 'left' or 'lt'
    p = tcdf(t,n-2);
end
end

function p = bstrap_pval(x,y,n_bstraps)
% compute p-value for Pearson correlation
% using the univariate bootstrap procedure of
% Lee & Rodgers. (1998). Psych. Meth. Vol 3, No 1, p 91-103
rho_obs = pcorr(x,y);
RHO_ = zeros(n_bstraps,1);
for i = 1 : n_bstraps
    x_ = randsample(x, length(x), true);
    y_ = randsample(y, length(y), true);
    RHO_(i) = pcorr(x_, y_);
end
p = sum(abs(RHO_) >= abs(rho_obs)) / n_bstraps;
end
