import numpy as np
import os
from shutil import copyfile

def get_mvgc_pvals(data, temp_folder='tmp', remove_temp=True,
                                verbose=True):
    """
    Runs mvgc on a dataset
    Args:
        data (list): list of datasets. Each dataset is a 2d numpy array with
            dimensions n_obs x n_vars.
        temp_folder (string): temporary folder
        remove_temp (bool): If True, removes temporary files after running.
        verbose (bool): If True, returns p-values and info. If False, returns
            p-values only.
    Returns:
        pvals: list of 2d matrices containing Granger causality p-values
        valid: list of booleans. valid[i] is False if MGVC encountered an error
            and True otherwise
        info: Matlab output
    """
    os.makedirs(temp_folder, exist_ok=True)
    # save data as csvs
    n_datasets = len(data)
    input_list = []
    output_list = []
    for i in range(n_datasets):
        np.savetxt(os.path.join(temp_folder,'data_%d.csv' % i),
                   data[i], delimiter=',')
        input_list.append("'" + ("data_%d.csv" % i) + "'")
        output_list.append("'" + ("pval_%d.csv" % i) + "'")
    # write MATLAB script
    input_paths = "{" + ",".join(input_list) + "}"
    output_paths = "{" + ",".join(output_list) + "}"
    lbracket = "{"
    rbracket = "}"
    program_string = f"""
mvgc_root = '/Users/alexyuan/OneDrive - Fred Hutchinson Cancer Research Center/Alex/software/mvgc_v1.0';
addpath(mvgc_root)
startup
input_paths = {input_paths};
output_paths = {output_paths};
for i = 1 : length(input_paths)
    mvgc_test(input_paths{lbracket}i{rbracket}, output_paths{lbracket}i{rbracket})
end"""
    program_file = open(os.path.join(temp_folder, "program.m"), "w")
    program_file.write(program_string)
    program_file.close()
    # copy mvgc test function into temp folder
    copyfile('mvgc_test.m', f"{temp_folder}/mvgc_test.m")
    copyfile('gcca_test.m', f"{temp_folder}/gcca_test.m")
    # run system command
    cmd = f"""cd {temp_folder}
    /Applications/MATLAB_R2018b.app/bin/matlab -nodisplay -nojvm -nosplash -nodesktop -r "program; quit" > info.txt
    """
    os.system(cmd)
    # collect p-values
    pvals = []
    valid = []
    for i in range(n_datasets):
        result = np.loadtxt(os.path.join(temp_folder, f"pval_{i}.csv"),
                            delimiter=",")
        pvals.append(result)
        valid.append({True: (1,1), False: (0,0)}[-1 in result])
    # collect info
    if verbose:
        with open(os.path.join(temp_folder, "info.txt"),"r") as info_file:
            info = info_file.read()
    # remove temp files
    if remove_temp:
        cmd = f"rm -r {temp_folder}"
        os.system(cmd)
    if verbose:
        return pvals, valid, info
    else:
        return pvals, valid
