%% MVGC "GCCA compatibility mode" demo
%
% Demonstrates usage of the MVGC toolbox in "GCCA compatibility mode"; see
% <mvgchelp.html#6 Miscellaneous issues> in the Help documentation. This is
% partly for the benefit of former users of the Granger Causal Connectivity
% Analysis (<http://www.sussex.ac.uk/Users/anils/aks_code.htm GCCA>) Toolbox
% [2], and partly as an implementation of a more "traditional" approach to
% Granger causality computation. The chief difference is that here two separate
% VAR regressions - the _full_ and _reduced_ regressions (see [1]) - are
% explicitly performed (see <GCCA_tsdata_to_pwcgc.html
% |GCCA_tsdata_to_pwcgc|>), in contrast to the MVGC Toolbox preferred
% approach (see <mvgc_demo.html |mvgc_demo|>), which only requires a full
% regression and is consequently more flexible and numerically accurate.
%
% Granger-causal pairwise-conditional analysis is demonstrated on generated
% VAR data for a 5-node network with known causal structure (see
% <var5_test.html |var5_test|>), as in the main MVGC Toolbox demonstration
% script, <mvgc_demo.html |mvgc_demo|>. A drawback of the traditional dual
% regression approach is that in the frequency domain, _conditional_
% spectral causalities cannot be estimated to an acceptable standard; see
% [1] and <GCCA_tsdata_to_smvgc.html |GCCA_tsdata_to_smvgc|> for more
% detail.
%
%% References
%
% [1] L. Barnett and A. K. Seth,
% <http://www.sciencedirect.com/science/article/pii/S0165027013003701 The MVGC
%     Multivariate Granger Causality Toolbox: A New Approach to Granger-causal
% Inference>, _J. Neurosci. Methods_ 223, 2014
% [ <matlab:open('mvgc_preprint.pdf') preprint> ].
%
% [2] A. K. Seth, "A MATLAB toolbox for Granger causal connectivity analysis",
% _J. Neurosci. Methods_ 186, 2010.
%
% (C) Lionel Barnett and Anil K. Seth, 2012. See file license.txt in
% installation directory for licensing terms.
%
%% setup and load in data
function F = gcca_test(X)
%% Parameters

ntrials   = 1;     % number of trials
[~, nobs]      = size(X);   % number of observations per trial

regmode   = 'OLS';  % VAR model estimation regression mode ('OLS', 'LWR' or empty for default)
icregmode = 'LWR';  % information criteria regression mode ('OLS', 'LWR' or empty for default)

morder    = 'AIC';  % model order to use ('actual', 'AIC', 'BIC' or supplied numerical value)
momax     = 50;     % maximum model order for model order estimation

tstat     = '';     % statistical test for MVGC:  'chi2' for Geweke's chi2 test (default) or'F' for Granger's F-test
alpha     = 0.05;   % significance level for significance test
mhtc      = 'FDR';  % multiple hypothesis test correction (see routine 'significance')

seed      = 0;      % random seed (0 for unseeded)

%% Model order estimation

% Calculate information criteria up to max model order

ptic('\nGCCA: *** tsdata_to_infocrit\n');
[AIC,BIC] = tsdata_to_infocrit(X,momax,icregmode);
ptoc('GCCA: *** tsdata_to_infocrit took ');

[~,bmo_AIC] = min(AIC);
[~,bmo_BIC] = min(BIC);

% Suppressing plot.

%figure(1); clf;
%plot((1:momax)',[AIC BIC]);
%legend('AIC','BIC');

fprintf('\nGCCA: best model order (AIC) = %d\n',bmo_AIC);
fprintf('GCCA: best model order (BIC) = %d\n',bmo_BIC);

% Select model order

if strcmpi(morder,'AIC')
    morder = bmo_AIC;
    fprintf('\nGCCA: using AIC best model order = %d\n',morder);
elseif strcmpi(morder,'BIC')
    morder = bmo_BIC;
    fprintf('\nGCCA: using BIC best model order = %d\n',morder);
else
    fprintf('\nGCCA: using specified model order = %d\n',morder);
end

assert(morder ~= momax,'Reached maximum model order');

%% Granger causality estimation

% Calculate time-domain pairwise-conditional causalities. Return VAR parameters
% so we can check VAR.

ptic('\nGCCA: *** GCCA_tsdata_to_pwcgc... ');
[F,A,SIG] = GCCA_tsdata_to_pwcgc(X,morder,regmode); % use same model order for reduced as for full regressions
ptoc;

% Check for failed (full) regression

assert(~isbad(A),'GCCA: VAR estimation failed');

% Check for failed GC calculation

assert(~isbad(F,false),'GCCA: GC calculation failed');

% Check VAR parameters (but don't bail out on error - GCCA mode is quite forgiving!)

rho = var_specrad(A);
fprintf('\nGCCA: spectral radius = %f\n',rho);
if rho >= 1,       fprintf(2,'GCCA: WARNING: unstable VAR (unit root)\n'); end
if ~isposdef(SIG), fprintf(2,'GCCA: WARNING: residuals covariance matrix not positive-definite\n'); end

% Significance test using theoretical null distribution, adjusting for multiple
% hypotheses.

%pval = mvgc_pval(F,morder,nobs,ntrials,1,1,nvars-2,tstat);
%sig  = significance(pval,alpha,mhtc);

end