%% Simulation of the aggregative groups
% Force profile looks like the following:
% Steric forces: Harmonic potential between overlapping cells
% Sticky forces: Harmonic potential for cells within a neighborhood
% Directed growth: big cells reproduce big cells, little cells reproduce
% little cells.

% Go to simulation data folder:
folder = '';
cd(folder);

% Inputs:
N               = 50;   % N sims
Ngens           = 5;    % Number of generations to bud
THETA           = 0;    % bud angle
DTHETA          = 15;   % bud angle spread
include_forces  = 1;    % do we include forces?
prob_same       = .9;    % probability that daughter is the same type as the mother
CHITIN_STRENGTH = 0;    % strength of chitin interactions
STERIC_STRENGTH = 3;    % strength of steric interactions
STICKY_STRENGTH = .25;  % strength of sticky interactions
mobility_pos    = .01;  % mobility
T               = 50;   % number of steps to allow relaxation
plot_opts       = 0;    % show plots?

% Polydispersity of sizes:
R1 = 1;
R2 = 2;

% Loop through N groups:
cluster = cell(N,1);
COM     = zeros(3,N);
Rad     = zeros(1,N);
for m = 1:N
    % Initialization:
    Ncells = 2^Ngens;
    cell_list = [];
    
    % First little cell:
    cell_list(1).Center  = [0;0;0];
    cell_list(1).Radii   = [R1; R1; R1];
    cell_list(1).Rmatrix = eye(3);
    cell_list(1).Chitin  = [];
    cell_list(1).Birth   = [];
    
    % First big cell:
    cell_list(2).Center  = [1;1;1];
    cell_list(2).Radii   = [R2; R2; R2];
    cell_list(2).Rmatrix = eye(3);
    cell_list(2).Chitin  = [];
    cell_list(2).Birth   = [];

    % Loop through generations:
    if plot_opts == 1
        figure;
    end
    tic
    for g = 1:Ngens

        % All cells divide:
        [cell_list] = DIVISION(cell_list, THETA, DTHETA, R1, R2, prob_same);

        % Visualize:
        if plot_opts == 1
            clf; hold on; box on; set(gca,'linewidth',2);
            view(3); axis equal;
            for n = 1:length(cell_list)
                [x,y,z] = VISUALIZE_ELLIPSOID(cell_list(n), 25);
                surf(x,y,z,'edgecolor','none','facealpha',1);
            end
            lightangle(30,30);
            lighting gouraud;
            material dull;
            drawnow;
        end

        % All cells rearrange:
        if include_forces == 1
            for t = 1:T
                % Rearrange
                [F_steric, F_sticky, F_chitin] = FORCES(cell_list, STERIC_STRENGTH, STICKY_STRENGTH, CHITIN_STRENGTH);
                [cell_list] = REARRANGE(cell_list, F_steric, F_sticky, F_chitin, mobility_pos);

                % Visualize:
                if plot_opts == 1
                    clf; hold on; box on; set(gca,'linewidth',2);
                    title(num2str(t));
                    view(3); axis equal;
                    for n = 1:length(cell_list)
                        [x,y,z] = VISUALIZE_ELLIPSOID(cell_list(n), 25);
                        surf(x,y,z,'edgecolor','none','facealpha',1);
                    end
                    lightangle(30,30);
                    lighting gouraud;
                    material dull;
                    drawnow;
                end
            end
        end
    end
    toc
    
    % Record and save to cell structure:
    cluster{m} = cell_list;
    savename   = 'cell_list.mat';
    if mod(m,10) == 0
        save(savename,'cluster');
    end
    
    % Record and save cell centers for voro++:
    centers   = [cell_list.Center];
    COM(:,m)  = mean(centers, 2);
    rCOM      = centers - COM(:,m);
    Rads      = vecnorm(rCOM);
    Rad(m)    = max(Rads) + 5;
    save_info = [1:length(cell_list); centers];
    fid = fopen(['cid=',num2str(m,'%03.f'),'.txt'],'w');
    fprintf(fid,'%d %f %f %f\n',save_info);
    fclose(fid);
end

% Store COM and R to file for each cluster:
fid = fopen('COM_file.txt','w');
fprintf(fid, '%f %f %f %f\n', [COM; Rad]);
fclose(fid);
cd ..;

%% FUNCTIONS
function [x,y,z] = VISUALIZE_ELLIPSOID(cell_of_interest, resolution)
%%C: the covariance matrix.
%%Dir: direction of the estimate, to be plotted together with the DT ellipsoid.
%%M: the mean vector, usually 0 in case of DT.
%%speed: time to pause between plotting, lowervalue = faster.
%%Dir: 1 or -1 for clockwise, anticlockwise.
%%time: number of iterations for which rotation is needed, higher = longer.
%%example: visualizeDTrot(diag([17 2 2]),[0 0 0],0.4,1,100)

% Extract info from cell_of_interest:
radii = cell_of_interest.Radii;
centers = cell_of_interest.Center;
R = cell_of_interest.Rmatrix;

% Generate data for "unrotated" ellipsoid
[xc,yc,zc] = ellipsoid(0,0,0,radii(1),radii(2),radii(3), resolution);

% Rotate data with orientation matrix R and center T
a = kron(R(:,1), xc);
b = kron(R(:,2), yc);
c = kron(R(:,3), zc);
data = a+b+c; n = size(data,2);

% Store for output:
x = data(1:n,:) + centers(1); 
y = data(n+1:2*n,:) + centers(2); 
z = data(2*n+1:end,:) + centers(3);

end

function [cell_list] = DIVISION(cell_list, THETA, DTHETA, R1, R2, prob_same)
    
    % List of polar and azimuthal angles:
    theta = DTHETA*randn(length(cell_list),1) + THETA;
    theta = deg2rad(theta);
    phi   = 360*rand(length(cell_list),1);
    phi   = deg2rad(phi);
    Rad   = [cell_list.Radii];
    Rad   = Rad(1,:);
    x     = sin(theta).*cos(phi);
    y     = sin(theta).*sin(phi);
    z     = cos(theta);
    
    % Create new cell on the surface of the mother cell:
    for n = 1:length(cell_list)
        
        % Determine position of new cell:
        [S,R,T] = GET_SURFACE_MATRICES(cell_list(n), 0);
        M = T*R*S;
        rg = M*[x(n);y(n);z(n);1];
        cg = M*[0;0;0;1];
        vg = rg - cg;
        vg = [vg(1:3)/norm(vg(1:3)); 0];
        newCell.Center = cg(1:3) + vg(1:3);
        
        % Determine radius of new cell:
        oldRad = Rad(n);
        if oldRad == R1
            weights = [prob_same, 1-prob_same];
        else
            weights = [1-prob_same, prob_same];
        end
        newRad = randsample([R1,R2],1,1,weights);
        newCell.Radii  = [newRad; newRad; newRad];
        newCell.Birth  = [n; size(cell_list(n).Chitin, 2)+1];
        newCell.Chitin = [];
        
        % Set rotation matrix:
        a = vg(1:3);
        b = [1; 1; -(a(1) + a(2))/a(3)];
        b = b/norm(b);
        c = cross(a,b);
        newCell.Rmatrix = [a, b, c];
        cell_list = [cell_list, newCell];
        
        % Record new bud scar:
        cell_list(n).Chitin = [cell_list(n).Chitin, [theta(n); phi(n); length(cell_list)] ];
    end
    
end

function [F_steric, F_sticky, F_chitin] = FORCES(cell_list, STERIC_STRENGTH, STICKY_STRENGTH, CHITIN_STRENGTH)

    F_steric = zeros(3,length(cell_list)); % steric forces
    F_sticky = zeros(3,length(cell_list)); % sticky forces
    F_chitin = zeros(3,length(cell_list)); % chitin forces
    cs = [cell_list.Center]; % list of all cell centers
    rs = [cell_list.Radii];
    rs = rs(1,:);
    for n = 1:length(cell_list)
        c     = cell_list(n).Center; % coi center
        rad   = cell_list(n).Radii(1); % cell radius
        r     = cs - c; % vector pointing from current cell to other cells
        R     = vecnorm(r); % distance between coi and all other cells
        ov    = R < rad + rs; % overlapping cells
        ov(n) = 0; % don't include current cell in its own forces
        nb    = R < rad+rs & R > rad+rs; % nearby cells
        nb(n) = 0; % don't include current cell in its own forces
        
        % Steric forces:
        Fmag = STERIC_STRENGTH * (R - (rad+rs)) .* ov;
        Fdir = r;
        Fs   = Fmag .* Fdir;
        F_steric(:,n) = sum(Fs, 2);
        
        % Sticky forces:
        Fmag = STICKY_STRENGTH * (R - .9*(rad+rs)) .* nb;
        Fdir = r;
        Fs   = Fmag .* Fdir;
        F_sticky(:,n) = sum(Fs, 2);
        
        % Chitin bond forces:
        % From daughter cells
        if CHITIN_STRENGTH > 0
        if ~isempty(cell_list(n).Chitin)
        cell_m = cell_list(n);
        [Sm,Rm,Tm] = GET_SURFACE_MATRICES(cell_m, 0);
        f_buds = zeros(3,size(cell_list(n).Chitin, 2));
        for j = 1:size(cell_list(n).Chitin, 2)
            budscar    = cell_m.Chitin(:,j);
            theta      = budscar(1);
            phi        = budscar(2);
            x          = sin(theta).*cos(phi);
            y          = sin(theta).*sin(phi);
            z          = cos(theta);
            r_bud      = Tm * Rm * Sm * [x;y;z;1];
            cell_d     = cell_list(budscar(3));
            [Sd,Rd,Td] = GET_SURFACE_MATRICES(cell_d, 0);
            r_birth    = Td * Rd * Sd * [-1; 0; 0; 1];
            r_disp     = r_birth(1:3) - r_bud(1:3);
            d_disp     = norm(r_disp);
            if d_disp >= 2
                f_buds(:,j) = CHITIN_STRENGTH * (d_disp - 2) * (r_disp/d_disp);
            end
        end
        end
        
        % From mother cell
        f_births = zeros(3,1);
        if ~isempty(cell_list(n).Birth)
        cell_m = cell_list(cell_list(n).Birth(1));
        cell_d = cell_list(n);
        [Sm,Rm,Tm] = GET_SURFACE_MATRICES(cell_m, 0);
        [Sd,Rd,Td] = GET_SURFACE_MATRICES(cell_d, 0);
        budscar    = cell_m.Chitin(:,cell_d.Birth(2));
        theta      = budscar(1);
        phi        = budscar(2);
        x          = sin(theta).*cos(phi);
        y          = sin(theta).*sin(phi);
        z          = cos(theta);  
        r_bud      = Tm * Rm * Sm * [x;y;z;1];
        r_birth    = Td * Rd * Sd * [-1;0;0;1];
        r_disp     = r_bud(1:3) - r_birth(1:3);
        d_disp     = norm(r_disp);
        if d_disp >= 2
            f_births = CHITIN_STRENGTH * (d_disp - 2) * (r_disp/d_disp);
        end
        end
        F_chitin(:,n) = sum([f_buds, f_births], 2);
        
        else % no chitin bonds
        
        F_chitin(:,n) = zeros(3,1);
        
        end
    end
        
end

function [cell_list] = REARRANGE(cell_list, F_steric, F_sticky, F_chitin, mobility_pos)

    % Move every cell in the list:
    Fnet = F_steric + F_sticky + F_chitin;
    for n = 1:length(cell_list)
        center = cell_list(n).Center;
        deltaX = Fnet(:,n) * mobility_pos;
        cell_list(n).Center = center + deltaX;
    end

end

function [S, R, T] = GET_SURFACE_MATRICES(cell_of_interest, scaling)

% Scaling matrix
S = [cell_of_interest.Radii(1) + scaling, 0, 0, 0;...
        0, cell_of_interest.Radii(2) + scaling, 0, 0; ...
        0, 0, cell_of_interest.Radii(3) + scaling, 0; ...
		0,0,0,1];
%  cell wall is usually ~75 nm thick, this accounts for that

% rotation matrix
R = cell_of_interest.Rmatrix;
R = [R; 0,0,0]; R = horzcat(R,[0;0;0;1]);

% translation matrix
T = eye(4); 
T(1,end) = cell_of_interest.Center(1);
T(2,end) = cell_of_interest.Center(2);
T(3,end) = cell_of_interest.Center(3);

end