%% SIMULATION OF ALGAL GROWTH IN SACS:
% Algae will have a single cell contained within a cell wall sac. We model
% this sac as a sphere. The single cell and the sac along with it will
% continue to grow over time until it reaches a size where the cell is too
% big for effective nutrient diffusion (or some other similar moment). The
% cell will then divide rapdily into many cells. Each of these cells
% retains its own cell wall sac. However, because this division process
% occurs rapidly (how fast exactly?) we can estimate that the packing
% fraction within the main sac will stay constant. Then, the question is,
% does this cell division result in higher local stresses on the sac? And,
% if it does, at what point will those local stresses rupture the sac?  In
% this case, the life cycle is similar to that of the fixed bonds...cell
% division happens until one local stress exceeds the membrane's threshold.

% Input parameters:
Ntrials           = 200; % number of simulations
phi               = .3; % starting packing fraction
radius            = 1; % cell radius
delta             = 2*radius; % optimal distance separating cells
steric_strength   = 1; % strength of steric interactions between cells
membrane_strength = 2; % strength of interactions with the membrane
thresh            = membrane_strength*10;
epsilon           = 1;
dt                = .1; % time step size
relax_time        = 500; % how many time steps for relaxation
n_gens            = 6; % generations of cellular growth
plot_opts         = 0; % view plots?
figure_save       = 0; % save figures?
data_save         = 1; % save centers for voro?

% Initialize:
centers = cell(Ntrials,1);

% Main script running simulations:
tic
for m = 1:Ntrials
    fprintf(['Trial # ',num2str(m,'%02.f'),'\n']);
    [centers{m}] = algae_sim(phi, delta, steric_strength, membrane_strength, thresh, epsilon, dt, relax_time, n_gens, plot_opts);
end
toc

% Write centers to file:
if data_save == 1
    folder = 'C:\Users\Thomas\Dropbox (GaTech)\Yunker Lab\Data\Project Statistical Mechanics\Algae\';
    cd(folder);
    sim_name = ['Ntrials=',num2str(Ntrials),'_G=',num2str(n_gens),'_S=',num2str(relax_time),...
                '_phi=',num2str(phi)];
    sim_name = strrep(sim_name,'.','-');        
    mkdir(sim_name);
    cd(sim_name);
    filename = 'particles_cid=';
    fid1 = fopen('algae_COMfile.txt','w');
    for m = 1:Ntrials
        r = centers{m}';
        COM = mean(r,1);
        rCOM = (r - COM)';
        rCOM = vecnorm(rCOM);
        R = max(rCOM) + 2;
        list = (1:length(r))';
        printable = [list, r];
        COMprints = [m, COM(1), COM(2), COM(3), R];    
        fid = fopen([filename,num2str(m,'%03.f'),'.txt'],'w');
        fprintf(fid,'%i %f %f %f\n', printable');
        fclose(fid);
        fprintf(fid1,'%i %f %f %f %f\n', COMprints);
    end
    fclose(fid1);
end

%% FUNCTIONS

function [output_centers] = algae_sim(phi, delta, steric_strength, membrane_strength, thresh, epsilon, dt, S, G, plot_opts)
    
    % Make one cell in a membrane: ----------------------------------------
    centers   = cell(G,1);
    cell_list = [];
    cell_list.Center  = [0;0;0];
    cell_list.Radii   = [1; 1; 1];
    cell_list.Rmatrix = eye(3);
    R = set_sphere_radius(cell_list, phi); % membrane size

    % Visualize: ----------------------------------------------------------
    if plot_opts == 1
        figure; hold on; box on; set(gca,'linewidth',2);
        [x,y,z] = VISUALIZE_ELLIPSOID(cell_list, 20);
        surf(x,y,z,'edgecolor','none','facealpha',1);
        [x,y,z] = sphere(20);
        surf(R*x,R*y,R*z,'edgecolor','none','facealpha',.3,'facecolor',[.8,.8,.8]);
        view(3); axis equal;
        drawnow;
    end
    
    % Time Loop: ----------------------------------------------------------
    for t = 1:G
        % Divide cells and make group membrane:
        [cell_list] = divide_cells(cell_list);
        R           = set_sphere_radius(cell_list, phi);
        COM         = mean([cell_list.Center],2);
        [X,Y,Z]     = sphere(30);
        X = R*X + COM(1);
        Y = R*Y + COM(2);
        Z = R*Z + COM(3);

        % Cell forces push cells around:
        for s = 1:S
            % Obtain forces and displacements:
            [forces] = get_forces(cell_list, delta, R, COM, steric_strength, membrane_strength, thresh, epsilon);
            deltaX   = forces*dt;
            for n = 1:length(cell_list)
                cell_list(n).Center = cell_list(n).Center + deltaX(n,:)';
            end

            % Visualize:
            if plot_opts == 1
                clf; hold on; box on; set(gca,'linewidth',2);
                for n = 1:length(cell_list)
                    [x,y,z] = VISUALIZE_ELLIPSOID(cell_list(n), 20);
                    surf(x,y,z,'facealpha',1,'edgecolor','none');
                end
                surf(X,Y,Z,'edgecolor','none','facecolor',[.8,.8,.8],'facealpha',.3);
                view(3); axis equal;
                title(num2str(s));
                drawnow;
            end
        end
        centers{t} = [cell_list.Center];

        % Final structure determines the local stresses on the membrane sac:
        %{
        V_intersect{t} = zeros(length(cell_list),1);
        A_intersect{t} = V_intersect{t};
        V_outside{t}   = V_intersect{t};
        for n = 1:length(cell_list)
            x_cell = cell_list(n).Center;
            x_memb = COM(:,t);
            r_cell = x_cell - x_memb;
            d      = norm(r_cell);
            r      = 1;
            if d + r <= R(t) % cell is entirely inside sac
                V_intersect{t}(n) = 4/3*pi*r^3;
                A_intersect{t}(n) = 0;
            elseif d >= R(t) + r % cell is etirely outside sac
                V_intersect{t}(n) = 0;
                A_intersect{t}(n) = pi*r^2;
            else % cell is partially inside/outside sac
                V_intersect{t}(n) = (pi * (R(t) + r - d)^2 * (d^2 + 2*d*r - 3*r^2 + 2*d*R(t) + 6*r*R(t) - 3*R(t)^2))/(12*d);
                cross_radius      = min([1/(2*d) * sqrt( 4*d^2*R(t)^2 - (d^2 - r^2 + R(t)^2)^2 ), 1]);
                A_intersect{t}(n) = pi*cross_radius^2;
            end
            V_outside{t}(n) = 4/3*pi*r^3 - V_intersect{t}(n);
        end
        V_outside{t} = V_outside{t}./(4/3*pi*R(t)^3);
        %}
        
    end

    % Convert Forces to Pressures:
    %{
    Pressure = cell(G,1);
    Pbar = zeros(1,G);
    Psig = zeros(1,G);
    Pnet = zeros(1,G);
    for t = 1:G
        AreaSac = 4*pi*R(t)^2; % area of the entire membrane sac
        Fmag = V_outside{t}'; % force from the cells onto the sac
        CrossSection = A_intersect{t}'./AreaSac;
        Pressure{t} = Fmag./CrossSection;
        Pbar(t) = nanmean(Pressure{t});
        Psig(t) = nanstd(Pressure{t});
        Pnet(t) = nansum(Pressure{t});
    end
    %}

    % Visualize the force vectors:
    %{
    for t = 1:G
        rho  = vecnorm(centers{t} - COM(:,t));   
        r    = centers{t} - COM(:,t);
        plotr = zeros(size(r));
        Fmag = V_outside{t}';
        for n = 1:size(r,2)
            if Fmag(n) > 0
                plotr(:,n) = r(:,n)./rho(n);
            else
                plotr(:,n) = zeros(3,1);
            end
        end
        Fmag = Fmag./max(Fmag);
        for n = 1:length(Fmag)
            Fvec(:,n) = Fmag(n)*plotr(:,n);
        end

        figure('units','inches','position',[1,1,8,4]);
        subplot(1,2,1); hold on; box on; set(gca,'linewidth',2);
        [X,Y,Z] = sphere(20);
        X = R(t)*X + COM(1,t);
        Y = R(t)*Y + COM(2,t);
        Z = R(t)*Z + COM(3,t);
        surf(X,Y,Z, 'facecolor',[.8,.8,.8],'edgecolor','none','facealpha',.3);
        for n = 1:size(centers{t},2)
            [q,w,a] = sphere(10);
            q = q + centers{t}(1,n);
            w = w + centers{t}(2,n);
            a = a + centers{t}(3,n);
            surf(q,w,a,'facealpha',.5,'edgecolor','none');
        end
        plot3(centers{t}(1,:), centers{t}(2,:), centers{t}(3,:),'rx','linewidth',2,'markersize',9);
        view(3); axis equal;
        lightangle(-30,30);
        lighting gouraud;
        material dull;
        xlabel('x'); ylabel('y'); zlabel('z');

        subplot(1,2,2); hold on; box on; set(gca,'linewidth',2);
        [xs,ys,zs] = sphere(20);
        surf(xs,ys,zs,'facecolor',[1,1,1],'edgecolor','k','linewidth',1,'facealpha',.5);
        plot3(plotr(1,:),plotr(2,:),plotr(3,:),'rx','linewidth',2,'markersize',8);
        for n = 1:length(Fmag)
            plot3([plotr(1,n),plotr(1,n)+Fvec(1,n)],[plotr(2,n),plotr(2,n)+Fvec(2,n)],[plotr(3,n),plotr(3,n)+Fvec(3,n)],'b-','linewidth',2);
        end
        view(3); axis equal;
        title(num2str(t));
        xlabel('x'); ylabel('y'); zlabel('z');
    end
    %}

    % Record the number of force vectors exceeding some threshold value Fmax:
    %{
    for Fmax = 0:.2:1
        Nover = zeros(G,1);
        for t = 1:G
            Fmag = V_outside{t}';
            ix   = find(Fmag > Fmax);
            Nover(t) = length(ix);
        end
        figure; hold on; box on; set(gca,'linewidth',2);
        plot(2.^(1:G), Nover, 'k.','markersize',15);
        xlabel('Ncells'); ylabel('Nover');
        title(['F > ',num2str(Fmax)]);
        ylim([0,max(Nover)+1]);
        xlim([0,2^G + 2]);
    end
    %}
    
    % Output the final cell center configuration:
    output_centers = centers{end};

end

function [x,y,z] = VISUALIZE_ELLIPSOID(cell_of_interest, resolution)
%%C: the covariance matrix.
%%Dir: direction of the estimate, to be plotted together with the DT ellipsoid.
%%M: the mean vector, usually 0 in case of DT.
%%speed: time to pause between plotting, lowervalue = faster.
%%Dir: 1 or -1 for clockwise, anticlockwise.
%%time: number of iterations for which rotation is needed, higher = longer.
%%example: visualizeDTrot(diag([17 2 2]),[0 0 0],0.4,1,100)

% Extract info from cell_of_interest:
radii = cell_of_interest.Radii;
centers = cell_of_interest.Center;
R = cell_of_interest.Rmatrix;

% Generate data for "unrotated" ellipsoid
[xc,yc,zc] = ellipsoid(0,0,0,radii(1),radii(2),radii(3), resolution);

% Rotate data with orientation matrix R and center T
a = kron(R(:,1), xc);
b = kron(R(:,2), yc);
c = kron(R(:,3), zc);
data = a+b+c; n = size(data,2);

% Store for output:
x = data(1:n,:) + centers(1); 
y = data(n+1:2*n,:) + centers(2); 
z = data(2*n+1:end,:) + centers(3);

end

function cell_list_out = divide_cells(cell_list)
    % Every cell divides randomly
    cell_list_out = cell_list;
    for n = 1:length(cell_list)
        az = 2*pi*rand;
        th = pi*rand;
        x_new = 2*sin(th)*cos(az);
        y_new = 2*sin(th)*sin(az);
        z_new = 2*cos(th);
        newCell.Center  = [x_new; y_new; z_new];
        newCell.Radii   = [1; 1; 1];
        newCell.Rmatrix = eye(3);
        cell_list_out = [cell_list_out, newCell];
    end
    
end

function R = set_sphere_radius(cell_list, phi)
    % Enclose cells in a sphere such that packing fraction is preserved
    radii = [cell_list.Radii];
    v_c = 4/3*pi*radii(1,:).^3;
    V_c = sum(v_c);
    V_s = V_c/phi;
    R   = (3/4 * (1/pi) * V_s)^(1/3);
    
end

function [forces] = get_forces(cell_list, delta, R, COM, steric_strength, membrane_strength, thresh, epsilon)

    % Find distances between all pairs of cells:
    r = [cell_list.Center];
    X = r(1,:); Y = r(2,:); Z = r(3,:);
    Dx = X - X';
    Dy = Y - Y';
    Dz = Z - Z';
    D  = sqrt(Dx.^2 + Dy.^2 + Dz.^2); % net distance between all cell centers
    F  = steric_strength*(D - delta).^2; % force interaction matrix
    F  = F .* ~eye(size(F)); % cells don't self interact
    
    % Consider all steric interactions:
    F(D > 2) = 0; % cells only interact when overlapping
    Fsteric = zeros(size(F,1),3);
    for m = 1:size(F,2) % loop through every cell
        ix = find(F(:,m)); % find all nonzeros entries in interaction matrix
        Fmag = F(ix,m);
        Fdir = [Dx(ix,m), Dy(ix,m), Dz(ix,m)];
        Fdir = Fdir./vecnorm(Fdir,2,2);
        Fcell = zeros(length(Fmag), 3);
        for j = 1:length(Fmag)
            Fcell(j,:) = Fmag(j)*Fdir(j,:);
        end
        Fsteric(m,:) = sum(Fcell,1);
    end
    
    % Interactions with the membrane:
    x    = r - COM; % vector pointing from COM to each cell
    d    = vecnorm(x); % distance between each cell and membrane COM
    xdir = zeros(size(x)); % direction of force
    V    = zeros(size(d)); % overlapping amount
    for i = 1:length(d)
        % Overlapping volume
        if d(i) < (R - 1) % the spheres totally overlap
            V(i) = 4/3*pi;
        elseif d > (R+1) % the spheres don't overlap at all
            V(i) = 0;
        else % the spheres overlap a little bit
            V(i) = pi*(R + 1 - d(i)).^2 .* (d(i).^2 + 2*d(i) - 2 + 2*R*d(i) + 6*R - 3*R^2)./(12*d(i)); % volume of a spherical lens from wolfram
        end
        
        % Direction
        if d(i) > epsilon
            xdir(:,i) = -x(:,i)./d(i);
        else
            xdir(:,i) = [0;0;0];
        end
    end
    vcell = 4/3*pi;
    Fmag  = membrane_strength*(vcell - V);
    Fmag(Fmag > thresh) = thresh; % set a threshold for no explosions
    Fmembrane = (Fmag .* xdir)';
    
    % Net forces:
    forces = Fsteric + Fmembrane;


end