% This script distributes points randomly around the surface of a sphere.
% The user can input the number of cells belonging on the surface, the
% number of trials (organisms) to simulate, and some other less important
% input information such as the minimum distance separating each cell. As
% output, the script saves the list of cell centers (xyz coords) and the
% solid angles resulting from a Voronoi tessellation on a spherical
% surface.

%% Inputs:
M         = 100;                % number of trials (organisms)
n         = 50;                 % number of cells on the surface
diam      = 1.4*sqrt((4/n));    % minimum distance between each cell (found to be a good standard in convergence time)

plot_opts = 0;                  % should we show plots?
apop      = 1;                  % is there apoptosis?
radius    = 2.00;               % how local is apoptosis
Ncellsdie = 10;                 % how many cells die
    
a = pi*(diam/2)^2;      % minimum solid angle of a cell on the surface
fprintf(['Minimum solid angle = ',num2str(a),'\n']);

% Directory to save data to:
folder = 'C:\Users\Thomas\Dropbox (GaTech)\Yunker Lab\Data\Project Statistical Mechanics\Choanos\planned_apop\R=200';
cd(folder);

%% Distribute random points around the surface of a sphere:
XYZcoords = cell(M,1);
for m = 1:M
    
    % Track progress:
    fprintf([num2str(m), ' / ',num2str(M),'\n']);
    
    %% Place particles on a sphere
    centers = [1,0,0];
    i = 1;
    while i < n
        u = rand;
        v = rand;
        ph = 2*pi*u;
        th = acos(2*v-1);
        x = sin(th).*cos(ph);
        y = sin(th).*sin(ph);
        z = cos(th);
        temp = [x,y,z];
        dist = centers - temp;
        dist = vecnorm(dist,2,2);
        if all(dist > diam)
            centers = [centers; x,y,z];
        end
        i = size(centers,1);
    end
    
    %% Apoptosis:
    if apop == 1
        ix = randsample(1:n, 1); % choose one cell to die
        coi = centers(ix,:);
        dist = centers - coi;
        dist = vecnorm(dist'); % distance between cells and chosen cell
        dist(ix) = 100; % don't care about how far the current cell is from itself
        death_list = find(dist <= radius); % cells with the potential to die
        if length(death_list) > Ncellsdie
            death = randsample(death_list, Ncellsdie, 0); % cells that actually die
            death = [ix, death];
        else
            death = [ix, death_list];
        end
    else
        death = [];
    end
    keep = 1:n;
    keep(death) = [];
    
    %% Voronoi tessellation on the sphere:
    xyz  = centers(keep,:)';
    [Vertices, K, voronoiboundary, s] = voronoisphere(xyz);
    
    %% Record cell centers to file
    XYZcoords{m} = xyz;
    if mod(m, 10) == 0
        save('centers.mat','XYZcoords');
    end    
    
    %% Graphic:
    if plot_opts == 1
        figure;
        hold on; box on; set(gca,'linewidth',1);
        plot3(xyz(1,:), xyz(2,:), xyz(3,:), 'o','markerfacecolor','w','markeredgecolor','k','linewidth',1,'markersize',2);
        for k = 1:size(xyz,2)
            X = voronoiboundary{k};
            fill3(X(1,:), X(2,:), X(3,:),rand(1,3), 'edgecolor','k','linewidth',1,'facealpha',.8);
        end
        axis equal;
        view(3);
    end
    
    %% Save solid angles to file:
    omega{m} = s;
    if mod(m, 10) == 0
        save('volumes_volvox_sim.mat','omega');
    end
    
end

%{
%%
figure('units','centimeters','position',[1,1,8.9/2,8.9/2]);
hold on; box on; set(gca,'linewidth',1);
colors = bone(3);
C = zeros(length(s),3);
for k = 1:length(voronoiboundary)
    if s(k) > mean(s)+std(s)
        C(k,:) = colors(3,:);
    elseif s(k) < mean(s) - std(s)
        C(k,:) = colors(1,:);
    else
        C(k,:) = colors(2,:);
    end
    verx = voronoiboundary{k}';
    [K,~] = convhull(verx);
    trisurf(K,verx(:,1), verx(:,2), verx(:,3),'facecolor',C(k,:),'edgecolor','none','facealpha',1);
end
view(3); axis equal;
xticks([]); yticks([]); zticks([]);


%% Correlations in voronoi area and position:
N_neighbors = 20;

% Delaunary tri:
Tri = delaunayTriangulation(centers(:,1), centers(:,2), centers(:,3));
DT  = convexHull(Tri);

% Map distances from cells to each other:
Yk  = zeros(length(s),N_neighbors);
for b = 1:length(s)
    % Progress bar
    if mod(b,20) == 0
        fprintf(['b=',num2str(b),'\n']);
    end
    
    % find all K neighbors of a node:
    node = b;
    ix = FIND_K_NEIGHBORS(DT, node, N_neighbors);
    
    % mean difference in area
    A0 = s(node); % area of interest
    Bs = zeros(1,length(ix)); % this records the normalized difference between neighboring areas and area of interest
    L  = zeros(1,length(ix)); % this records the number of cells in the neighboring region
    for k = 1:length(ix)
        cell_list = ix{k};
        As    = s(cell_list);
        Bs(k) = mean(As - mean(s));
    end
    
    % Record to data structure:
    Yk(b,:) = Bs;
end

%% Correlation function
X     = s - mean(s);
sigX  = std(X);
sigYk = std(Yk,0,1);
CORR  = [1,mean(X.*Yk)./(sigX*sigYk)];

figure; hold on; box on; set(gca,'linewidth',1);
line([0,N_neighbors],[0,0],'linestyle','--','linewidth',1,'color','k');
plot(0:N_neighbors, CORR, 'ko-','markerfacecolor',[.65,.65,.65],'markersize',5,'linewidth',1);
save('random_sphere_correlations.mat','CORR');
%}

%% Analysis and plot:
%{
filename = 'volumes_volvox_sim.mat';
savename = [filename(1:end-4),'_omega.mat'];
load(filename);
omega = [];
for c = 1:length(choanos)
    omega = [omega; choanos{c}];
end

IQR = iqr(omega);
bw  = 2*IQR*length(omega)^(-1/3);
nb  = round((max(omega) - min(omega))/bw);
[C,ed] = histcounts(omega, nb, 'normalization','pdf');
bi  = mean([ed(1:end-1); ed(2:end)]);

sbar = mean(omega);
ssig = std(omega);
smin = pi*(diam/2)^2;
kgam = (sbar - smin)^2/ssig^2;
xgam = linspace(smin, max(omega), 1e3);
pgam = get_gamma(kgam, sbar, smin, xgam);

if plot_opts == 1
    figure;
    hold on; box on; set(gca,'linewidth',2);
    bar(bi, C, 1, 'facecolor',[.8,.8,.8],'edgecolor','none','facealpha',1);
    plot(xgam, pgam, 'k-','linewidth',1);
    xlabel('Solid Angle');
    ylabel('pdf');
end

%% Save to file:
save(savename,'omega','sbar','ssig','smin','kgam','xgam','pgam');
%}