# Late-life fitness gains explain the delay of the selection shadow in ants
# 
# Authors: Jaimes Nino, Heinze, J., LM, Oettler, J
# Affiliation: Zoologie/Evolutionsbiologie, Universität Regensburg

# Libraries used
library(openxlsx)
library(ggplot2)
library(ggbeeswarm)
library(gridExtra)
library(MASS)
library(geepack)
library(PMCMR)
library(PMCMRplus)
library(dplyr)
library(DHARMa)
library(glmmTMB)
library(mgcv)

# Population survey data
setwd("~/Documents/Tesis/01 _MS_Social_Aging")
col_dead <- read.xlsx("AgingCobs_popsurvey.xlsx",sheet = 1)
cols <- c("Code", "Replicate")
col_dead $queen.id <- do.call(paste, c(col_dead[cols], sep="_"))
# Dry weight measurements of freshly ecloded queens and workes
dry_weig <- read.xlsx("AgingCobs_popsurvey.xlsx",sheet = 2)
## Head width 
workers <- read.xlsx("AgingCobs_popsurvey.xlsx",sheet = 4)
## Field colony size 
size <- read.xlsx("AgingCobs_popsurvey.xlsx",sheet=5)

g_legend<-function(a.plot){
  tmp <- ggplot_gtable(ggplot_build(a.plot))
  leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
  legend <- tmp$grobs[[leg]]
  return(legend)}
####### Queen pupae #######
QueenPTotal <- aggregate(col_dead$Pupae_Q, by = list(col_dead$queen.id), sum)
Treatment <- c(rep(30,23), rep(20,19 ), rep (10, 21), rep(30, 11), rep(20, 15), rep(10, 10) )
QueenPTotal <- cbind(QueenPTotal, Treatment)
QueenPTotal$Treatment <- as.factor(QueenPTotal$Treatment)

p5 <- ggplot(data= QueenPTotal, aes(y = x, x = Treatment, group = Treatment, fill = Treatment))+
  geom_boxplot()+
  ylim(0,450)+
  geom_beeswarm(cex =  3)+
  #  ggtitle("Total No. Queen pupae, 99 colonies")+
  labs(x ="Number of workers", y= "No. queen pupae")+
  theme_minimal()+
  theme(axis.title.x = element_blank(),
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        legend.position = "none")+
  annotate("segment", x = 1, xend = 2, y = 380, yend=380 )+
  annotate("text", x = 1.5, y = 390, label = "**", size= 8)+
  annotate("segment", x = 1, xend = 3, y = 430, yend=430 )+
  annotate("text", x = 2, y = 440, label = "**", size= 8)

col_censored2 <- merge(QueenPTotal, col_dead, by.x = "Group.1", by.y="queen.id",  all=F) 
col_censored2 <- col_censored2[!duplicated(col_censored2$Group.1), ]

# glmmTMB test
fit.qp <- glmmTMB(x  ~  Treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=col_censored2, family = nbinom1) 
res1 <- simulateResiduals(fit.qp, plot =T)
plot(res1, col_censored2$W, rank = T)
summary(fit.qp)

#### Egg output   ####
EggsTotal <- aggregate(col_dead$Eggs, list(col_dead$queen.id), sum)
Treatment <- c(rep(30,23), rep(20,19 ), rep (10, 21), rep(30, 11), rep(20, 15), rep(10, 10) )
EggsTotal <- cbind(EggsTotal, Treatment)
EggsTotal$Treatment <- as.factor(EggsTotal$Treatment)

# Plot Eggs
p3 <- ggplot(data= EggsTotal, aes(y = x, x = Treatment, group = Treatment, fill = Treatment))+
  geom_boxplot()+
  ylim(0,1100)+
  labs(  y= "No. eggs")+
  theme_minimal()+
  geom_beeswarm(cex =  3)+
  theme(legend.position = "none",
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        axis.title.x = element_blank())

####  Stats Eggs total production ####
col_censored2 <- merge(EggsTotal, col_censored, by.x = "Group.1", by.y="Subject",  all=F) 
col_censored2 <- as.factor(col_censored2$Group.1)
col_censored2 <- col_censored2[!duplicated(col_censored2$Group.1), ]
head(col_censored2)
col_censored2$nW <- as.numeric(col_censored2$W)
fit.4 <- glmmTMB(x  ~  W + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=col_censored2, family = nbinom1) 
res1 <- simulateResiduals(fit.4, plot =T)
plot(res1, col_censored2$W, rank = T)
summary(fit.4)
# Family: nbinom1  ( log )
# Formula:          x ~ W + (1 | Setup_date) + (1 | Box) + (1 | Stock_box) + (1 |      Origin_nest)
# Data: col_censored2
# 
# AIC      BIC   logLik deviance df.resid 
# 1323.4   1344.2   -653.7   1307.4       91 
# 
# Random effects:
#   
#   Conditional model:
#   Groups      Name        Variance  Std.Dev. 
# Setup_date  (Intercept) 7.474e-02 2.734e-01
# Box         (Intercept) 5.031e-03 7.093e-02
# Stock_box   (Intercept) 9.295e-10 3.049e-05
# Origin_nest (Intercept) 1.046e-02 1.023e-01
# Number of obs: 99, groups:  Setup_date, 39; Box, 9; Stock_box, 4; Origin_nest, 21
# 
# Overdispersion parameter for nbinom1 family (): 94.5 
# 
# Conditional model:
#               Estimate Std. Error z value Pr(>|z|)    
# (Intercept)  5.76571    0.08636   66.76   <2e-16 ***
# W.L         -0.03829    0.10087   -0.38    0.704    
# W.Q         -0.10688    0.11187   -0.96    0.339 


######## Worker Pupae total output ####### 
WorkerTotal <- aggregate(col_dead$Pupae_W, list(col_dead$queen.id), sum)
WorkerTotal <- aggregate(col_dead$Pupae_W, list(col_dead$queen.id), function (x){ sum(x, na.rm=TRUE) })
Treatment <- c(rep(30,23), rep(20,19 ), rep (10, 21), rep(30, 11), rep(20, 15), rep(10, 10) )
WorkerTotal <- cbind(WorkerTotal, Treatment)
summary(WorkerTotal)
WorkerTotal$Treatment <- as.factor(WorkerTotal$Treatment)

p2 <- ggplot(data = WorkerTotal, aes(y =  x, x = Treatment, fill = Treatment))+
  geom_boxplot()+
  geom_quasirandom(width = 0.2)+
  ylim(0,1400)+
  ylab("No. worker pupae")+
  xlab("No. Workers")+
  theme_minimal()+
  theme(legend.position = "bottom",
        legend.title = element_text(size = 15),
        legend.text = element_text(size = 15),
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        axis.title.x = element_blank())+
  labs(fill = "No. of workers")

col_censored <- col_censored[!is.na(col_censored$Pupae_W),]
WoPuTotal <- aggregate(col_censored$Pupae_W, list(col_censored$Subject), sum)

col_censored2 <- merge(WoPuTotal, col_censored, by.x = "Group.1", by.y="Subject",  all=F) 
col_censored2 <- col_censored2[!duplicated(col_censored2$Group.1), ]
head(col_censored2)

fit.5 <- glmmTMB(x  ~  W + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=col_censored2, family = nbinom1) 
res1 <- simulateResiduals(fit.5, plot =T)
plot(res1, col_censored2$W, rank = T)
summary(fit.5)
# Family: nbinom1  ( log )
# Formula:          x ~ W + (1 | Setup_date) + (1 | Box) + (1 | Stock_box) + (1 |      Origin_nest)
# Data: col_censored2
# 
# AIC      BIC   logLik deviance df.resid 
# 1342.5   1363.2   -663.2   1326.5       91 
# 
# Random effects:
#   
#   Conditional model:
#   Groups      Name        Variance  Std.Dev. 
# Setup_date  (Intercept) 6.687e-02 2.586e-01
# Box         (Intercept) 1.215e-03 3.485e-02
# Stock_box   (Intercept) 8.447e-10 2.906e-05
# Origin_nest (Intercept) 1.817e-02 1.348e-01
# Number of obs: 99, groups:  Setup_date, 39; Box, 9; Stock_box, 4; Origin_nest, 21
# 
# Overdispersion parameter for nbinom1 family ():  137 
# 
# Conditional model:
#   Estimate Std. Error z value Pr(>|z|)    
# (Intercept)  5.75641    0.09517   60.48   <2e-16 ***
#   W.L          0.01022    0.11574    0.09    0.930    
# W.Q         -0.05016    0.12892   -0.39    0.697
#### Caste invest #####
caste_inv <- merge( QueenPTotal, WorkerTotal,
                    by = "Group.1", all = TRUE)
colnames(caste_inv) <- c("colony", "queenp", "treatment", "worker", "treatment2")
head(caste_inv)
caste_inv$treatment <- as.factor(caste_inv$treatment)
dry_weig$Weight_g <- as.numeric(dry_weig$Weight_g)
dry_weig$Per_ind_mg <- dry_weig$Weight_g/5
dry_weig$Per_ind_mg <- dry_weig$Per_ind_mg*1000
aggregate(dry_weig$Per_ind_mg, by = list(dry_weig$Caste), shapiro.test)
# Group.1         x
# 1   Queen 0.7869448
# 2  Worker 0.9698909

aggregate(dry_weig$Per_ind_mg,  by = list(dry_weig$Caste), mean)
# Group.1        x
# 1   Queen 0.063044
# 2  Worker 0.040006
caste_coeffic0.6 <- (0.063044 / 0.040006)^0.6 # 1.313745
caste_coeffic0.7 <- (0.063044 / 0.040006)^0.7 # 1.374874 
caste_coeffic0.8 <- (0.063044 / 0.040006)^0.8 # 1.438848
caste_coeffic0.9 <- (0.063044 / 0.040006)^0.9 # 1.505798
caste_coeffic1 <- (0.063044 / 0.040006)^1 # 1.575864

caste_inv$castecoef_queen0.6 <- caste_inv$queenp * caste_coeffic0.6
caste_inv$castecoef_queen0.7 <- caste_inv$queenp * caste_coeffic0.7
caste_inv$castecoef_queen0.8 <- caste_inv$queenp * caste_coeffic0.8
caste_inv$castecoef_queen0.9 <- caste_inv$queenp * caste_coeffic0.9
caste_inv$castecoef_queen1 <- caste_inv$queenp * caste_coeffic1

caste_inv$Q_QW_N <- caste_inv$queenp / (caste_inv$queenp + caste_inv$worker)
caste_inv$Q_QW_0.6 <- caste_inv$castecoef_queen0.6 / (caste_inv$castecoef_queen0.6 + caste_inv$worker)
caste_inv$Q_QW_0.7 <- caste_inv$castecoef_queen0.7 / (caste_inv$castecoef_queen0.7 + caste_inv$worker)
caste_inv$Q_QW_0.8 <- caste_inv$castecoef_queen0.8 / (caste_inv$castecoef_queen0.8 + caste_inv$worker)
caste_inv$Q_QW_0.9 <- caste_inv$castecoef_queen0.9 / (caste_inv$castecoef_queen0.9 + caste_inv$worker)
caste_inv$Q_QW_1 <- caste_inv$castecoef_queen1 / (caste_inv$castecoef_queen1 + caste_inv$worker)

p07ratio <- ggplot(data= caste_inv, aes(y = Q_QW_0.7, x = treatment, fill = treatment))+
  geom_boxplot()+
  ylim(0,0.6)+
  labs( x="No. of workers", y= "Qc/Qc+W investment")+
  theme_minimal()+
  theme(plot.title = element_text(hjust = 0.5),
        legend.position = "none",
        axis.title.x = element_blank(),
        axis.title.y = element_text(size = 15),
        axis.text = element_text(size = 15))+
  annotate("segment", x = 1, xend = 2, y = 0.45, yend=0.45 )+
  annotate("text", x = 1.5, y = 0.47, label = "***", size= 8)+
  annotate("segment", x = 1, xend = 3, y = 0.52, yend=0.52)+
  annotate("text", x = 2, y = 0.54, label = "***", size= 8)+
  geom_beeswarm(cex = 3)+
  scale_fill_manual(values=c("#F8766D", "#00BA38", "#619CFF"))

caste_inv2 <- merge(caste_inv, col_dead, by.x = "colony", by.y="queen.id",  all=F) 
caste_inv2 <- caste_inv2[!duplicated(caste_inv2$colony), ]

fit.qwc6 <- glmmTMB(Q_QW_0.6 ~  treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=caste_inv2) 
res1 <- simulateResiduals(fit.qwc, plot =T)
plot(res1, caste_inv2$treatment, rank = T)
summary(fit.qwc6)

fit.qwc7 <- glmmTMB(Q_QW_0.7 ~  treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=caste_inv2) 
res1 <- simulateResiduals(fit.qwc7, plot =T)
plot(res1, caste_inv2$treatment, rank = T)
summary(fit.qwc7)

fit.qwc8 <- glmmTMB(Q_QW_0.8 ~  treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=caste_inv2) 
res1 <- simulateResiduals(fit.qwc, plot =T)
plot(res1, caste_inv2$treatment, rank = T)
summary(fit.qwc8)

fit.qwc9 <- glmmTMB(Q_QW_0.9 ~  treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=caste_inv2) 
res1 <- simulateResiduals(fit.qwc, plot =T)
plot(res1, caste_inv2$treatment, rank = T)
summary(fit.qwc9)

fit.qwc1 <- glmmTMB(Q_QW_1 ~  treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Origin_nest) , data=caste_inv2) 
res1 <- simulateResiduals(fit.qwc, plot =T)
plot(res1, caste_inv2$treatment, rank = T)
summary(fit.qwc1)

# ###### Loess QueenPupae and Worker Pupae##########
recordsWeek <- as.data.frame.matrix(table(col_dead$Week, col_dead$Queen_alive))
colnames(recordsWeek) <- c("recordedDead", "Alive" )
recordsWeek$Week <- rownames(recordsWeek)
recordsWeek$Week <- as.numeric(unlist(recordsWeek$Week))

p1 <-ggplot(data = col_dead) +
  geom_smooth( aes(x= Week, y = Pupae_Q, color = W, fill = W, linetype = "twodash"), method = "auto", alpha = 0.2)+
  #  geom_jitter(aes(x= Week, y = Pupae_Q, color = W), height = 0.1, width = 0.5, size = 0.1, alpha = 0.4)+
  geom_smooth(aes(x=col_dead$Week, y=col_dead$Pupae_W, color = W, fill = W, linetype = "solid"), alpha = 0.2)+
  #  geom_jitter(aes(x= Week, y = Pupae_W, color = W), height = 0.1, width = 0.5, size = 0.1, alpha = 0.4)+
  scale_linetype_manual(name= "Pupae",
                        values = c( solid = "solid", twodash = "twodash"),
                        labels = c( "Worker", "Queen"))+
  xlab("Weeks")+
  ylab("No. pupae")+
  #  ggtitle("Weekly laying rate of pupae")+
  theme_minimal()+
  theme(legend.position = "bottom",
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        legend.text = element_text(size = 15),
        legend.title = element_text(size = 15))

#### Figure 1 ####
mylegend2<-g_legend(p1)
grid.arrange(
  arrangeGrob(
    p5, p07ratio, p3+ theme(legend.position="none"), 
    arrangeGrob(p1+ theme(axis.title.x = element_blank(),
                          legend.position="none"), nrow=1, ncol=1),
    ncol=2,heights=c(1,1)), 
  mylegend2, nrow = 2, heights=c(5, 1))

#### Head width ####
workers <- read.xlsx("AgingCobs_popsurvey.xlsx",sheet = 4)
# We make an unique identifier for our batch of workers. Each batch was collected from a specific colony, coming from a maternal colony, in an specific point in time. This means "Name of the maternal colony + Date collected". This will be useful later
cols_w <- c("MatColony", "Collection")
workers$colony_date<- do.call(paste, c(workers[cols_w], sep="_"))

# Now we merge both data sets in to one using the maternal colony identifier. I create a new object "measurement" for this, prefer always to create new objects, instead of overwriting them for security reasons. Tracking the process later and debugging its easier. You can check how the function "merge" works in the help search engine. 
col_dead$colony <- paste(col_dead$Code, col_dead$W, "_", col_dead$Replicate, sep ="")
col_deadHW <- col_dead[, -c(10:20)]
col_deadHW <- col_deadHW[!duplicated(col_deadHW$queen.id), ]
measurement <- merge(workers, col_deadHW,
                     by.x = "MatColony", by.y = "colony", all.y = F)

# I want to know, for each measurement of head of each worker, how old was the maternal colony until then. For this I substract the date when maternal colony was setup -  the collection of that worker was made. This "time" object is measured in days.

measurement$time <- measurement$Collection - measurement$Setup_date

measurement$W <- as.factor(measurement$W)
measurement$HW <- as.numeric(measurement$HW)
# Simple boxplot showing the headwidth lenght based on the treatment
plot(HW ~  W, data =measurement)
points(HW ~  W, data =measurement)

# We want a graph showing if there is any trend across time on the head width length.
#Data from 231 workers, and 76 biological replicates (meaning time points from colony)
p2b <- ggplot(data = measurement, aes(x=W, y=HW, fill = W))+
  geom_boxplot()+
  #  geom_beeswarm()+
  geom_quasirandom(width = 0.2)+
  ylab("Workers' head width")+
  xlab("No. Workers")+
  theme_minimal()+
  ylim(290,440)+
  theme(legend.position = "none",
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        axis.title.x = element_blank())+
  annotate("segment", x = 1, xend = 2, y = 415, yend=415 )+
  annotate("text", x = 1.5, y = 420, label = "**", size= 8)+
  annotate("segment", x = 1, xend = 3, y = 430, yend=430 )+
  annotate("text", x = 2, y = 435, label = "**", size= 8)

mylegend2<-g_legend(p1)
grid.arrange(
  arrangeGrob(
    p1+ theme(legend.position="none"),p2b + theme(legend.position="none"),
    nrow = 1),
  mylegend2, nrow = 2,heights=c(4, 1))


#### Select above and below mean ####
# Queen / Queen + Worker Pupae ratio across time ##
col_dead$Q_W_ratio <- c(col_dead$Pupae_Q / (col_dead$Pupae_Q + col_dead$Pupae_W))
p1 <- ggplot(data = col_dead, aes(x=Week, y=Q_W_ratio, color = W)) +
  geom_smooth(method = "auto" , level = 0.90)+
  geom_jitter(height = 0.1, width = 1.5, size = 0.5,  alpha = 0.5)+
  ylab("Q/Q+W ratio")+
  scale_color_discrete(name = "No. workers")+
  theme_minimal()+
  theme(legend.position = "top",
        legend.text=element_text(size=15),
        legend.title = element_text(size=15),
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        axis.title.x = element_blank())
# First remove the rows that the queen was dead but still data was recorded
col_dead_only <- col_dead[!(col_dead$Queen_alive == 0), ] 
# Obtain lifespan of queen
MaxWeek <- aggregate (col_dead_only$Week, list(col_dead_only$queen.id), max) 
colnames(MaxWeek) <- c("queen.id", "Lifespan")
summary(MaxWeek)

col_dead$Q_W_ratio <- c(col_dead$Pupae_Q / (col_dead$Pupae_Q + col_dead$Pupae_W))
above_qid <- MaxWeek[MaxWeek$Lifespan >= 25, ]
above_colonies <- col_dead[col_dead$queen.id %in% above_qid$queen.id, ]

below_qid <- MaxWeek[MaxWeek$Lifespan < 25, ]
below_colonies <- col_dead[col_dead$queen.id %in% below_qid$queen.id, ]

p2 <- ggplot(data = below_colonies) +
  geom_smooth( aes(x= Week, y = Q_W_ratio, color = W, fill = W), method = "auto", alpha = 0.2)+
  #  geom_jitter(aes(x= Week, y = Pupae_Q, color = W), height = 0.1, width = 0.5, size = 0.1, alpha = 0.4)+
  xlab("Weeks")+
  ylab("Q/Q+W ratio")+
  xlim(0,45)+
  ylim(0,1.2)+
  ggtitle("Below 25 weeks")+
  theme_minimal()+
  theme(plot.title = element_text(size = 15,  hjust = 0.5),
        legend.position = "none", 
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))

p3 <- ggplot(data = above_colonies) +
  geom_smooth( aes(x= Week, y = Q_W_ratio, color = W, fill = W), method = "auto", alpha = 0.2)+
  #  geom_jitter(aes(x= Week, y = Pupae_Q, color = W), height = 0.1, width = 0.5, size = 0.1, alpha = 0.4)+
  xlab("Weeks")+
  ylab("Q/Q+W ratio")+
  xlim(0,45)+
  ylim(0,1.2)+
  ggtitle("Above 25 weeks")+
  theme_minimal()+
  theme(plot.title = element_text(size = 15, hjust = 0.5),
        legend.position = "none",
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))
#### Figure 3 ####
grid.arrange(
  arrangeGrob(
    p1, 
    arrangeGrob(p2, p3, ncol = 2),
    nrow= 2))


# Queen / Queen + Worker Pupae per queen ##
colnames(QueenPTotal) <- c("queen.id", "QueenP", "Treatment")
colnames(WorkerTotal) <- c("queen.id", "WorkerP", "Treatment2")
Q_W_ratio99 <- merge(QueenPTotal, WorkerTotal, by = "queen.id")
Q_W_ratio99$ratio <- c(Q_W_ratio99$QueenP / (Q_W_ratio99$QueenP + Q_W_ratio99$WorkerP))
Q_W_ratio99$logratio <- log(Q_W_ratio99$ratio+1) 
Q_W_ratio99 <- merge(Q_W_ratio99, col_dead, by = "queen.id",  all=F) 
Q_W_ratio99 <- Q_W_ratio99[!duplicated(Q_W_ratio99$queen.id), ]
Q_W_ratio99$Treatment <- as.factor(Q_W_ratio99$Treatment)
Q_W_ratio99$Box_nest <- as.factor(Q_W_ratio99$Box_nest)
Q_W_ratio99$Stock_box <- as.factor(Q_W_ratio99$Stock_box)

Q_W_ratio99 <- Q_W_ratio99[, -c(17:31)]
Q_W_ratio99 <- Q_W_ratio99[, -c(8)]

fit.qw <- glmmTMB(logratio  ~  Treatment + (1 | Setup_date) + (1|Box) + (1|Stock_box) + (1|Box_nest) , data=Q_W_ratio99) 

res1 <- simulateResiduals(fit.qw, plot =T)
plot(res1, Q_W_ratio99$Treatment, rank = T)
summary(fit.qw)

##### Figure S1 ####
ggplot(data = col_dead, aes(x=Week, y=Workers, color = W)) +
  geom_smooth(method = "auto" )+
  geom_jitter(height = 0.1, width = 1.5, size = 0.5)+
  xlab("Weeks")+
  ylim(0,70)+
  ylab("No. of adult Workers")+
  scale_color_discrete(name = "No. workers")+
  theme_minimal()+
  theme(axis.title = element_text(size = 15),
        axis.text = element_text(size = 15),
        legend.position = "bottom",
        legend.title = element_text(size = 15),
        legend.text = element_text(size = 15))
##### Figure S2 ####
size <- size[apply(size!=0, 1, all),]
size$origin <- as.factor(size$origin)

ggplot(size, aes (x=origin, y=worker), aes(fill = factor(origin))) +
  theme_minimal()+
  theme(axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))+
  geom_boxplot(aes(fill = factor(origin))) + 
  ylim (0,237)+
  labs(x ="Population of Origin", y= "No. of workers")+
  annotate("segment", x = 2, xend = 3, y = 217, yend=217 )+
  annotate("text", x = 2.5, y = 220, label = "*", size= 6)+
  annotate("segment", x = 1, xend = 3, y = 228, yend=228 )+
  annotate("text", x = 2, y = 235, label = "n.s.", size= 5)+
  geom_beeswarm(cex =  3)+
  guides(fill=guide_legend(title="origin"))
#### Survival curve ###
# and Relative mortality and fecundity as a function of age.   

col_dead3 <- col_dead[!is.na(col_dead$Status),] ##do to get ride of lines of queens after they died
queen_dead <- subset(col_dead3, Status == 1) #queen with 1

#### Figure S3 #####
p1 <- ggsurvplot(
  fit = survfit(Surv(Week, Status) ~ W, data = queen_dead), 
  xlab = "Weeks", 
  ylab = "Overall survival probability",
  conf.int = TRUE,
  legend.title = "No. w
  orkers",
  legend = "bottom",
  font.legend = 15,
  font.y =15,
  font.tickslab = 15,
  legend.labs = c("10", "20", "30"))
p2 <- ggplot(data = queen_dead, aes( x = W, y = Week, fill = W))+
  geom_boxplot()+
  ylim(5,43)+
  xlab("No. workers")+
  ylab("Weeks")+
  theme_minimal()+
  theme(legend.position="bottom",
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))+
  geom_beeswarm(cex =  3)
mylegend<-g_legend(p2)
grid.arrange(
  arrangeGrob(
    p1$plot + theme(legend.position="none"), p2+ theme(legend.position="none"), nrow = 1), mylegend,heights=c(6,1),nrow = 2)

#### Figure 4 #####
sf1 <- survfit(Surv(Week, Status)~+1, data = queen_dead)
curve_surv_all <- read.xlsx("AgingCobs_popsurvey.xlsx", sheet = 3) # sf1
curve_surv_all$qx <- curve_surv_all$` n.deads`/curve_surv_all$Nalive
mean(curve_surv_all$qx) # 0.1454519

curve_surv_all$mortality_stad <-curve_surv_all$qx/0.1454519

col_dead$W <- as.factor(col_dead$W)

fertility<- aggregate(col_dead$Eggs, by =list(col_dead$Week), mean)
colnames(fertility) <- c("Week", "Mean_fert")
mean(fertility$Mean_fert) #[1] 8.90862 Average fertility
fertility$Stand_fert <- fertility$Mean_fert / 8.90862

queenspfert <- aggregate(col_dead$Pupae_Q, by =list(col_dead$Week), mean, na.action = na.omit)
colnames(queenspfert) <- c("Week", "Mean_fert")
mean(queenspfert$Mean_fert, na.rm = TRUE) #[1] 2.099742 Average fertility
queenspfert$Stand_fert <- queenspfert$Mean_fert / 2.099742


# Graph here
ggplot ()+
  geom_smooth(data= queenspfert, aes(x=Week, y=Stand_fert,  col = "darkgreen"), alpha = 0.2)+
  geom_smooth(data= fertility, aes(x=Week, y=Stand_fert, col = "blue"), alpha = 0.2)+
  geom_smooth(data = curve_surv_all, aes(x= Week, y = mortality_stad, col = "red"), alpha = 0.2)+
  geom_abline(slope = 0, intercept = 1, linetype = "dashed", col = "gray")+
  xlab("Weeks")+
  ylab("Standarized mortality and fertility")+
  geom_smooth( data = curve_surv_all, aes(x= Week, y = survival*5, col = "grey"), method = "auto", alpha = 0.2)+
  scale_y_continuous(sec.axis = sec_axis(~./5, name = "Survival"))+
  scale_color_manual(values = c("blue","darkgreen",  "grey","red"),
                     labels = c( "Eggs","Queen Pupae",  "Survival",  "Mortality"),
  guide = guide_legend(override.aes=aes(fill=NA)))+
  theme_minimal()+
  theme(legend.title = element_blank(),
        legend.key = element_rect(colour = "white"),
        legend.position = "bottom",
        legend.text=element_text(size=15),
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))

#### Figure S4 ####
colnames(EggsTotal) <- c("queen.id", "Eggs", "Treatment")
colnames(WorkerTotal) <- c("queen.id", "WorkerP", "Treatment2")
colnames(QueenPTotal) <- c("queen.id", "QueenP", "Treatment3")

workpEgg <- merge(EggsTotal, WorkerTotal, by = "queen.id")
cor.test(workpEgg$Eggs, workpEgg$WorkerP, method=c("pearson"))
pW <- ggplot(data=workpEgg, aes (x = Eggs, y = WorkerP, col=Treatment))+
  xlim(0,1500)+
  ylim(0,1500)+
  labs( y = "Worker Pupae")+
  geom_point()+
  geom_smooth(method = lm, alpha = 0.3)+
  geom_abline()+
  theme_minimal()+
  theme(legend.title = element_blank(),
        legend.key = element_rect(colour = "white"),
        legend.position = "bottom",
        legend.text=element_text(size=15),
        axis.title = element_text(size = 15),
        axis.text = element_text(size = 15))

workpEgg <- merge(workpEgg, QueenPTotal, by = "queen.id")
workpEgg$SumWQP <- workpEgg$WorkerP + workpEgg$QueenP
cor.test(workpEgg$Eggs, workpEgg$QueenP, method=c("pearson"))
cor.test(workpEgg$Eggs, workpEgg$SumWQP, method=c("pearson"))
pQ <- ggplot(data=workpEgg, aes (x = Eggs, y = QueenP, col=Treatment))+
  xlim(0,1500)+
  ylim(0,1500)+
  labs(y = "Queen Pupae")+
  geom_point()+
  geom_smooth(method = lm, alpha = 0.3)+ 
  geom_abline()+
  theme_minimal()+
  theme(legend.title = element_blank(),
          legend.key = element_rect(colour = "white"),
          legend.position = "bottom",
          legend.text=element_text(size=15),
          axis.title = element_text(size = 15),
          axis.text = element_text(size = 15))

pWQ <- ggplot(data=workpEgg, aes (x = Eggs, y = SumWQP, col=Treatment))+
  xlim(0,1500)+
  ylim(0,1500)+
  geom_point()+
  geom_smooth(method = lm, alpha = 0.3)+ 
  geom_abline()+
  labs(y = "Queen and Worker Pupae")+
  theme_minimal()+
    theme(legend.title = element_blank(),
          legend.key = element_rect(colour = "white"),
          legend.position = "bottom",
          legend.text=element_text(size=15),
          axis.title = element_text(size = 15),
          axis.text = element_text(size = 15))

mylegend<-g_legend(pW)
grid.arrange(  arrangeGrob(pQ+ theme(legend.position="none"), pW+ theme(legend.position="none"), pWQ+ theme(legend.position="none"), nrow = 1, ncol = 3), mylegend, nrow=2,heights=c(6, 1) )

