function [fusetimes] = find_fusetimes(par, SS_eq, Calcium, p_vals)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                                                 %%%
%%%               This code was written to be used in our publication               %%%
%%%             "Allosteric stabilization of calcium and lipid binding              %%%
%%%                    three synaptotagmins in fast exocytosis"                     %%% 
%%%                          Kobbersmed et al., Journal Year                        %%%
%%%                           DOI:  XX.XXXX/XXXX.XXXXXXXXXX                         %%%
%%%                                                                                 %%%
%%%             When reusing this code in a publication, in original or             %%%
%%%                       modified form, please cite our work.                      %%%
%%%                                                                                 %%%
%%%                                     Authors:                                    %%%
%%%                   Janus R. L. Kobbersmed (januslind@math.ku.dk)                 %%%
%%%                    Manon M.M. Berns (manon.berns@sund.ku.dk)                    %%%
%%%                    Alexander M. Walter (awalter@sund.ku.dk)                     %%%
%%%         Department of neuroscience, University of Copenhagen, Denmark           %%%
%%%                                    October 2021                                 %%%
%%%                                                                                 %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%This script determines fusion times given model parameters, steady state, and
%%%%p-values (p in (0,1) )

%INPUTS:
%par - model parameters vector
%SS_eq - steady state vector normalized to 1
%Calcium - calcium concentration
%p_vals - p values between 0 and 1 to be evaluated from the inverse of the
% cumulative release probability function.


%OUTPUTS:
%fusetimes - time points of fusion




%%%Author: Janus R. L. Kobbersmed, 
%%%Department of Mathematics, University of Copenhagen

p_vals_shaped = p_vals(:);
n_draw = length(p_vals_shaped);
n_draw_dim = size(p_vals);

fusetimes_in = zeros(size(p_vals_shaped));

QMatrix = call_QMatrix(par, Calcium);

prec_step = 0.00001;
t_max = 0.1;
lb = 0;
ub = t_max;
step_sz = 0.001;
step_fac = 1;

% if par(14)>5
%     step_sz = 0.001;
%     t_max = 0.001;
% end

cdf_test = 0;
t_vec = [];
cdf_vals = [];
while max(p_vals_shaped) > cdf_test

    t_add = (lb:step_sz:ub)';
    t_vec = [t_vec; t_add];
    cdf_vals_add = P_function(QMatrix, SS_eq, t_add);  
    cdf_vals = [cdf_vals; cdf_vals_add];
    if sum(imag(cdf_vals)~=0)
        warning('Complex numbers appeared when determining the delay')
        cdf_vals = abs(cdf_vals);
    end
    step_fac = step_fac+1;
    lb = ub+step_sz;
    ub = t_max*step_fac;
    cdf_test = cdf_vals(end);
end


new_step_fac = 10;

for m = 1:n_draw
    goon = 1;


    p_val = p_vals(m);
    
    if isnan(p_val)
        fusetimes_in(m) = NaN;
    else

        while goon == 1      

            [cdf_vals_un, un_ind] = unique(cdf_vals, 'stable');
            t_vec_un = t_vec(un_ind);

            nan_vals = isnan(cdf_vals_un);
            if sum(nan_vals(:))
                cdf_vals_un(nan_vals) = [];
                t_vec_un(nan_vals) = [];
            end
            
            if max(cdf_vals_un>1)
                end_ind = find(cdf_vals_un>1, 1, 'first');
            else
                end_ind = length(cdf_vals_un);
            end

            t_guess = interp1(cdf_vals_un(1:end_ind), t_vec_un(1:end_ind), p_val);
            t_test = floor(t_guess/prec_step)*prec_step;

            if (min(abs(t_vec-t_test))<prec_step) && (min(abs(t_vec-(t_test+prec_step)))<prec_step)
                fusetimes_in(m) = t_guess;
                goon = 0;
            else
                t_vec_test = t_vec - t_guess;
                low_ind = find(t_vec_test<0, 1, 'last');
                high_ind = find(t_vec_test>0, 1, 'first');
                stepsz = (t_vec(high_ind)-t_vec(low_ind))/new_step_fac;
                t_add = ((t_vec(low_ind)+stepsz):stepsz:(t_vec(high_ind)-stepsz))';
                p_add = P_function(QMatrix, SS_eq, t_add);

                if sum(imag(p_add)~=0)
                    warning('Complex numbers appeared when determining the delay')
                    cdf_vals = abs(cdf_vals);
                end

                t_vec = [t_vec(1:low_ind); t_add; t_vec(high_ind:end)];
                cdf_vals = [cdf_vals(1:low_ind); abs(p_add); cdf_vals(high_ind:end)];
                goon = 1;
            end
        end
    end
end

fusetimes = reshape(fusetimes_in, n_draw_dim);