function [peak_rates, rates_est_mean, rates_est_q2_5, rates_est_q97_5, Calciums, par] = simulate_peak_release_rate(n_rep, varRRP, varSyt, varSlot, smallRRP_IO, fitslot, excludeOutl, Lplus, lambda, allost_IO, param_no, param_fac, nsyts_mean, nslots_mean)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                                                 %%%
%%%               This code was written to be used in our publication               %%%
%%%             "Allosteric stabilization of calcium and lipid binding              %%%
%%%                    three synaptotagmins in fast exocytosis"                     %%% 
%%%                          Kobbersmed et al., Journal Year                        %%%
%%%                           DOI:  XX.XXXX/XXXX.XXXXXXXXXX                         %%%
%%%                                                                                 %%%
%%%             When reusing this code in a publication, in original or             %%%
%%%                       modified form, please cite our work.                      %%%
%%%                                                                                 %%%
%%%                                     Authors:                                    %%%
%%%                   Janus R. L. Kobbersmed (januslind@math.ku.dk)                 %%%
%%%                    Manon M.M. Berns (manon.berns@sund.ku.dk)                    %%%
%%%                    Alexander M. Walter (awalter@sund.ku.dk)                     %%%
%%%         Department of neuroscience, University of Copenhagen, Denmark           %%%
%%%                                    October 2021                                 %%%
%%%                                                                                 %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%This script uses the pre-calculated functions (see scripts in folder
%%%%"generic_functions") to determine peak release rates.

%INPUTS
%n_rep - number of repetitions of release latency simulation
%varRRP - whether the RRP size is variable (0 or 1) 
%varSyt - whether the number of syts per SV is variable (0 or 1) 
%varSlot - whether the number of slots per SV is variable (0 or 1) 
%smallRRP_IO - whether the RRP size is [if 0]: mean 4000 (SD 2000 if variable) 
% or [if 1]: mean 10 (SD 5 if variable)
%fitslot - the number of slots used in the fitting routine.
%excludeOutl - Whether or not the second release latency observation was
% fitted to
%Lplus - The Calcium independent fusion rate
%lambda - The lambda used in cost value calculation
%allost_IO - Whether the fitting routine was with or without allostericity
% (0 or 1)
%param_no - [if 0]: Parameters are best fit values. [if ~= 0]: Indicates
% the parameter to be varied (Figure 2 - figure supplement 3)
%param_fac - The factor on the parameter, if a parameter is varied (Figure
% 2 - figure supplement 3)
%nsyts_mean - mean number of syts per SV
%nslots_mean - mean number of slots per SV

%OUTPUTS
%peak_rates - all peak rates determined (all calcium concentrations, all
% repetitions)
%rates_est_mean - mean peak release rates for all calcium concentrations
%rates_est_q2_5 - 2.5 % quantile of peak release rates for all calcium 
% concentrations
%rates_est_q97_5 - 97.5 % quantile of peak release rates for all calcium 
% concentrations
%Calciums - Calcium concentrations used in simulation
%par - parameters used (loaded from best fit parameters file)

PIPup = 0
if PIPup
    warning('PIPup simulation')
end

if (~allost_IO && param_no)
    warning('Trying to simulate varying parameters without allostericity')
    return
end

if allost_IO && ~param_no
    if ~Lplus
        filename = ['./Outputs/Figure_results/Generic_functions/generic_functions_bestfit_slot' num2str(fitslot) '_fitchoice17_exclude' num2str(excludeOutl) '_lambda' num2str(lambda) '.mat'];
    else
        filename = ['./Outputs/Figure_results/Generic_functions/generic_functions_bestfit_slot' num2str(fitslot) '_fitchoice6_exclude' num2str(excludeOutl) '_Lplus' num2str(Lplus) '_lambda' num2str(lambda) '.mat'];
    end
end
    
if ~allost_IO
    filename = ['./Outputs/Figure_results/Generic_functions/generic_functions_bestfit_slot' num2str(fitslot) '_fitchoice6_exclude' num2str(excludeOutl) '_Lplus' num2str(Lplus) '_lambda' num2str(lambda) '_A1.mat'];
end   


if allost_IO && param_no
    filename = ['./Outputs/Figure_results/Generic_functions/varyparams/generic_functions_bestfit_slot' num2str(fitslot) '_fitchoice6_exclude' num2str(excludeOutl) '_Lplus' num2str(Lplus) '_lambda' num2str(lambda) '_varyparams' num2str(param_no) '_paramfact' num2str(param_fac) '.mat'];
end

if PIPup
    filename = ['./Outputs/Figure_results/Generic_functions/generic_functions_bestfit_slot' num2str(fitslot) '_fitchoice6_exclude' num2str(excludeOutl) '_Lplus' num2str(Lplus) '_lambda' num2str(lambda) '_3sytsPIPup.mat'];
    warning('Refitted PIP2 concentration. Make sure parameters are right')
end

load(filename)

if PIPup
    warning('PIPup simulation')
    par(8) = 3;
end

if allost_IO && param_no
    param_ind = param_inds(param_no);
    par_temp = par;
    par(param_ind) = par_temp(param_ind)*param_fac;
end

% Calciums = [0.01 0.1 0.5 1 1.5 2 5 10 20 30 40 50 60 70 80];

if smallRRP_IO
    par(11) = 10;
    par(23) = 5;
end

nves_mean = par(11);
nves_sd = par(23);


peak_rates = zeros(length(Calciums), n_rep);

num_warn = 0;

if smallRRP_IO
    Ca_inds = 28;
else
    Ca_inds = 1:length(Calciums);
end



for kCa = Ca_inds
    Calcium = Calciums(kCa);

    [QMatrix_test] = MakeQMatrix(par, Calcium);
    testsyt = par(8);
    testslot = par(14);
    dP_test = dP_cells{kCa,testsyt,testslot};
    QMatrix_load = QMatrix_cells{kCa, testsyt, testslot};
    Q_test = sum(QMatrix_test~=QMatrix_load);
    if sum(Q_test)
        warning('WARNING: Calcium choice does not match loaded file')
        return
    end
    
    
    dP_all = zeros(n_rep,length(dP_test));

    if nves_sd && varRRP
        k_gamma = (nves_mean^2)/(nves_sd^2);
        thet_gamma = (nves_sd^2)/nves_mean;

        nves_draws = round(gamrnd(k_gamma, thet_gamma, length(Calcium), n_rep));

        if ~smallRRP_IO
            low_limit = 5;
        else
            low_limit = 1;
        end
        
%         if ~smallRRP_IO
        goon = sum(nves_draws(:)<low_limit);
        while goon
            bad_inds = find(nves_draws(:)<low_limit);
            nves_draws(bad_inds) = round(gamrnd(k_gamma, thet_gamma, length(bad_inds), 1));

            goon = sum(nves_draws(:)<low_limit);
        end
%         end
    else
        nves_draws = nves_mean*ones(length(Calciums), n_rep);
    end

    nves_max = max(nves_draws(:));

    [syt_set, slot_set] = variability_choice(n_rep, nves_max, varSyt, varSlot, nsyts_mean, nslots_mean);
    
    
    for k = 1:n_rep
        nves_here = nves_draws(k);
        if nves_here<nves_max
            syt_set(k,(nves_here+1):end) = NaN;
            slot_set(k,(nves_here+1):end) = NaN;
        end
    end
        
    syts_uniq = unique(syt_set);
    slots_uniq = unique(slot_set);
    
    for nsyt_loop = max(min(syts_uniq),1):max(syts_uniq)
        for nslot_loop = max(min(slots_uniq),1):max(slots_uniq)
            
            dP_func = dP_cells{kCa, nsyt_loop, nslot_loop};
            
            [~, dPmax_ind] = max(dP_func);
            
            if dPmax_ind == length(dP_func)
                warning('Peak release rate occured at the last index')
                disp(['Calcium=' num2str(Calcium) ', nsyt=' num2str(nsyt_loop) ', nslots=' num2str(nslot_loop)])
                disp('Calculate more time points!')
                disp(['Vary param no ' num2str(param_no)])
                disp(['Param fact ' num2str(param_fac)])
                break
            end
            
            
            if ~isempty(dP_func)
                valid_inds = (syt_set==nsyt_loop).*(slot_set==nslot_loop);
                num_inds = sum(valid_inds,2);
                
                for krep = 1:n_rep
                    dP_all(krep, :) = dP_all(krep,:) + num_inds(krep)*dP_func';
                end
            else
                interp_inds_test = find((syt_set==nsyt_loop).*(slot_set==nslot_loop));
%                 disp(['Empty: nsyt=' num2str(nsyt_loop) ', nslot=' num2str(nslot_loop)])
                if isempty(interp_inds_test)
%                     disp('Alright')
                else
                    disp(['Empty: nsyt=' num2str(nsyt_loop) ', nslot=' num2str(nslot_loop)])
                    warning('NOT OK!!!')
                    disp('Calculate combination of syts and slots')
                    num_warn = num_warn +1;
                    disp(['Ca2+:  ' num2str(Calcium)])
                    disp(['Vary param no ' num2str(param_no)])
                    disp(['Param fact ' num2str(param_fac)])
                    return
                end
            end
            
        end
        if dPmax_ind == length(dP_func)
            break
        end
    end
    
    
    t_vec_rate = tvec_rate_cells{kCa, testsyt,fitslot};
    
    basal_rate = par(6)*exp(-par(6)*t_vec_rate);
    noSlot_inds = (slot_set==0);
    num_inds = sum(noSlot_inds,2);
    for krep = 1:n_rep
        dP_all(krep, :) = dP_all(krep,:) + num_inds(krep)*basal_rate;
    end
    
        
    peak_rates(kCa, :) = max(dP_all,[],2);
%     disp(['Number of warnings: ' num2str(num_warn)])
    
end

rates_est_mean = mean(peak_rates,2);
rates_est_q2_5 = quantile(peak_rates, 0.025, 2);
rates_est_q97_5 = quantile(peak_rates, 0.975, 2);