function [delay_likeli] = Likelihood(par, ExpDelayCa, ExpDelay, exclude_two)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                                                 %%%
%%%               This code was written to be used in our publication               %%%
%%%             "Allosteric stabilization of calcium and lipid binding              %%%
%%%                    three synaptotagmins in fast exocytosis"                     %%% 
%%%                          Kobbersmed et al., Journal Year                        %%%
%%%                           DOI:  XX.XXXX/XXXX.XXXXXXXXXX                         %%%
%%%                                                                                 %%%
%%%             When reusing this code in a publication, in original or             %%%
%%%                       modified form, please cite our work.                      %%%
%%%                                                                                 %%%
%%%                                     Authors:                                    %%%
%%%                   Janus R. L. Kobbersmed (januslind@math.ku.dk)                 %%%
%%%                    Manon M.M. Berns (manon.berns@sund.ku.dk)                    %%%
%%%                    Alexander M. Walter (awalter@sund.ku.dk)                     %%%
%%%         Department of neuroscience, University of Copenhagen, Denmark           %%%
%%%                                    October 2021                                 %%%
%%%                                                                                 %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%This script obtains the likelihood values in the PIP2-synaptotagmin 
%%%model based on a set of parameters and observations.

%Inputs:
%par - all model parameters
%ExpDelayca- Calcium concentrations for which release latency values are
%determined
%ExpDelay - Release latency values
%exclude_two - Whether or not to exclude the second data point

%Output
%dealy likeli - likelihood of observing the release latencies according to
%the model

fit_choice = par(15);


if sum(par <= 0)>0 || par(12) > min(ExpDelay) || par(12) <0.3e-3 || sum(par([1,3,4])>1e10)>0 %|| par(7)^par(14) > 1e18
    delay_likeli = 0; 
        %If conditions on parameters are not met, likelihood is -infty
        %Conditions: Parameters must be positive. If fit_constraint>0, we require par(7)=f<fit_constrant 
else
Ca_0 = par(10); %Background calcium concentration. Used for intial distribution.
% 


if fit_choice>=100
    SS_eq_proto = steady_state_LOU(par, 1, Ca_0);
    SS_eq_proto(end+1) = 0; %Fused state
else
    SS_eq_proto = steady_state_newmethod(par, 1, Ca_0);
    SS_eq_proto(end+1) = 0; %Fused state
end

[uTest, dP_vec] = deal(zeros(length(ExpDelay),1)); %Initialise vector for u-values.
        %The u-values are the value P(tau<t), tau is fusion time point.
        %I.e. u-values are the cumulative fusion probability.
        %Oder statistics: The pdf of the ordered fusion events are the
        %beta-distribution after transforming to uniform stochastic
        %variable (i.e. the  u-values).


if exclude_two == 1 %Excluding the second observation
    [uTest, dP_vec] = deal(zeros(length(ExpDelay)-1,1)); %Initialize a smaller vector
    nums_loop = 1:(length(ExpDelay)-1); %Fewer runs on the loop
elseif exclude_two == 2 %Excluding the first and second observations
    [uTest, dP_vec] = deal(zeros(length(ExpDelay)-2,1));    
    nums_loop = 1:(length(ExpDelay)-2);
else
    nums_loop = 1:length(ExpDelay);
end



parfor k = nums_loop %Looping according to number of considered observations. 
                     %Finding the likelihood at each observed delay.

    if exclude_two == 1 
        ind = k+1*(k>1) %Exclude second observation
    elseif exclude_two == 2
        ind = k+2 %Exclude the two first observations
    else
        ind = k; %Consider all observations
    end

    Calcium = ExpDelayCa(ind); %Experimental calcium


    if Calcium >= Ca_0
        SS_eq = SS_eq_proto;
    elseif Calcium < Ca_0

        if par(15) < 100
            SS_eq = steady_state_newmethod(par, 1, Calcium);
            SS_eq(end+1) = 0;
        else
            SS_eq = steady_state_LOU(par, 1, Calcium);
            SS_eq(end+1) = 0;
        end
    end



    del = par(12); %The delay which is independent of the model
    del_obs = ExpDelay(ind)-del; %Subtract the independent delay.

    QMatrix = call_QMatrix(par, Calcium);

    %         
%         [~,D,W] = eig(QMatrix*del_obs); %Find diagonalization of QMatrix*del_obs
%         
%         transVec = (SS_eq*(inv(W')*expm(D)*W')) % = SS_eq * expm(QMatrix*del_obs)
    uTest(k) = P_function(QMatrix, SS_eq, del_obs);
    dP_vec(k) = dP_function(QMatrix, SS_eq, del_obs);

%         uTest(k) = transVec(end); %Take only the fused state probability, i.e. P(tau<t)
end

nves_mean = par(11);
nves_sd = par(23);

if ~nves_sd
     %Get number of vesicles
    beta_likeli = betapdf(uTest, 5, nves_mean-4); %From order statistics, equivalent to: (n!/((5-1)!(nves-5)!)) * u^(5-1)*(1-u)^(nves-5)

    delay_likeli = beta_likeli.*abs(dP_vec);
else
    k_gamma = (nves_mean^2)/(nves_sd^2);
    thet_gamma = (nves_sd^2)/nves_mean;

    a_const = ((uTest.^4).*exp(-5*log(1-uTest)).*dP_vec)./(factorial(4)*(1-thet_gamma*log(1-uTest)).^k_gamma);
    tild_thet = thet_gamma./(1-thet_gamma*log(1-uTest));
    tild_thet = real(tild_thet);
    %k_gamma = single(k_gamma)

    I_func = @(n)((prod((k_gamma + (0:(n-1)))).*tild_thet.^n).*(1-gamcdf(5, k_gamma+n, tild_thet)));

    delay_likeli = a_const.*(I_func(5)-10*I_func(4) +35*I_func(3)-50*I_func(2)+24*I_func(1));
end

delay_likeli(delay_likeli< 1e-100) = 1e-100;

end



