#Load the packages required for data extraction and processing.
require(tidyverse)
require(readxl)

#load data from excel file using local directory. 
#file should contain a single nucleus FRAP experiment with FRAP signal and background signal. 
#Refer to Example
filepath = "/Users/michaelmauro/Desktop/OD1496_npp22_RNAi_11_27_2019_010004.xlsx"
data <- read_excel(filepath)
data <- as_tibble(data) %>% drop_na(Label_NE)

#Plot the raw data
ggplot(data) + 
  geom_line(mapping = aes(x = Time, y = Mean_NE, color = "Mean_NE")) + 
  geom_point(mapping = aes(x = Time, y = Mean_NE, color = "Mean_NE")) + 
  geom_point(mapping = aes(x = Time, y = Mean_BG, color = "Mean_BG")) + 
  geom_line(mapping = aes(x = Time, y = Mean_BG, color = "Mean_BG")) + 
  ggtitle("Raw Data") + 
  ylab("Intensity")

#Subtract Background signal from FRAP signal
#For each image, Bleach_corr = Bleach - Background, Reff_corr = Reference - Background
data <- data %>% 
  transmute(Time, Bleach_corr = Mean_NE - Mean_BG)

#Plot the corrected data 
ggplot(data) + 
  geom_line(mapping = aes(x = Time, y = Bleach_corr)) + 
  geom_point(mapping = aes(x = Time, y = Bleach_corr)) + 
  ggtitle("Background Corrected Data")

#Scale the data from 0 to 100%. 100% is defined as the last pre-bleach point. 
#0% is defined as the bleach point
#Normalized data set is refered to as data_norm
data_norm <- data %>% 
  mutate(Bleach_sub = Bleach_corr - Bleach_corr[4]) %>% 
  transmute(Time, It = Bleach_sub/mean(Bleach_sub[1:3]))

#Plot data_norm to view the raw data results and quality%
ggplot(data_norm) + 
  geom_line(mapping = aes(x = Time, y = It)) + 
  geom_point(mapping = aes(x = Time, y = It)) + 
  ggtitle("Background Corrected and Scaled Data") +
  ylab("Intensity")
  
#cut data at plateu if necessary
#data_norm <- data_norm %>% 
  #filter(Time <= 400)

#Replot the truncated data
ggplot(data_norm) + 
  geom_line(mapping = aes(x = Time, y = It)) + 
  geom_point(mapping = aes(x = Time, y = It)) + 
  ggtitle("Truncated Data") +
  ylab("Intensity")
  
#Calculate mobile fraction 

#Define Ii as the mean of the 3 initial points
Ii <- mean(data_norm$It[1:3])

#I0 <- I at bleach
#Define I0 as intensity at the bleach point (should be equal to 0)
I0 <- data_norm$It[4]

#Fit data from 0 to endTime to the equaton f(Time) = A*(1-exp(-tau*Time))
#Define mod1 as the equation above
mod1 <- function(A, tau, Time){A*(1-exp(-tau*Time))}

#Restrict data to be fit from time 0 to endTime
data_fit <- filter(data_norm, Time >= 0)

#Calculate tau based on the fit
taufit <- nls(It~mod1(Ie,tau ,data_fit$Time),data_fit, start=list(tau=0.1, Ie=0.1))

#Extract tau from the fit output
out <- summary(taufit)
tau <- out$coefficient[1,1]
Ie <- out$coefficient[2,1]
tau
Ie

#Calculate mobile fraction (Fm) based on defined intensity values
Fm = (Ie - I0)/(Ii - I0)

#Calculate immobile fraction (Im) based on Fm 
Im = 1 - Fm
#Calculate thalf based on the derived tau value
thalf = log(0.5)/(-1*tau)
thalf

#Create a model dataset based on mod1 and the derived tau value
fit1 <- data_fit %>% 
  transmute(Time, It = mod1(Ie,tau,data_fit$Time))

tH = mod1(Ie,tau ,thalf)

#plot the raw data and the model curve to visualize fit quality
#include thalf points in the plot 
ggplot() + 
  geom_point(data_norm, mapping = aes(x = Time, y = It)) +
  geom_segment(data_norm, mapping = aes(x = -30, y = Ii, xend = (fit1$Time %>% last()), yend = Ii), linetype = "dotted") +
  geom_line(fit1, mapping = aes(x = Time, y = It)) +
  geom_segment(fit1, mapping = aes(x = -30, y = tH, xend=thalf, yend=tH), linetype = "dotted") +
  geom_segment(fit1, mapping = aes(x = thalf, y = 0, xend = thalf, yend = tH), linetype = "dotted") + 
  geom_segment(fit1, mapping = aes(x = -30, y = Ie, xend = (fit1$Time %>% last()), yend = Ie), linetype = "dotted") +
  ggtitle("Fit Data") + 
  ylab("Intensity")
  





#Fit the post bleach data to model to calculate diffusion coefficient 
#If*(1-(w^2*(w^2+4*pi*D*Time)^-1)^(1/2))
#Equation from reference Ellenberg et al. 1997 Journal of Cell Bio
mod2 <- function(If,w,D,Time){If*(1-(w^2*(w^2+4*pi*D*Time)^-1)^(1/2))}

#Fit the data to a single parameter
#Be sure that w is the correct match for the collected data
Dfit <- nls(It~mod2(Ie,4,D,data_fit$Time),data_fit, start=list(D=0.01))
Dfit 

#Extract tau from the fit output
out <- summary(Dfit)
D <- out$coefficient[1,1]
D

#Create a model dataset based on mod1 and the derived D value
fit2 <- data_fit %>% 
  transmute(Time, It = mod2(Ie,4,D,data_fit$Time))

F#plot the raw data and the model curve to visualize fit quality
ggplot() + 
  geom_point(data_fit, mapping = aes(x = Time, y = It)) +
  geom_line(fit2, mapping = aes(x = Time, y = It)) + 
  ggtitle("Fit to Ellenberg et. al. equation")

