function [sweep_out]=PIV_Chromatin_CREEP_RELAX_v6(sweep_mode)

%% For CREEP RELXATION EXPERIMENTS (cycle constant force ON/OFF)
% Generates chromatin displacement data over time with reference to the
% first frame.
% From this, detects segments of applied stress and relaxation and fits a Kelvin-Voigt model to obtain elastic modulus and dyn. viscosity

% Analysis can be differntially performed for chromatin density

if exist('sweep_mode','var')==1; sweep_mode=1;
else, clc; clear all; sweep_mode=0; end

sweep_mode

%% INITIATION

all_figs = findobj(0, 'type', 'figure'); % check if figures open

close all;

p=genpath([cd '/MatPIV/MatPIV 1.7_no output']);
addpath(p);


%% OUTLINE

% 0) Parameters
%
% I. Load Data
% 
% I.a) Detect Nuclear Outlines
% I.b) Image Difference
% 
% II. RUN PIV using Multipass
% 
% III. ANALYSIS (COMMEN PART) 
% 
% III.a) Nuclear Area and Centroid
% III.b) STRAIN CALCULATION
% III.c) Display Strain MAPs for selected frame(s)
% III.d) Average Dislacement and Strain Tracks over Time
% III.e) Average Displacement Maps 
% III.f) Chromatin Density Segmentation
% III.g) Kelvin-Voigt Model Fitting
% 

% SUB_FUNCTIONS:
% 
% 1) histogram_segmentation
% 2) KV_Model_Segmentation



%% 0. Parameters


% Stack Properties:
resize_stack=1; % resize image stack (more data points for PIV), 1=off
res=0.123 * resize_stack; % resolution [um/pix]
fps=4; % frame per second [1/sec]; 0=auto detect from txt file
channel_num=1;

ref_time=20; % time over which reference is averaged [sec],  default: 20
seg_time=10; % time length of fitted segment [sec]          default: 10  


% Image Difference:
start_mode=1; % mode for detection of start: 1) from peak until below threshold; 2) from start until above threshold
    
frame_lim=100; % limit frames for segmentation (detection intersections only before this frame)
smooth_stack=0; % smooth image stack for segmentation
smooth_deriv=1; % smooth derivative track (gaussian, =number of data points used, 1=off)

start_correction=0; % correction start position of Stress Segment

bgr_substract=0; % perform background substraction for nuclear detection (turn off for deconvoluted stacks)
nuc_only=1; % uses only nucleus (background=0) for image difference

fit_image_diff=0; % fit exponential function to image difference curve (rudimentary analysis of relaxation times)



% ANALYSIS:

KV_analysis=0; % fit strain data to KV model

distance_analysis=0;
    pre_fit=1; % for small initial strains, fit Em and eta over short time window first
    dist_bin=2; % bin size for distance analysis (pix)
    regression_dist=10; % range for log-log regression of lamina properties (pix)
%     lamina_correction=0; % correct compliance modulus 
        iter=1; % number of iterations for correction (2 for correction check)

        
chromatin_analysis=1; % analyse displacement magnitudes of different chromatin types (intensity based)
    chrom_bins=7; % number of chromatin intensity segments
    seg_mode=1; % mode of segmentation, 1: equal bin size, 2; equal bin intensity
    marker_border_clearance=6; % clear border to avoid wrong results from lamins (default=6)
    
    load_marker_stack=0;
    chromatin_channel=1; % position of chromatin image in marker stack (frame); 0=use average from stack
%     allow_marker_overlapp=1; % 0= marker will be deleted from chromatin region
    
    nucleoli_detect_mode=1; % 0 OFF
                            % 1 Nucleolus from inverted chromatin
                            % 2 Nucleolus from inverted marker (CDK9)
                            % 3 Nucleolus from cytopainter
                            nucleoli_size=100; % nucleoli size [area in pix]
                            
    marker_detect_mode=0;   % e.g. CDK9 from marker stack                   
        cdk9_bin_num=3;
        
        
    nucleolus_displacement_analysis=0; % measure displ. of nucleolus and surrounding chromatin
        
        
% Kelvin-Voigt Model
power=800; % Laser Power [mA] to corresp. temperature profile

e_model=2; % type of strain for KV model; 1=total, 2=hydro, 3=shear, 4=dxdx, 5=dydy

flucs_position=1; % position of flucs line, 1:bottom, 2: right, 3: top, 4: left
    off_set= 2; % moves flucs line (away from border) (pix)

sigma= 1; % constant applied stress [Pa]
strain_baseline_correction=0; % correct for strain normal cell motion over time

        

% PIV
wind_sz= 32 * resize_stack; % size of interogation window
overlapp=0.75; % window overlapp [0.875]
glob_filt=3; % filter vectors that deviate from MEAN +/- f*STD (f=input, 0=off)
loc_filt=5;  % node-size of area that local filter takes into consideration

crop_vectors=1; % Use binary nuclear mask to delete vectors outside nucleus
    border_clearance=0; % number of pixels deleted from nuclear border (0=off)



% GRAPHS
show_frame=0; % show graphs for frame(s) (muplitple entries allowed, 0=first peak, -1=off)
show_avg_disp_maps=2; % generate average disp. and strain maps during each stress segment, 2=only 1. segment, 0=off
    avg_duration=5; % amount of time that is averaged (before peak) [sec], 0=total length
    clear_nuclear_periphery=1; % clear quiver arrows outside of nuclear bounds
    quiver_half_density=1; % also show quiver plot with only half the arrows (for vsibility)


% VIDEOS
save_vid=0; % save video files (quiver-overlay + segmented chromatin maps)
    quiver_half_video=0; % half density of quiver in video
    quiver_overlay=0; % overlay displacement arrows onto strain maps
    strain_axis=[-0.4,0.4];



%% Thermophoretic Correction


kb= 1.38064852 * 10^(-23); % boltzmann constant [N*m /K]
Tc=36; % chamber temperature [C]
St=0.03; % Soret Coefficient [1/K]
A= (10^-10)^2; % Area (Atomic scale)

Tph=kb*(273.15+Tc)*St / A % thermophoretic factor



%% I. LOAD DATA


in=input('Use Previous Data? (enter/n/l):','s');


if ~strcmp(in,'n')
    
    if strcmp(in,'l') % load previous data from .mat file
        
        
        % Load Tif Stack
        stored_path=evalin('base','path'); % get current data filepath (if stored before)

        [file_name,path]=uigetfile('*.mat','MultiSelect','off','Choose Files',stored_path); % load names and pathes
        assignin('base','path',path); % store current filepath for next execution in workspace
        assignin('base','filename',file_name); % store current filename for next execution in workspace
    
        stored_path=path;
        
        % Load Data from file
        MAT=load([path,file_name])
        
        % Assign data to workspace
        assignin('base','IM_Nuc',MAT.IM_Nuc);
        assignin('base','Nuc_A',MAT.Nuc_A);
        assignin('base','Nuc_Cent',MAT.Nuc_Cent);
        assignin('base','Nuc_MASK',MAT.Nuc_MASK);
        assignin('base','Nuc_VAL',MAT.Nuc_VAL);
        assignin('base','Radial_MASK',MAT.Radial_MASK);
        assignin('base','STACK',MAT.STACK);
          
              assignin('base','U',MAT.U);
              assignin('base','V',MAT.V);
              assignin('base','W',MAT.W);
              assignin('base','X',MAT.X);
              assignin('base','Y',MAT.Y);
            
        assignin('base','border',MAT.border);
        assignin('base','diff',MAT.diff); 

        assignin('base','start',MAT.start);
        assignin('base','stop',MAT.stop);
        
%         RES=MAT.ans % Previous Results
        if load_marker_stack==1
            assignin('base','marker_file',MAT.marker_file);
        end
        assignin('base','filename',MAT.filename);
        
        if isfield(MAT,'resize_stack')==1
            assignin('base','resize_stack',resize_stack); % save resize factor if applicable
        end
        
    else
        if ~ismember('W',evalin('base','who')) % check if previous data is present in workspace
            no_data_found % terminate and display error message
        end
    end
        
    
    % load values from workspace, if present
    W=evalin('base','W');
    X=evalin('base','X');
    Y=evalin('base','Y');
    U=evalin('base','U');
    V=evalin('base','V');
    
    if ismember('resize_stack',evalin('base','who'))
        resize_stack=evalin('base','resize_stack');
    else
        resize_stack=1; % set resize factor to 1 if not found
    end
    
    
    stored_path=evalin('base','path'); % get current data filepath (if stored before)
    file_name=evalin('base','filename'); % get current data filepath (if stored before)
    path=stored_path;
    
    % data stack properties
    info=imfinfo([stored_path file_name]);
    stack_sz=info(1).Width * resize_stack; % window size (pix)
    frame_num=numel(info)/channel_num; % number of stacks

%     STACK=zeros(stack_sz,stack_sz,frame_num); % Image Data Stack
%     for f=1:frame_num
%         frame=imread([stored_path file_name],'Index',f);
%         STACK(:,:,f)=frame;
%     end


    STACK=evalin('base','STACK');
    Nuc_VAL=evalin('base','Nuc_VAL');
    IM_Nuc=evalin('base','IM_Nuc');
    border=evalin('base','border');
    Nuc_A=evalin('base','Nuc_A');
    Nuc_Cent=evalin('base','Nuc_Cent');
    Nuc_MASK=evalin('base','Nuc_MASK');
    Radial_MASK=evalin('base','Radial_MASK');
    
    im_diff=evalin('base','diff');    
    start=evalin('base','start');
    stop=evalin('base','stop');
    
    
    
    % PLOT SEGMENTATION
    figure('name','Image Difference')
    plot(im_diff,'k','Markersize',5), hold on
    plot(start,im_diff(start),'go') % plot start points
    plot(stop,im_diff(stop),'ro') % plot start points
    xlabel('Time (frames)')
    ylabel('Image Difference')
    
    
else % execute new PIV calculation
    
    clear('base','marker_file') % delete marker file name if new sample is loaded
    
    % Load Tif Stack
    stored_path=evalin('base','path'); % get current data filepath (if stored before)
    
    if sweep_mode==0 % open directory to select file, if not opened through sweep
        [file_name,path]=uigetfile('*.tif','MultiSelect','off','Choose Files',stored_path); % load names and pathes
        assignin('base','path',path); % store current filepath for next execution in workspace
        assignin('base','filename',file_name); % store current filename for next execution in workspace
    else
        file_name=evalin('base','filename');
        path=stored_path;
    end
    % Data Stack Properties
    info=imfinfo([path file_name]);
    stack_sz=info(1).Width * resize_stack; % window size (pix)
    frame_num=numel(info); % number of stacks

    STACK=zeros(stack_sz,stack_sz,frame_num); % Image Data Stack
    for f=1:frame_num
        frame=imread([path file_name],'Index',f);

        if resize_stack>1 % resize frame (optional)
            frame=imresize(frame,resize_stack);
        end
        STACK(:,:,f)=frame;
    end
    
    if channel_num>1
        CH_STACK=STACK; % save total stack in seperate variable
        STACK=CH_STACK(:,:,1:channel_num:frame_num);
%         CH_2=CH_STACK(:,:,channel_num:2:frame_num);

        frame_num=frame_num/channel_num; % update number of frames to number in each individual stack
    end
    
    
    %% I.a) Detect Nuclear Outlines
    
    disp('detecting nuclear outlines')
    [border, IM_Nuc, Nuc_MASK, Nuc_VAL, Nuc_A, Nuc_Cent]=FIND_NUCLEAR_OUTLINE_v2(STACK, border_clearance, bgr_substract);
    

    % store data for next execution in workspace
    
    assignin('base','STACK',STACK); % store data STACK (MATRIX)
    assignin('base','border',border); % pixel coordinates of nuclear border (CELL), [f,2]
    assignin('base','Nuc_VAL',Nuc_VAL); % linear nuclear intensity values  (CELL)
    assignin('base','IM_Nuc',IM_Nuc); % nucleus only map (outside pix=0)
    assignin('base','Nuc_A',Nuc_A); % nuclear area for each frame (VECTOR)
    assignin('base','Nuc_Cent',Nuc_Cent); % nuclear centroid coordinates [pix]
    assignin('base','Nuc_MASK',Nuc_MASK); % binary nuclear map [0 or 1]
        
    if resize_stack>1
        assignin('base','resize_stack',resize_stack); % save resize factor if applicable
    end
    
    %% I.b) Image Difference
    % calculate image difference between frames to segment time-track
    % and detect the start of each creep-relaxation repeat
    
    disp('calc. image difference for segmentation')
 
    % average first 10 frames as reference (nucleus only or total stack)
    if nuc_only==1
        NUC_REF=mean(IM_Nuc(:,:,1:ref_time*fps),3);
    else
        NUC_REF=mean(STACK(:,:,1:ref_time*fps),3);
    end
        
    if smooth_stack==1 % smooth image stack (optional)
        NUC_REF=imgaussfilt(NUC_REF,5);
    end

    for f=1:frame_num
        
        if nuc_only==1
            NUC=IM_Nuc(:,:,f); % get image of nucleus only
        else
            NUC=STACK(:,:,f); % get complete image
        end
        
        if smooth_stack==1  % smooth image before substraction (optional)
            NUC=imgaussfilt(NUC,5);
        end
        im_diff(f)=sum(sum((NUC-NUC_REF).^2)); % Image Difference (rel. to first image)
    end
    im_diff(1)=im_diff(2); % avoid peak at start as a result of subtracting 0
    
    diff_smt=smoothdata(im_diff,'gaussian',smooth_deriv); % smoothed averge displacement track
    
    im_diff_d=zeros(size(im_diff));
    for i=3:length(im_diff)
        im_diff_d(i)=diff_smt(i)-diff_smt(i-1);
    end

    % Find Peaks

    % Find ON
    mph=max(im_diff_d)/3; % MinPeakProminence
    [~,on_id,w]=findpeaks(im_diff_d,'MinPeakHeight',mph,'MinPeakDistance',fps*seg_time*2,'MinPeakWidth',1); % index and width of peak
    
    % Find Off
    mph=-min(im_diff_d)/3; % MinPeakProminence
    [~,off_id,w]=findpeaks(-im_diff_d,'MinPeakHeight',mph,'MinPeakDistance',fps*seg_time*2,'MinPeakWidth',1); % index and width of peak

    
    % Find Start and Stop
    
    start = on_id;
    stop  = off_id;
    
    if start_mode==1 % find by going backwards on derivative from peak

        for i=1:length(on_id)
            k_on=on_id(i); % index variable
            thr=im_diff_d(k_on)/3;
            % find index where slope falls below threshold
            k_on=k_on-1;
            slope=im_diff_d(k_on);
            while slope > thr ; k_on=k_on-1; slope=im_diff_d(k_on);  end

            start(i)=k_on;

        end

        for i=1:length(off_id)
            k_off=off_id(i); % index variable
            thr=im_diff_d(k_off)/5;
            % find index where slope falls below threshold
            k_off=k_off-1;
            slope=im_diff_d(k_off);
            while slope < thr && im_diff_d(k_off-1)<0; k_off=k_off-1; slope=im_diff_d(k_off); end

            stop(i)=k_off-1;
        end

    end
    
    
    if start_mode==2 % find start by moving along derivate track from beginning until threshold is reached
        
%         im_thr=10*std(diff_smt(1:ref_time*fps)); % image difference threshold = 3*STD
        im_mean=mean(diff_smt(1:ref_time*fps)); % image difference threshold = 3*STD
        
        start=find(diff_smt(seg_time*fps:end)>2*im_mean,1)-1; % find first data point above threshold, use index before
        
        start=start+seg_time*fps;
        
        stop=start + seg_time*fps; % use segment time to define stop from start
    end
    
    
    % correct based on frame limit
    if frame_lim>0
        start(start>frame_lim)=[];
        stop(stop>frame_lim)=[];
    end
    
    % Manual Correction
    if start_correction~=0
        disp(['Manual Start Correction: ' num2str(start_correction)])
        start=start + start_correction;
        stop=stop + start_correction;
    end
    
    
    start
    stop
    
    
    % PLOT
    
    figure('name','Image Difference Derivative')
    plot(im_diff_d,'k'), hold on

    plot(on_id,im_diff_d(on_id),'go') % plot found ON peaks
    plot(off_id,im_diff_d(off_id),'ro') % plot found OFF peaks
    plot(start,im_diff_d(start),'gx') % plot start points
    plot(stop,im_diff_d(stop),'rx') % plot stop points
    
    
    figure('name','Image Difference')
    plot(im_diff,'k.','Markersize',5), hold on
    plot(diff_smt)
    plot(start,im_diff(start),'go') % plot start points
    plot(stop,im_diff(stop),'ro') % plot start points
    
    
    if fit_image_diff==1
        
        im_diff_rel=im_diff./max(im_diff);
        yy=im_diff_rel(start(1):start(1)+seg_time*fps*2)';
        base=min(yy);
        yy=yy-base;
%         yy=yy./max(yy);
        
        
        % STRESS PHASE
        yy_s=yy(1:seg_time*fps+1);
        xx=[0:length(yy_s)-1]'./fps;
        
        P0=[1,1]; % initial parameters: E, lambda
        P = fminsearch(@FIT_KVModel_Stress,P0,[],xx,yy_s,sigma);

        t_s_im = P(2);         % retardation time [s]
        yy_s_fit = sigma./P(1) .* (1-exp(-xx./P(2))); % calculate fitted y-values
        e_t1=yy_s_fit(end); % use last fitted value for start of relaxation

        % RELAX PHASE
        yy_r = yy(seg_time*fps+1:end); % image difference
        P0=[t_s_im,yy_s(end)]; % initial parameters: lambda

        % FIT MATERIAL PROPERTIES - KV RELAX
        P = fminsearch(@FIT_KVModel_Relax_RL,P0,[],xx,yy_r,e_t1);
    
        t_r_im= P(1);  % relaxation time [s]
        yy_r_fit= (e_t1 - P(2)) .* exp(-xx./P(1)) + P(2); % calculate fitted y-values


        % PLOT Fits
        figure('name','Image Diff - Fit')
        subplot(1,2,1)
        plot(xx,yy_s,'ko'), hold on
        plot(xx,yy_s_fit,'r')

        subplot(1,2,2)
        plot(xx,yy_r,'ko'), hold on
        plot(xx,yy_r_fit,'r')
    
        time=(1:frame_num)'/fps
        
        im_diff_rel'
        xx
        [yy_s+base,yy_s_fit+base]
       
        [yy_r+base,yy_r_fit+base]
        
        t_s_im
        t_r_im
        
        
        
        
    end
    
    
    % COMMAND Prompt - Continue with PIV using detetcted start value?
    if sweep_mode==0
        in=input('continue with PIV? (Enter/n): ','s')
    else
        in=''; % Enter to coninue
    end
    
    if strcmp(in,'n') % Else type 'n' to abort
        donotcontinue
    end
    
    
    % Alternative: enter number to manually override start value
    if ~strcmp(in,'')
        disp('Manual Start Correction:')

        in_num=str2num(in);
        start=in_num
        stop=start+seg_time*fps
    end
    
        
    %% I.c) Calculate Radial Distance Mask
    disp('calculating distance masks')
    
    % RADIAL MASK
    Radial_MASK=zeros(size(STACK)); % mask with distance to border for each location [pix]
    Radial_MASK(Radial_MASK==0)=NaN;
    for f=1:frame_num
        for i=1:stack_sz
            for j=1:stack_sz
                % if pixel is part of nucleus
                if Nuc_MASK(i,j,f)==1
                    bd=border{f}; % get border coordinates
                    r=sqrt((bd(:,1)-i).^2 + (bd(:,2)-j).^2); % calculate distance to each border pixel
                    Radial_MASK(i,j,f)=round(min(r)); % select shortest distance and save in matrix
                end
            end
        end
    end

    % store in workspace
    assignin('base','Radial_MASK',Radial_MASK); % binary nuclear map [0 or 1]  


    
    
    %% II. RUN PIV
    
    % show in command
    disp('Run PIV')
    if glob_filt>0
        disp(['Overlapp: ' num2str(overlapp)])
        disp(['Global Filter with ' num2str(glob_filt) '*STD'])
    end
    disp('________________________')
    
    REF=STACK(:,:,1); % reference image for PIV calculation
%     REF=mean(STACK(:,:,1:3),3); % reference image for PIV calculation
    seg=1; % segment index
    for f=2:frame_num % MatPIV loop for all frames

        % uodate reference to start frame of each creep-relax cycle
        if f==start(seg)
            disp(['updated reference to frame ' num2str(start(seg)) ' (Segment ' num2str(seg) ')'])

            % update image reference, average last 3 frames to enable PIV
            % including the reference image (identical images cause problems)
%             REF=mean(STACK(:,:,start(seg)-2:start(seg)),3); 
            REF=mean(STACK(:,:,start(seg)-1*fps:start(seg)),3); 
            
            if seg<length(start), seg=seg+1; end % next segment
            
        end
        
        FRAME=STACK(:,:,f);
        
        % RUN PIV using a the start of a segment as reference
        [x,y,u,v]=matpiv(REF,FRAME,wind_sz,1,overlapp,'multi');
        
        if glob_filt>0

            
            
%             [u,v]=globfilt(x,y,u,v,glob_filt,'interp'); % global filter, mean -/+ f*STD
        
            [u,v]=globfilt(x,y,u,v,glob_filt); % global filter, mean -/+ f*STD
            
            [u,v]=localfilt(x,y,u,v,2,'median',loc_filt);
            
            [u,v]=naninterp2(u,v);
            
            
            if isempty(u(isnan(u)))==0
                disp('Warning: NaNs detetcted during PIV!!!')
            end
        end

        w=magnitude(x,y,u,v); % Vector Magnitudes
        
        w=w.*res.*1000; % convert from pix to [nm]
        u=u.*res.*1000; % convert from pix to [nm]
        v=v.*res.*1000; % convert from pix to [nm]
        
        % Save in Cell and assign to Workspace

        X{f}=x; % [pix]
        Y{f}=y; % [pix]
        U{f}=u; % [nm]
        V{f}=v; % [nm]
        W{f}=w; % [nm]
        
        % display progress in CMD
        disp(['Calc. ' num2str(f) ' of ' num2str(frame_num)])
    end
    
    % include first frame (set to 0)
%     u=U{2}; u(~isnan(u))=0;
    
    X{1}=X{2};
    Y{1}=Y{2};
    W{1}=W{2};
    U{1}=U{2};
    V{1}=U{2};

    assignin('base','W',W); assignin('base','X',X); assignin('base','Y',Y); assignin('base','U',U); assignin('base','V',V);
    assignin('base','diff',im_diff); assignin('base','start',start); assignin('base','stop',stop);
    
end



%% ANALYSIS (COMMEN PART) 

%% III.a) Correct Start and Stop


   
% delete vectors outsise nuclear bounds
if clear_nuclear_periphery==1
    for f=1:frame_num
        x=X{f}; y=Y{f};
        u=U{f}; v=V{f};
        w=W{f};
        
        if crop_vectors==1
            for i=1:length(x)
                for j=1:length(y)
                    % generate crop mask for vector coordinates
                    DEL(i,j)=Nuc_MASK(x(1,i),y(j,1),f);
                end
            end
            u(isnan(u))=0; v(isnan(v))=0;
            u(DEL==0)=NaN; v(DEL==0)=NaN; % delete outside entries
            w(DEL==0)=NaN;
        end
        
        U{f}=u; V{f}=v;
        W{f}=w;
    end
end



%% III.b) Nuclear Area and Centroid

figure('name','Nuclear Area & Centroid')

% Check relative change in nuclear area

subplot(1,3,1)
plot((1:frame_num)/fps, Nuc_A ./Nuc_A(1).*100,'k')
xlabel('time [s]')
% ylabel('Area [um2]')
ylabel('Area [%]')

% Plot Centroid Movement

% draw 2D centroid change with time-colored line 
map=jet(20); % colormap
step=round(frame_num/20); % frames in each colored segment

subplot(1,3,2), hold on
for i=0:19
    id1=(i*step)+1; % first ID of segment
    id2=(i+1)*step+1; if i==19, id2=frame_num; end % second ID of segment, note: overlapping, correct last

    plot(Nuc_Cent(id1:id2,1).*res.*1000,Nuc_Cent(id1:id2,2).*res.*1000,'.-','Color',map(i+1,:))
end
xlabel('X-pos [nm]'), ylabel('Y-pos [nm]')

% relative distance change of centroid
cent_dist=zeros(frame_num,1); % initiliaze vector for centroid distance
for i=1:frame_num
    cent_dist(i)= (sqrt ( (Nuc_Cent(i,1)-Nuc_Cent(1,1))^2 + (Nuc_Cent(i,2)-Nuc_Cent(1,2))^2 )) *res*1000; % nm
end

subplot(1,3,3)
plot(cent_dist,'k.-')
xlabel('Frames'), ylabel('Distance from start [nm]')


% figure('name','Nuclear Area')
% 
% A_crp=Nuc_A(45:44+160)
% 
% plot((1:length(A_crp))/fps, A_crp ./A_crp(1).*100,'k')
% xlabel('time [s]')
% % ylabel('Area [um2]')
% ylabel('Area [%]')




%% III.c) STRESS PROFILE

    
% DISTANCE MASK - distance to FLUCS vector (1D)
Dist_MASK=zeros(stack_sz,stack_sz); % mask with distance to border for each location [pix]

Nuc_MASK_avg=ceil(mean(Nuc_MASK(:,:,1:10),3)); % average detected nuclear area over first frames

% find colum numbers of all nuclear pixels
[row,col]=find(Nuc_MASK_avg==1);

switch flucs_position
    case 1 % bottom
        flucs_start=max(row) + border_clearance + off_set;
    case 2 % right
        flucs_start=max(col) + border_clearance + off_set;
    case 3 % top
        flucs_start=min(row) - border_clearance - off_set;
    case 4 % left
        flucs_start=min(col) - border_clearance - off_set;
end

dist=flucs_start;
for i=1:flucs_start
    Dist_MASK(i,:)=dist;
    dist=dist-1; % increase distance
end



% Load Temperature Gradient to normalize for stress

Grad_MAT=load('Chamber_Temp-Gradient_Andor_750-800-900.mat');
GRAD=Grad_MAT.RES;

figure('name','Temperature Gradient'), 
plot(GRAD(:,1),GRAD(:,2:4))
xlabel('Dist (um)'), ylabel('T-Gradient [K]')
legend('750 mA','800 mA','900 mA')

switch power
    case 750
        t_grad=GRAD(:,2);
    case 800
        t_grad=GRAD(:,3);
    case 900
        t_grad=GRAD(:,4);
end


if resize_stack>1
    L=length(t_grad);
    t_grad=interp1(1:L,t_grad,1:1/resize_stack:L);
end


% nuc_start=flucs_start
dist_stress= zeros(size(Dist_MASK));
for i=1:stack_sz
    idx=Dist_MASK(i,1);
    if idx>1
        dist_stress(i,:)= -t_grad(idx);
    end
end

dist_stress=dist_stress.*Tph; % thermophoretic correction

figure('name',['Stress Profile [' num2str(power) ' mA]'])
pcolor(dist_stress), colormap hot, caxis auto, shading flat, axis square, colorbar, hold on
bound=border{1}; % nuclear border of first frame
plot(bound(:,2),bound(:,1),'g','linewidth',2)
line([0, stack_sz],[flucs_start,flucs_start],'color','white','linewidth',2)




%% III.c) STRAIN CALCULATION

disp('Strain Calculation')

% Characteristic Length
x=X{1}; 
L_char=x(1,3)-x(1,1); % characteristic lenght = node distance [pix]
L_char=L_char*res*1000; % convert to nm

J=zeros(size(STACK)); STRAIN=zeros(size(STACK));
for f=1:frame_num
    u=U{f}; % load deformations in x and y of current frame
    v=V{f};

    % initialize strain maps
    ehy=zeros(size(u));
    exy=zeros(size(u));
    str=zeros(size(u));
    edx=zeros(size(u));
    edy=zeros(size(u));

    for xi=1:length(u)
        for yi=1:length(v)

            % X-Component
            if yi==1
                % upper boundary
                dudx= (u(xi,yi+1) - u(xi,yi))*2  ;
                dvdx= (v(xi,yi+1) - v(xi,yi))*2 ;
            else
                % lower boundary
            if yi==length(v)
                dudx= (u(xi,yi) - u(xi,yi-1))*2;
                dvdx= (v(xi,yi) - v(xi,yi-1))*2;
            else
                % center nodes
                dudx=u(xi,yi+1) - u(xi,yi-1);
                dvdx=v(xi,yi+1) - v(xi,yi-1);
            end
            end

 
            % Y-Component
            if xi==1 
                % left boundary
                dudy= (u(xi+1,yi) - u(xi,yi))*2 ;
                dvdy= (v(xi+1,yi) - v(xi,yi))*2 ;
            else
            if xi==length(u)
                % right boundary
                dudy= (u(xi,yi) - u(xi-1,yi))*2;
                dvdy= (v(xi,yi) - v(xi-1,yi))*2;
            else
                % center nodes
                dudy=u(xi+1,yi) - u(xi-1,yi);
                dvdy=v(xi+1,yi) - v(xi-1,yi); 
            end
            end
            

            % HYDROSTATIC STRAINs
            ehy(xi,yi)=(dudx+dvdy)./(2*L_char); % hydrost. strain [rel.]    
            edx(xi,yi)= dudx ./L_char;          % X-direction only
            edy(xi,yi)= dvdy ./L_char;          % Y-direction only
            
            % SHEAR STRAIN
            exy(xi,yi)=(dudy+dvdx)./L_char; % strain [rel.]

            % TOTAL Strain (Tensor Magnitude)
%             str(xi,yi)=sqrt((dudx^2 + dvdy^2 + dudy^2 + dvdx^2)/L_char^2);
%             str(xi,yi)=(dudx + dvdy + dudy + dvdx)./L_char;
            
%             str(xi,yi)=(edx(xi,yi) - ehy(xi,yi));
            str(xi,yi)=(edy(xi,yi) - ehy(xi,yi));
        end
    end

    % clear all NaNs and save in MATRIX
    
    str(isnan(str))=0; 
    str=imresize(str,[stack_sz,stack_sz]);      
    if crop_vectors==1, str(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    STR(:,:,f)=str; % [rel.]

    ehy(isnan(ehy))=0;
    ehy=imresize(ehy,[stack_sz,stack_sz]);
    if crop_vectors==1, ehy(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    Ehy(:,:,f)=ehy; % [rel.]
    
    exy(isnan(exy))=0;
    exy=imresize(exy,[stack_sz,stack_sz]);
    if crop_vectors==1, exy(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    Exy(:,:,f)=exy; % [rel.]
    
    edx(isnan(edx))=0;
    edx=imresize(edx,[stack_sz,stack_sz]);
    if crop_vectors==1, edx(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    Edx(:,:,f)=edx; % [rel.]
    
    edy(isnan(edy))=0;
    edy=imresize(edy,[stack_sz,stack_sz]);
    if crop_vectors==1, edy(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    Edy(:,:,f)=edy; % [rel.]
    
    
    % Compliance Modulus
    switch e_model % retrive selected strain mode
        case 1
            strain=str;
        case 2
            strain=ehy;
        case 3
            strain=exy;
        case 4
            strain=edx;
        case 5
            strain=edy;
    end
    
    j = strain ./ dist_stress;
    if crop_vectors==1, j(Nuc_MASK(:,:,f)==0)=0; end % delete vectors outsise nuclear bounds
    j(isnan(j))=0; j(isinf(j))=0;

    J(:,:,f)=j; % save compliance modulus
    STRAIN(:,:,f)=strain; % save strain used for analysis
end



%% III.d) Average Dislacement and Strain Tracks over Time

time=(1:frame_num)'./fps; % time vector [sec]

% show average magnitude over time
for f=1:frame_num
    w=W{f}; % get from cell
    
    % average of absolute displacement [nm]
    w_avg(f)=mean(mean(abs(w(~isnan(w))))); 
    
    % average of absolute strains
    str=STR(:,:,f);
    e_avg(f,1)=mean(mean(abs(str(str~=0))));
    
    ehy=Ehy(:,:,f);
    e_avg(f,2)=mean(mean(abs(ehy(ehy~=0))));

    e_avg(f,6)=mean(mean((ehy(ehy~=0)))); % total hydrostatic
    
    exy=Exy(:,:,f);
    e_avg(f,3)=mean(mean(abs(exy(exy~=0))));
    
    edx=Edx(:,:,f);
    e_avg(f,4)=mean(mean(abs(edx(edx~=0))));
    
    edy=Edy(:,:,f);
    e_avg(f,5)=mean(mean(abs(edy(edy~=0))));
    
    % Compliance Modulus
    j=J(:,:,f);
    j_avg(f)=mean(mean(abs(j(j~=0))));

end


% correct for baseline chromatin movement using initial resting phase

if strain_baseline_correction==1
    
    % STRAIN CORRECTION
    str_avg=e_avg(:,e_model); % average all strain types during initial rest

    % linear fit to extract slope
    yy=str_avg(10:ref_time*fps); % average of start
    xx=(1:length(yy))' ./ fps;

    [e_fit,b]=lin_reg(xx,yy); % linear regression
    base_slope_e=b(2); % slope of baseline correction for strain

    % substract baseline from strain maps 
    str_cor=str_avg - time.* base_slope_e;
    
    
    % Visual Output
    figure('name','baseline correction')
    subplot(2,2,1)
    plot(xx,yy,'ko'), hold on
    plot(xx,e_fit,'r')
    ylabel('Analysis Strain (rel.)')
    title('Baseline Fit')
    
    subplot(2,2,2)
    plot(time, str_avg, 'k'), hold on
    plot(time, str_cor, 'r')
    title('Corrected Average Track')
    
    
    % COMPLIANCE CORRECTION

    % linear fit to extract slope
    yy=j_avg(10:ref_time*fps);

    [j_fit,b]=lin_reg(xx,yy'); % linear regression
    base_slope_j=b(2); % slope of baseline correction for compliance

    % substract baseline from strain maps 
    j_cor=j_avg - time'.* base_slope_j;
    
    % Visual Output
    subplot(2,2,3)
    plot(xx,yy,'ko'), hold on
    plot(xx,j_fit,'r')
    ylabel('Comliance (rel./Pa)'), xlabel('Time [sec]')
    
    subplot(2,2,4)
    plot(time, j_avg, 'k'), hold on
    plot(time, j_cor, 'r')
    xlabel('Time [sec]')
    
    base_slope_j
end



% % strain baseline correction
% if strain_baseline_correction==1
%     for m=1:5
%         e_avg(:,m)=e_avg(:,m) - time .*base_slope_e;
%     end
%     j_avg=j_cor;
% end


% PLOT

figure('name','MAG v. TIME')
plot(w_avg), hold on
xlabel('Time (frames)')
ylabel('Avg. Abs. Displacement (nm)')
% show starts and stops
for i=1:length(start)
    plot(start(i):start(i)+fps*seg_time,w_avg(start(i):start(i)+fps*seg_time),'g--')
end
for i=1:length(stop)
    plot(stop(i):stop(i)+fps*seg_time,w_avg(stop(i):stop(i)+fps*seg_time),'r--')
end

figure('name','STRAIN v. TIME')
plot(e_avg(:,1),'k'), hold on
plot(e_avg(:,2),'b')
plot(e_avg(:,3),'r')
plot(e_avg(:,4),'b:')
plot(e_avg(:,5),'b--')
for s=1:length(start)
    line([start(s) start(s)], [min(min(e_avg)), max(max(e_avg))],'Color','k','LineStyle','--')
end


legend('Dev','Hyd','Exy','dUdX','dVdY')
xlabel('Time (frames)')
ylabel('Strain (rel.)')

% yyaxis right
% plot(j_avg)

figure('name','Compliance v. TIME')
plot(e_avg(:,e_model),'k'), hold on
ylabel('Strain (rel.)')
yyaxis right
plot(j_avg)
plot(start,j_avg(start),'g.','Markersize',20)
plot(stop,j_avg(stop),'r.','Markersize',20)
ylabel('Compliance')
xlabel('Time (frames)')



% CROPPED STRAIN v TIME

crp_id=start(1)- round(5*fps) : stop(1) + round(15*fps); % index to crop a 30s time window (5s before and after stimulation)

e_avg_crp= e_avg(crp_id , :);
time_crp=(0:length(crp_id)-1)' ./ fps;

figure('name','STRAIN v. TIME - CROP')
plot(time_crp', e_avg_crp(:,1), 'k'), hold on
plot(time_crp, e_avg_crp(:,2), 'b')
plot(time_crp, e_avg_crp(:,3), 'r')
plot(time_crp, e_avg_crp(:,4), 'b:')
plot(time_crp, e_avg_crp(:,5), 'b--')
plot(time_crp, e_avg_crp(:,6), 'k--')
% for s=1:length(start)
%     line([start(s) start(s)], [min(min(e_avg)), max(max(e_avg))],'Color','k','LineStyle','--')
% end


legend('Dev','Hyd','Exy','dUdX','dVdY','tot. hyd')
xlabel('Time (s)')
ylabel('Strain (rel.)')
ylim([-0.005, 0.2])




%% III.c) Display Strain MAPs for selected frame(s)

if show_frame>-1
    
    if show_frame==0, show_frame=stop(1)-1; end % auto-mode, choose first peak
    
    for i=1:length(show_frame)
        f=show_frame(i);
        w=W{f}; x=X{f}; y=Y{f}; u=U{f}; v=V{f}; % get from cell
        
        
%         [shear,normal]=strain_calc(x,y,u,v,'richardson')
%         
%         
%         figure('name','MatPIV Strain')
%         subplot(1,2,1)
%         pcolor(shear), shading flat, axis square, caxis auto, hold on
%         
%         subplot(1,2,2)
%         pcolor(normal), shading flat, axis square, caxis auto, hold on
        
        
        
        str=STR(:,:,f); ehy=Ehy(:,:,f); exy=Exy(:,:,f); edx=Edx(:,:,f); edy=Edy(:,:,f);
        strain=STRAIN(:,:,f); strain(strain==0)=NaN; % get analysis strain
        j=J(:,:,f); j(j==0)=NaN; % compliance modulus
        bound=border{f};

        % DISPLACEMENT
        figure('name', ['Qiver - f' num2str(f)]) % quiver plot
        subplot(1,2,1)
        quiver(x,y,u./1000./res,v./1000./res,0), axis square, hold on
        plot(bound(:,2),bound(:,1),'r')
        title('Displacement Map (to scale)')
        xlim([0 stack_sz]), ylim([0 stack_sz])
        
        subplot(1,2,2)
        frame=STACK(:,:,f); frame(Nuc_MASK(:,:,f)==0)=NaN;
        pcolor(frame), shading flat, axis square, caxis auto, hold on
        set(gca,'Color','k')
        quiver(x,y,u,v,2,'color','w')
        title('Quiver Overlay')
        xlim([0 stack_sz]), ylim([0 stack_sz])
        
        
        % QUIVER at HALF DENSITY     
        if quiver_half_density==1
            x2=x; y2=y; u2=u; v2=v; % delete every other arrow

            x2(:,2:2:end)=[]; x2(2:2:end,:)=[];
            u2(:,2:2:end)=[]; u2(2:2:end,:)=[];

            y2(2:2:end,:)=[]; y2(:,2:2:end)=[];
            v2(2:2:end,:)=[]; v2(:,2:2:end)=[];

            figure('name', ['Qiver HALF - f' num2str(f)]) % quiver plot
            subplot(1,2,1)
            quiver(x2,y2,u2./1000./res,v2./1000./res,0,'color','k'), axis square, hold on
            plot(bound(:,2),bound(:,1),'r')
            title('Displacement Map (to scale)')
            xlim([0 stack_sz]), ylim([0 stack_sz])
        
            
            subplot(1,2,2)
            frame=STACK(:,:,f); frame(Nuc_MASK(:,:,f)==0)=NaN;
            pcolor(frame), shading flat, axis square, caxis auto, hold on
            set(gca,'Color','k')
            quiver(x2,y2,u2./1000./res,v2./1000./res,0,'color','w')
            title('Chromatin Overlay (to scale)')
            xlim([0 stack_sz]), ylim([0 stack_sz])
            
            
            figure('name', ['Comliance HALF - f' num2str(f)]) % quiver plot
            subplot(1,2,1)
            pcolor(j), shading flat, axis square, colormap jet, colorbar
%             caxis auto; 
            caxis([-400, 400])
            hold on
            quiver(x2,y2,u2./1000./res,v2./1000./res,0,'color','k')
            title('Compliance Overlay')
            
            subplot(1,2,2)
            pcolor(strain), shading flat, axis square, colormap jet, colorbar
%             caxis auto; 
            caxis([-1, 1.5])
            hold on
            quiver(x2,y2,u2./1000./res,v2./1000./res,0,'color','k')
            title('Analysis Strain Overlay')
            
        end
        

        % STRAIN MAPS
        figure('name',['Strain Maps + Mag - f' num2str(f)]) % show adjusted magnitude plot [nm]
        subplot(2,2,1)
        pcolor(x,y,w), shading flat, axis square, colorbar
        caxis([0 max(max(w))])
        title('displacement (nm)')

        subplot(2,2,2)
        pcolor(str), shading flat, axis square, colormap jet, caxis auto, colorbar
        title('total strain')
        subplot(2,2,3)
        pcolor(ehy), shading flat, axis square, colormap jet, caxis auto, colorbar
        title('hydrostatic')
        subplot(2,2,4)
        pcolor(exy), shading flat, axis square, colormap jet, caxis auto, colorbar
        title('shear')

        
        % analysis strain map 
        figure('name',['Strain Map: Analysis Strain - f' num2str(f)])
        pcolor(strain), shading flat, axis square, colormap jet, caxis auto, colorbar
        
        % show strain map overlay for current strain model
        figure('name',['Strain Map Overlay - f' num2str(f)])
        pcolor(strain), shading flat, axis square, colormap jet, caxis auto, colorbar
        hold on, quiver(x,y,u,v,2,'color','k')

        % figure titel
        switch e_model
            case 1
                title('Total Strain Map')
            case 2
                title('Hydro Map Overlay')
            case 3
                title('Shear Map')
            case 4
                title('dUdX Map')
            case 5
                title('dVdY Map')
        end

        
        % Compliance Map
        figure('name', ['Compliance Modulus Map - ' num2str(f)]) % quiver plot
        subplot(1,2,1)
        pcolor(j), shading flat, axis square, caxis auto, colorbar
        % plot(bound(:,2),bound(:,1),'r')
        title('Compliance Modulus')
        
        subplot(1,2,2) % show analysis strain in comparison
        pcolor(strain), shading flat, axis square, colormap jet, caxis auto, colorbar
        title('Analysis Strain')
    end
end    




%% III.e) Average Displacement Maps 
% generate average displacement maps during each stress segment (start to stop)

max_dist=max(max(Radial_MASK(:,:,start(1)))); % maximum distance from boder [pix]
dist_bin=dist_bin * resize_stack; % adjust size of distance bin if stack was resized
    regression_dist=regression_dist * resize_stack;

dist_num=floor(max_dist/dist_bin); % number of distance bins

if show_avg_disp_maps>0

    x=X{1}; y=Y{1}; % load deformations in x and y of first frame

    % determine number of segments to be shown
    if show_avg_disp_maps==2, seg_num_show=1; % show only seg. 1
    else, seg_num_show=length(start); end % show all segments
        
    for s=1:seg_num_show
        % load 
        u=[]; v=[]; % clear variables
        
        if avg_duration==0
            frames= start(s):stop(s); % use all frames during stimulation
        else
            frames= stop(s)- round(avg_duration*fps) :stop(s); % use certain time window (from peak)
        end
        
        for f=1:length(frames) % Load displacements in X,Y of current frames
            u(:,:,f)=U{frames(f)};
            v(:,:,f)=V{frames(f)}; 
        end
        str=STR(:,:,frames); % Load Strains in X,Y of current frames
        
        ehy=Ehy(:,:,frames); % Load Strains in X,Y of current frames
        exy=Exy(:,:,frames); % Load Strains in X,Y of current frames
        
        % average components over frames
        U_avg(:,:,s)=mean(u,3);
        V_avg(:,:,s)=mean(v,3);
%         TOT_avg(:,:,s)=mean(str,3); % total strain 
        Ehy_avg(:,:,s)=mean(ehy,3);
        Exy_avg(:,:,s)=mean(exy,3);

        
        % Half Quiver
        x2=x; y2=y; u2=U_avg; v2=V_avg; % delete every other arrow
        x2(:,2:2:end)=[]; x2(2:2:end,:)=[];
        u2(:,2:2:end)=[]; u2(2:2:end,:)=[];

        y2(2:2:end,:)=[]; y2(:,2:2:end)=[];
        v2(2:2:end,:)=[]; v2(:,2:2:end)=[]; 
        
        x=x2; y=y2; U_avg=u2; V_avg=v2; % delete every other arrow
                        
        % Quarter Quiver 
        
%         x2=x; y2=y; u2=U_avg; v2=V_avg; % delete every other arrow
%         x2(:,2:2:end)=[]; x2(2:2:end,:)=[];
%         u2(:,2:2:end)=[]; u2(2:2:end,:)=[];
% 
%         y2(2:2:end,:)=[]; y2(:,2:2:end)=[];
%         v2(2:2:end,:)=[]; v2(:,2:2:end)=[]; 
%         
%         x=x2; y=y2; U_avg=u2; V_avg=v2; % delete every other arrow
        
        
        
        % PLOT
        figure('name',['AVG Quiver - Segment #' num2str(s)])
        subplot(1,2,1)
         
        bound=border{start(1)}; % nuclear border at peak
        quiver(x,y,U_avg(:,:,s),V_avg(:,:,s),2,'color','k'), axis square, hold on
        plot(bound(:,2),bound(:,1),'r')
        ylim([0 stack_sz]), xlim([0 stack_sz])
        
        subplot(1,2,2)
        im_nuc=STACK(:,:,start(1)); im_nuc(Nuc_MASK(:,:,f)==0)=NaN;
        pcolor(im_nuc), shading flat, axis square, caxis auto, hold on
        quiver(x,y,U_avg(:,:,s),V_avg(:,:,s),2,'color','w')
        
        
        
        figure('name',['AVG Strain Maps - Segment #' num2str(s)])
        subplot(1,2,1)
        pcolor(Ehy_avg(:,:,s)), shading flat, axis square, colormap jet, hold on
        plot(bound(:,2),bound(:,1),'w')
        colorbar
        caxis([-0.5 0.5])
        title('Hydrostatic Strain')
        
        subplot(1,2,2)
        pcolor(Exy_avg(:,:,s)), shading flat, axis square, colormap jet, hold on
        plot(bound(:,2),bound(:,1),'w')
        colorbar
        caxis([-1 1])
        title('Shear')
        
        figure('name','AVG Shear Map')
        pcolor(Exy_avg(:,:,s)), shading flat, axis square, colormap jet, hold on
        plot(bound(:,2),bound(:,1),'w')
        colorbar
        caxis([-1 1])
        title('Shear')
        
        figure('name','AVG ABS Shear Map')
        abs_shear_map=abs(Exy_avg(:,:,s)); 
%         abs_shear_map(abs_shear_map==0)=NaN; 
        pcolor(abs_shear_map), shading flat, axis square, colormap jet, hold on
        plot(bound(:,2),bound(:,1),'w')
        colorbar
        caxis([0 0.8])
        title('Shear')
        
        
        figure('name',['Component Average - Segment #' num2str(s)])
        subplot(1,2,1)
        u_res=imresize(U_avg(:,:,s),[stack_sz,stack_sz]);
        pcolor(u_res), shading flat, axis square, colormap jet, caxis auto
        title('x-displacement')

        subplot(1,2,2)
        v_res=imresize(V_avg(:,:,s),[stack_sz,stack_sz]);
        pcolor(v_res), shading flat, axis square, colormap jet, caxis auto  
        title('y-displacement')
        
    
        % Component Displcement with Center Distance
        
        % VERTICAL
        u_dist=zeros(stack_sz,1); v_dist=zeros(stack_sz,1);
        for i=1:stack_sz
            % x-component
            u_i=u_res(:,i); % get current strip with same distance
            u_dist(i)=mean( (u_i(~isnan(u_i))) ); % average all displacement at same distance
            % y-component
            v_i=v_res(:,i); % get current strip with same distance
            v_dist(i)=mean( (v_i(~isnan(v_i))) ); % average all displacement at same distance
        end
        u_dist(isnan(u_dist))=0; % set NaNs to 0
        v_dist(isnan(v_dist))=0;
        
        w_dist=sqrt(u_dist.^2+v_dist.^2); % total displ magnitude
        
        figure('name',['Component Displ. v Dist - Segment #' num2str(s)])
        subplot(2,3,1), plot(1:length(u_dist),u_dist), ylabel('x-component') %, ylim([-100, 1000])
        subplot(2,3,2), plot(1:length(v_dist),v_dist), ylabel('y-component')
        subplot(2,3,3), plot(w_dist,'k'), ylabel('abs displ. magnitude')
        
        % HORIZONTAL
        u_dist=zeros(stack_sz,1); v_dist=zeros(stack_sz,1);
        for i=1:stack_sz
            % x-component
            u_i=u_res(i,:); % get current strip with same distance
            u_dist(i)=mean( (u_i(~isnan(u_i))) ); % average all displacement at same distance
            % y-component
            v_i=v_res(i,:); % get current strip with same distance
            v_dist(i)=mean( (v_i(~isnan(v_i))) ); % average all displacement at same distance
        end
        u_dist(isnan(u_dist))=0; % set NaNs to 0
        v_dist(isnan(v_dist))=0;
        
        w_dist=sqrt(u_dist.^2+v_dist.^2); % total displ magnitude
        
        subplot(2,3,4), plot(u_dist,1:length(u_dist)), xlabel('x-component')
        subplot(2,3,5), plot(v_dist,1:length(v_dist)), xlabel('y-component') % ,xlim([-50, 600]), ylim([0 stack_sz])
        subplot(2,3,6), plot(w_dist,1:length(w_dist),'k'), xlabel('abs displ. magnitude')  
        
        % displacement vs. Temp Gradient
        
%         figure('name','Displacemnt v. Temp Gradient')
%         plot(w_dist,'k')
%         yyaxis right
%         plot((mean(dist_stress,2)),'r')
        
    end
end



%% III.f) Chromatin Density Segmentation

seg_num=length(start); % number of stress-relax segments
Chro_MASK=zeros(stack_sz,stack_sz,seg_num); % chromatin mask with values = to bin chromatin intensity bin


if chromatin_analysis==1
        
    if load_marker_stack==1       
        
         % Load Tif Stack

        if ~ismember('marker_file',evalin('base','who')) && sweep_mode==0 && nucleoli_detect_mode>0 % open directory to select file, if no marker_file loaded
            [marker_file,path]=uigetfile('*.tif','MultiSelect','off','Choose Files',stored_path); % load names and pathes
%             assignin('base','path',path); % store current filepath for next execution in workspace
%             assignin('base','filename',file_name); % store current filename for next execution in workspace
            assignin('base','marker_file',marker_file); % store marker filename in workspace
        else
            marker_file=evalin('base','marker_file');
            path=stored_path;
        end
        
        % Data Stack Properties
        info=imfinfo([path marker_file]);
        marker_num=numel(info); % number of stacks

        figure('name','Chromatin Marker')
        MARKER=zeros(stack_sz,stack_sz,marker_num); % Image Data Stack
        for m=1:marker_num
            frame=imread([path marker_file],'Index',m);

            if resize_stack>1 % resize frame (optional)
                frame=imresize(frame,resize_stack);
            end
            MARKER(:,:,m)=frame;
            
            % show marker images
            subplot(1,marker_num,m)
            imshow(frame), caxis auto
        end     
        
    elseif load_marker_stack==0 % average over time, if no z-stack is loaded
        
            frames=start(1)-10:start(1); % average first 11 frames before start of stimulation
            im_nuc=mean(STACK(:,:,frames),3); % average nuclear images
        
            MARKER(:,:,chromatin_channel)=round(im_nuc);
%             marker_num=1;
            chromatin_channel=1; % override chromatin channel (if set to different numer)
    end
    
    
    % Nuclear Outline of Chromatin Channel
    [chro_border, im_chromatin, nuc_mask_seg, ~, Area_seg, ~]=FIND_NUCLEAR_OUTLINE_v2(MARKER(:,:,chromatin_channel), marker_border_clearance, bgr_substract);

    
        
    % Nucleoli detection (detect_mode 0=off)
    
    % select channel and find background intensity through histogram
    if nucleoli_detect_mode>0 % invert chromatin channel for nucleoli auto-detection
        
        if nucleoli_detect_mode==1 || nucleoli_detect_mode==2
            marker=MARKER(:,:,nucleoli_detect_mode);        
            marker(nuc_mask_seg==0)=0; % delete pixels outside nucleus
        
            marker= (marker-max(max(marker))) .*(-1); % invert image
            marker(nuc_mask_seg==0)=0; % delte pixels outside nucleus
            
            % crop high entries of border for marker image
            if nucleoli_detect_mode==2
                SE=strel('disk',5);
                b_mask=imerode(nuc_mask_seg,SE);
                marker(b_mask==0)=0;
            end
            
            
            figure('name','inverted image for nucleoli detection')
            imshow(marker), caxis auto
            
        else
            marker=MARKER(:,:,2); % nucleoli from cytopainter
            marker(nuc_mask_seg==0)=0; % delete pixels outside nucleus
        end
    
        % histogram-guided thresholding (find background peak)
        [m_count,m_cent]=hist(marker(marker~=0),200);
        m_count = smoothdata(m_count,'gaussian');
        mph=max(m_count)*0.9; % min peak height
        [~,idx,wd,]=findpeaks(m_count,'MinPeakHeight',mph,'MinPeakDistance',40);
    end
    
    % make binary mask
        
    if nucleoli_detect_mode==1 || nucleoli_detect_mode==2 % use inverted chromatin or marker channel for nucleoli detection
        
        % process image
        m_thr=m_cent(round(idx+wd/3)); % marker intensity threshold

        marker=marker-m_thr; marker(marker<0)=0; % background substraction

        marker=uint8(marker./max(max(marker)).*256);
        gt=graythresh(marker) % threshold
        bw=im2bw(marker,gt); % convert to binary

        bw=bwareaopen(bw,nucleoli_size,4); % delete small objects (connectivity=4)
        bw=imfill(bw,'holes');
        
        % smooth border
        SE=strel('disk',3);
        bw=imdilate(bw,SE);
        bw=imerode(bw,SE);

        nl_bound=bwboundaries(bw); % get boundaries
        
        nucleoli_mask=bw; % save binary mask for nucleoli
    end    
    
    
    if nucleoli_detect_mode==3 % Nucleolus from cytopainter
        
        m_thr=m_cent(round(idx+wd)); % marker intensity threshold    

        marker=marker-m_thr; marker(marker<0)=0; % background substraction

        marker=uint8(marker./max(max(marker)).*256);
        gt=graythresh(marker) % threshold
        bw=im2bw(marker,gt); % convert to binary

        bw=bwareaopen(bw,nucleoli_size,4); % delete small objects (connectivity=4)
        
        % smooth border
        SE=strel('disk',3); bw=imdilate(bw,SE); 
        SE=strel('disk',3); bw=imerode(bw,SE);

        nl_bound=bwboundaries(bw); % get boundaries
        
        nucleoli_mask=bw; % save binary mask for nucleoli                
    end     
    
    % delete nucleoli from chromatin channel
    if nucleoli_detect_mode>0, im_chromatin(nucleoli_mask==1)=0; end
        

    
    % PLOT
    
    if nucleoli_detect_mode>0
        
        % PLOT Histogram
        figure('name','Nucleoli BG histogram')
        plot(m_cent,m_count)
        line([m_thr,m_thr],[0,mph],'Color','Red','Linestyle','--')

        figure('name','Nuceloli Detection')
        subplot(1,3,1), imshow(marker), caxis auto, title('channel')
        subplot(1,3,2), imshow(nucleoli_mask), caxis auto, title('nucleoli mask')
        subplot(1,3,3), imshow(im_chromatin), caxis auto, title('chromatin channel')
    end
    
    
    if nucleoli_detect_mode==2
        marker_num=1; % additional marker present
    else
        marker_num=0; % no additional marker, only Nucleolus
    end
    
    
    % MARKER CHANNEL
    
    if marker_num>0
        
        marker=MARKER(:,:,2); % load marker image
        marker(nuc_mask_seg==0)=0;
        
        % delete nucleoli (if marker image was used)
        if nucleoli_detect_mode==2
            marker(nucleoli_mask==1)=0;
        end
        
        % histogram-guided thresholding (find background peak)
        [m_count,m_cent]=hist(marker(marker~=0),80);
        mph=max(m_count)*0.9; % min peak height
        [~,idx,wd,]=findpeaks(m_count,'MinPeakHeight',mph,'MinPeakDistance',50);

        
        if marker_detect_mode==1 % CDK9

            m_cum=m_count(1);
            for i=2:length(m_count)
                m_cum(i)=m_cum(i-1)+m_count(i); % cummulative histogram
            end
            m_cum=m_cum./max(m_cum); % normalize [0 1]

            % Segment histogram into bins of equal size (same pixel number)
            m_id = histogram_segmentation(m_cum, m_cent, cdk9_bin_num+1, 1) % get indecies of intensities threshold between bins


%             m_thr=m_cent(round(idx-w)); % marker intensity threshold (peak + width)
            m_thr=m_cent(m_id(cdk9_bin_num)); % marker intensity threshold (peak + width)

            marker=marker-m_thr; marker(marker<0)=0; % background substraction

            marker=uint8(marker./max(max(marker)).*256);
%                     gt=graythresh(marker) % threshold
            bw=imbinarize(marker,'adaptive','Sensitivity',0.3); % convert to binary
% ,
        end


        % smooth marker borders
        SE=strel('disk',3); bw=imdilate(bw,SE); 
        SE=strel('disk',3); bw=imerode(bw,SE); 

%                 bw=imfill(bw);
        m_bound=bwboundaries(bw); % get boundaries

        marker_mask=bw; % save binary mask
        im_marker=marker; % thresholded marker image (not binary)


        % PLOT

        figure('name','Marker Histogram')
        subplot(1,2,1), plot(m_cent,m_count)
        subplot(1,2,2), plot(m_cent,m_cum), hold on,
        subplot(1,2,2), plot(m_cent(m_id),m_cum(m_id),'ro')
        
        figure('name','Marker Segmentation')
        subplot(1,3,1)
        imshow(MARKER(:,:,2)), caxis auto, hold on
        for b=1:length(m_bound) % overlay object boundaries
            m_b=m_bound{b};
            plot(m_b(:,2),m_b(:,1),'r')
        end

        subplot(1,3,2)
        imshow(im_marker), caxis auto

        subplot(1,3,3)
        imshow(marker_mask), caxis auto
    end 
    
    
    
    % Chromatin Density Segmentation
    
    for s=1:seg_num
        
        im_nuc=im_chromatin; % else use loaded marker
        
        % Histogram
        [count,cent] = hist(im_nuc(im_nuc>0),150); % histogram of chromatin intensities using x bins
        c_sum=count(1);
        for i=2:length(count)
            c_sum(i)=c_sum(i-1)+count(i); % cummulative histogram
        end
        c_sum=c_sum./max(c_sum); % normalize [0 1]


        % Segment histogram into bins of equal size (same pixel number)
        seg_id = histogram_segmentation(c_sum, cent, chrom_bins, seg_mode); % get indecies of intensities threshold between bins

        % Generate Discrete Chromatin Map (in 10 steps)
        th_low=cent(1); % lower intensity threshold
        chrom_map=zeros(stack_sz,stack_sz); % init. segmented chromatin map
        for b=1:chrom_bins-1
            th_high=cent(seg_id(b)); % higher intensity threshold

            % generate segmented chromatin map of current frame
            chrom_map(im_nuc>=th_low & im_nuc<th_high)=b*10; 
            th_low=th_high; % update lower intensity threshold
        end
        % do last bin seperatly
        chrom_map(im_nuc>=th_low)=chrom_bins*10; % segmented chromatin map

%         % include other marker
%         if marker_num>0
%             chrom_map(marker_mask==1)=(chrom_bins+1)*10;
%         end
        
        

        if s==1 % visuals of first segment

            % segmented chromatin histogram
            
            figure('name',['chromatin histogram - seg ' num2str(s)])
            subplot(1,2,1), plot(cent,count), hold on
            for b=1:chrom_bins - 1 % draw segmentation lines
                line([cent(seg_id(b)),cent(seg_id(b))],[0 , count(seg_id(b))],'Color','Red','Linestyle','--')
            end

            subplot(1,2,2), plot(cent,c_sum,'r'), hold on
            plot(cent(seg_id),c_sum(seg_id),'ro')

            
            % segmented chromatin map
            
            figure('name',['segmented chromatin map - f' num2str(s)])
%             subplot(1,2,1), imshow(IM_Nuc(:,:,start(s)-1)-min(int)), caxis auto
            subplot(1,3,1), imshow(im_nuc), caxis auto
%             if marker_num>0
            bound=chro_border{1}; % load outline from first frame
            subplot(1,3,2), imshow(chrom_map,[0,0,0;jet((chrom_bins)*10); 1,1,1]), hold on
            plot(bound(:,2),bound(:,1),'r') % nuclear boundary
            
            if marker_num>0
                for b=1:length(m_bound) % overlay marker boundaries
                    m_b=m_bound{b};
                    plot(m_b(:,2),m_b(:,1),'w','linewidth',2)
                end                           

                subplot(1,3,3), imshow(marker_mask), caxis auto, hold on
                plot(bound(:,2),bound(:,1),'r')
            
            elseif nucleoli_detect_mode~=0
                
                subplot(1,3,3), imshow(MARKER(:,:,chromatin_channel)), caxis auto, hold on
                
                for b=1:length(nl_bound) % overlay marker boundaries
                    nl_b=nl_bound{b};
                    plot(nl_b(:,2),nl_b(:,1),'r','linewidth',1)
                end
            end
        end
        
        Chro_MASK(:,:,s)=chrom_map; % segmented chromatin map
    end
end


%% Nucleolus Displacement Analysis

if nucleolus_displacement_analysis==1 && nucleoli_detect_mode>0
    
    % fetch displacement and shear data during peak def. (~ 1 second)
    
    k=1; % index variable
    for f=stop-round(fps):stop
        displ(:,:,k)=W{f}; 
        shear(:,:,k)=abs(Exy(:,:,f));
        k=k+1;
    end
    
    % average and resize
    
    displ=mean(displ,3); % average displacement 
    w_res=imresize(displ,[stack_sz,stack_sz]); 
    
    shear=mean(shear,3); % average displacement 
    exy_res=imresize(shear,[stack_sz,stack_sz]);
    
    
    % PERINUCLEAR MASKs
    % perinucleolar chromatin shells with different distance to nucleus
    
    SE_pn=strel('disk',4); % thickness of surrounding area
    perinuc_mask_1=imdilate(nucleoli_mask,SE_pn); % expand area around nucleoli
    
    SE_pn=strel('disk',8); % thickness of surrounding area
    perinuc_mask_2=imdilate(nucleoli_mask,SE_pn); % expand area around nucleoli
    
    SE_pn=strel('disk',12); % thickness of surrounding area
    perinuc_mask_3=imdilate(nucleoli_mask,SE_pn); % expand area around nucleoli
    
    perinuc_mask_3(perinuc_mask_2==1)=0; % delete nesting perinnucleolar regions
    perinuc_mask_2(perinuc_mask_1==1)=0; % delete nesting perinnucleolar regions
    perinuc_mask_1(nucleoli_mask==1)=0; % delete nucleoli regions
    
    perinuc_mask_1(nuc_mask_seg==0)=0; % delete entries outsie nucleus
    perinuc_mask_2(nuc_mask_seg==0)=0; % delete entries outsie nucleus
    perinuc_mask_3(nuc_mask_seg==0)=0; % delete entries outsie nucleus
    
    pn_bound_1=bwboundaries(perinuc_mask_1); % outlines of perinuclear regions
    pn_bound_2=bwboundaries(perinuc_mask_2); % outlines of perinuclear regions
    pn_bound_3=bwboundaries(perinuc_mask_3); % outlines of perinuclear regions

    
    % Average Displacements for each PN shell
    PN_DISP(:,1)=mean(mean(w_res(nucleoli_mask==1)))  ; % #1: Nucleoli
    PN_DISP(:,2)=mean(mean(w_res(perinuc_mask_1==1))) ; % #2: PN-1 - 0-0.5 um
    PN_DISP(:,3)=mean(mean(w_res(perinuc_mask_2==1))) ; % #3: PN-2 - 0.5-1 um
    PN_DISP(:,4)=mean(mean(w_res(perinuc_mask_3==1))) ; % #4: PN-3 - 1-1.5 um
    w_res(nuc_mask_seg==0)=NaN; % w_res(isnan(w_res))=0;
    PN_DISP(:,5)=mean(mean(w_res(~isnan(w_res))))   ; % #5: Nucleus AVG
    
    PN_DISP
    
    % Average shear for each PN shell
    PN_EXY(:,1)=mean(mean(exy_res(nucleoli_mask==1)))  ; % #1: Nucleoli
    PN_EXY(:,2)=mean(mean(exy_res(perinuc_mask_1==1))) ; % #2: PN-1 - 0-0.5 um
    PN_EXY(:,3)=mean(mean(exy_res(perinuc_mask_2==1))) ; % #3: PN-2 - 0.5-1 um
    PN_EXY(:,4)=mean(mean(exy_res(perinuc_mask_3==1))) ; % #4: PN-3 - 1-1.5 um
%     exy_res(nuc_mask_seg==0)=NaN;    
    exy_res(isnan(exy_res))=0; % remove any NaNs
    PN_EXY(:,5)=mean(mean(exy_res(exy_res>0))); % #5: Nucleus AVG
    
    PN_EXY
    
    
    
    % PLOT
    figure('name','Perinucleolar Displacement')
%     w_im=round(w_res); w_im(isnan(w_im))=0; % w_im=uint8(w_im);

%     pcolor(w_res), colormap gray, caxis auto, axis square, shading flat, hold on
    pcolor(exy_res), colormap gray, caxis auto, axis square, shading flat, hold on %
           
    for b=1:length(pn_bound_3) % overlay perinucl boundaries
        pn_b=pn_bound_3{b};
        plot(pn_b(:,2),pn_b(:,1),'y','linewidth',1)
    end
    
        
    for b=1:length(pn_bound_2) % overlay perinucl boundaries
        pn_b=pn_bound_2{b};
        plot(pn_b(:,2),pn_b(:,1),'g','linewidth',1)
    end
    
    for b=1:length(pn_bound_1) % overlay perinucl boundaries
        pn_b=pn_bound_1{b};
        plot(pn_b(:,2),pn_b(:,1),'b','linewidth',1)
    end
        
    for b=1:length(nl_bound) % overlay nucleoli boundaries
        nl_b=nl_bound{b};
        plot(nl_b(:,2),nl_b(:,1),'r','linewidth',1)
    end
    
    bound=chro_border{1}; % load outline from first frame
%     subplot(1,3,2), imshow(chrom_map,[0,0,0;jet((chrom_bins)*10); 1,1,1]), hold on
    plot(bound(:,2),bound(:,1),'r') % nuclear boundary
    
end



%% Hydrostatic Strain Analysis [Moritz]


% Calculate Radial Mask from Chromatin Stack Image
Rad_MASK=zeros(stack_sz,stack_sz); % mask with distance to border for each location [pix]
Rad_MASK(Rad_MASK==0)=NaN; % set all values to NaN
bd=chro_border{1}; % get border coordinates

for i=1:stack_sz
    for j=1:stack_sz
        % if pixel is part of nucleus
        if nuc_mask_seg(i,j)==1
            
            r=sqrt((bd(:,1)-i).^2 + (bd(:,2)-j).^2); % calculate distance to each border pixel
            Rad_MASK(i,j)=round(min(r)); % select shortest distance and save in matrix
        end
    end
end


% Make binary radial masks using cut-off distance (rad_thr)
rad_thr=5; % cut off between outside mask and inside mask (pix)
% Outside mask
RM_out=zeros(size(Rad_MASK));
RM_out(Rad_MASK<=rad_thr)=1;
% Inside mask
RM_in=zeros(size(Rad_MASK));
RM_in(Rad_MASK>rad_thr)=1;


% Visual of Radial Masks
figure('name','Radial Mask')
subplot(1,3,1)
imshow(Rad_MASK), colormap jet, caxis auto, hold on
plot(bd(:,2),bd(:,1),'r'), title('Radial Mask')

subplot(1,3,2)
imshow(RM_out), colormap jet, caxis auto, title('Outer Mask')

subplot(1,3,3)
imshow(RM_in), colormap jet, caxis auto, title('Inner Mask')



% Marker Channel

for f= 1:frame_num
    
    j=J(:,:,f);
    
    strain_map=STRAIN(:,:,f); % get current map of analysis strain
    displ_map =W{f}; % get displacement magnitude from cell
    displ_map=imresize(displ_map,[stack_sz,stack_sz]);
    
%     strain_map(nuc_mask_seg==0)=0; % delete entries outside of the nucleus
%     strain_map_abs=abs(strain_map); % absolute strain map


    avg_str_tot(f)=mean(mean(strain_map(strain_map~=0)));
    avg_str_abs(f)=mean(mean(abs(strain_map(strain_map~=0))));


    % CHROMATIN DENSITY
    
    for c=1:chrom_bins
        CMap=zeros(size(stack_sz,stack_sz)); % initiate chromatin density mask
        CMap(chrom_map==c*10)=1;
    
        CD_tot(f,c) = mean(mean(strain_map(CMap==1)));     
        CD_abs(f,c) = mean(mean(abs(strain_map(CMap==1))));     
        
%         CD(f,c) = sum(sum( strain_map  .* CMap)) / ( sum(sum(CMap))  * sum(sum( strain_map)) ) * Area_seg;
    end
    
    % CHROMATIN MARKER
    if marker_num>0
        MA_tot(f,1) = mean(mean(strain_map(marker_mask==1)));     
        MA_abs(f,1) = mean(mean(abs(strain_map(marker_mask==1)))); 
    end
    
    
    % NUCLEOLI
    if nucleoli_detect_mode>0
        NC_tot(f,1) = mean(mean(strain_map(nucleoli_mask==1)));     
        NC_abs(f,1) = mean(mean(abs(strain_map(nucleoli_mask==1)))); 
        
        % displacement
        NC_disp(f,1)= mean(mean(abs(displ_map(nucleoli_mask==1)))); % average nucleoli displacement [nm]
        avg_disp(f,1)= mean(mean(abs(displ_map(~isnan(displ_map))))); % average nucleoli displacement [nm]        
    end
    
    
    % RADIAL DISTANCE
        
    
    % Correlation
%     CR(f,1) = sum(sum( strain_map  .* RM_out)) / ( sum(sum( strain_map)) * sum(sum(RM_out)) ) * Area_seg;
%     CR(f,2) = sum(sum( strain_map  .* RM_in )) / ( sum(sum( strain_map)) * sum(sum(RM_in))  ) * Area_seg;
%     CR(f,3) = sum(sum( strain_map  .* nuc_mask_seg )) / ( sum(sum( strain_map)) * sum(sum(nuc_mask_seg)) ) * Area_seg;

   
%     CR(f,1)=mean(mean(strain_map(Radial_MASK(:,:,f) <5 )));
%     CR(f,2)=mean(mean(strain_map(Radial_MASK(:,:,f)>=5 )));
%     CR(f,3)=mean(mean(strain_map(nuc_mask_seg==1)));
    
end


% PEAK STRAIN INTENSITIEs for different areas
for c=1:chrom_bins
    peak_strain_tot(c)=mean(CD_tot(stop-fps:stop,c));
    peak_strain_abs(c)=mean(CD_abs(stop-fps:stop,c));
end

if marker_num>0
    peak_strain_tot(end+1)=mean(MA_tot(stop-fps:stop));
    peak_strain_abs(end+1)=mean(MA_abs(stop-fps:stop));
end

if nucleoli_detect_mode>0
    peak_strain_tot(end+1)=mean(NC_tot(stop-fps:stop));
    peak_strain_abs(end+1)=mean(NC_abs(stop-fps:stop));
end



% crop time window
% crp_time_40s=start(1)-5*fps:stop(1)+20*fps;

% crop data
avg_str_tot    =avg_str_tot(crp_id);
avg_str_abs=avg_str_abs(crp_id);

% CC=CC(crp_time_40s,:);
% CR=CR(crp_time_40s,:);

CD_tot=CD_tot(crp_id,:);
CD_abs=CD_abs(crp_id,:);

if marker_num>0
    MA_tot=MA_tot(crp_id,:);
    MA_abs=MA_abs(crp_id,:);
end

if nucleoli_detect_mode>0
    NC_tot=NC_tot(crp_id,:);
    NC_abs=NC_abs(crp_id,:);

    NC_disp=NC_disp(crp_id,:);
    avg_disp=avg_disp(crp_id);
end


% Plotting

figure('name','Average Strain (All Nucleus)')
plot(time_crp,avg_str_abs,'g'), hold on
plot(time_crp,avg_str_tot,'k--')


% figure('name','Average Strain (MARKER)')
% plot(time_crp,MA_tot,'r'), hold on
% plot(time_crp,MA_abs,'r--')


% figure('name','Image Correlation - Distance')
% plot(time_cc,CR(:,3),'g'), hold on
% plot(time_cc,CR(:,1),'k--') % OUT
% plot(time_cc,CR(:,2),'k')   % IN



figure('name','Average Strain - Density'), hold on, colormap jet

chrom_cmp=jet((chrom_bins)*10);
                
for c=1:chrom_bins
    cmp=chrom_cmp(c*10,:);
    plot(time_crp,CD_tot(:,c),'Color',cmp)
end
if marker_num>0
    plot(time_crp,MA_tot,'k') % CDK9
end
if nucleoli_detect_mode>0
    plot(time_crp,NC_tot,'k--') % Nucleoli
end

figure('name','ABS Average Strain - Density'), hold on, colormap jet
for c=1:chrom_bins
    cmp=chrom_cmp(c*10,:);
    plot(time_crp,CD_abs(:,c),'Color',cmp)
end
if marker_num>0
    plot(time_crp,MA_abs,'k') % CDK9
end
if nucleoli_detect_mode>0
    plot(time_crp,NC_abs,'k--') % Nucleoli
end
% ylim([0 0.1])


figure('name','Peak Strains')
subplot(1,2,1)
plot(peak_strain_tot,'k.-','markersize',10), title('Total Analysis Strain')
subplot(1,2,2)
plot(peak_strain_abs,'r.-','markersize',10), title('Absolute Analysis Strain')



% figure('name','Average Displacement')
% plot(time_crp,NC_disp,'k'), hold on
% plot(time_crp,avg_disp,'g')



%% AREAs OF HIGH STRAIN

for r=1:2 % two rounds: first with total strain, than absolute strain
    
    % collect average strain maps (around peak) of analysis strain
    if r==1
        e_map=mean(STRAIN(:,:,stop(1)-fps:stop(1)),3); % get current map of analysis strain
    else
        e_map=mean(abs(STRAIN(:,:,stop(1)-fps:stop(1))),3); % absolute map of analysis strain
    end
    e_map(nuc_mask_seg==0)=0; % delete outside entries
    

    % STRAIN MAP Segmentation
    [count,cent] = hist(e_map(e_map~=0),300); % histogram of chromatin intensities using x bins

    e_sum=count(1);
    for i=2:length(count)
        e_sum(i)=e_sum(i-1)+count(i); % cummulative histogram
    end
    e_sum=e_sum./max(e_sum); % normalize [0 1]


    % Segment histogram into bins of equal size (same pixel number)
    seg_id = histogram_segmentation(e_sum, cent, 10, 1); % get indecies of intensities threshold between bins

    % Mask of highest and lowest strain
    e_Mask=zeros(size(e_map));
    e_Mask(e_map<cent(seg_id(1)))=1; % lowest 10% of strain
    e_Mask(e_map>cent(seg_id(9)))=2; % highest 10% of strain
    e_Mask(nuc_mask_seg==0)=0; % delete outside entries (important for absolut map)

    low =chrom_map(e_Mask==1);
    high=chrom_map(e_Mask==2);


    for c=1:chrom_bins
        area_low_strain(c,r)=length(low(low==c*10)) ./length(low);
        area_high_strain(c,r)=length(high(high==c*10)) ./length(high);
    end
    
    if nucleoli_detect_mode>1
        area_low_strain(chrom_bins+1,r)=length(low(low==0)) ./length(low);
        area_high_strain(chrom_bins+1,r)=length(high(high==0)) ./length(high);
    end
    

    figure('name','Strain Histogram Segmentation')
    subplot(1,2,1), plot(cent, count)
    subplot(1,2,2), plot(cent, e_sum), hold on, plot(cent(seg_id), e_sum(seg_id),'ro')


    figure('name','Segmentated Strain Map')
    subplot(1,2,1)
    pcolor(e_map), shading flat, axis square, colormap jet
    title('Strain Map')
    subplot(1,2,2)
    pcolor(e_Mask), shading flat, axis square % colormap([1,1,1; jet(2)])
    title('Strain Segmentation')
end

area_low_strain
area_high_strain
    

% Plot Final Results:

figure('name','Areas of High Strain')
subplot(1,2,1)
plot(area_low_strain(:,1),'b.-'), hold on
plot(area_high_strain(:,1),'r.-'), hold on
title('Total Strain')

subplot(1,2,2)
plot(area_low_strain(:,2),'b.-'), hold on
plot(area_high_strain(:,2),'r.-'), hold on
title('Absolute Strain')



%% III.g) Kelvin-Voigt Model Fitting



if KV_analysis==1
    
    % Distance Analysis
    
    if distance_analysis==1

    %     if lamina_correction==0
    %         iter=1; % override iteration count if no correction is performed
    %     end

        for it=1:iter % 

            % Collect STRAINs over Radial Distance (from border) for each frame

            e_dist=zeros(frame_num,dist_num); % matrix for averaged strains for each distance bin
            reg_lim_bin=round(regression_dist./dist_bin); % convert regression limit from pix to bin

            for f=1:frame_num

                % load strain maps
                if it==1
                j=J(:,:,f); % get current map of analysis strain
                end

                % bin strain maps along radial distance (from border)
                th_low=0;
                for d=1:dist_num-1
                    th_high=d*dist_bin;
                    e_dist(f,d)=mean(mean(abs(j(Radial_MASK(:,:,f)>=th_low & Radial_MASK(:,:,f)<th_high))));
                    th_low=th_high;
                end
                e_dist(f,dist_num)=mean(mean(abs(j(Radial_MASK(:,:,f)>=th_low))));        
            end





            % FIT MATERIAL PROPERTIES - KV RELAX
            figure('name','KV Distance - RELAX')

            relax_end=stop(1) + 10*fps ;
            t_1= (stop(1)-start(1)) / fps; % time at start of relaxation phase 
    %         dist_cmp=hsv(reg_lim_bin+2);
            dist_cmp=cool(reg_lim_bin);
            dist_cmp=flipud(dist_cmp);

            for d=1:dist_num

                % strain and time tracks
                e_r = e_dist(stop(1):relax_end,d)';  % strain [rel.]
                t_r = (0:length(e_r)-1) ./fps; % time vector [sec]


                % Strain Correction from background movement
                if strain_baseline_correction==1
                    e_r= e_r - (t_1+t_r) .*base_slope_j;
                end


                P0=[0.1,e_r(end),e_r(1)]; % initial parameters: t_relax, recovery loss

                P = fminsearch(@FIT_KVModel_Relax_RL_et1,P0,[],t_r,e_r);
                if P(2)<0 % repeat without P(2) if baseline off_set is negative
                    P = fminsearch(@FIT_KVModel_Relax,P0,[],t_r,e_r,e_r(1));
                    P(2)=0;
                    P(3)=e_r(1);
                end

                t_r_dist(d)= P(1); % relxation time [sec]
                np_r_dist(d)=  t_1/P(2); % recovery loss (nucleoplasm viscosity)


                % calculate fitted y-values
                e_r_fit= (P(3) - P(2)) .* exp(-t_r./P(1)) + P(2);

                % PLOT FITs
                if d<=reg_lim_bin
                    plot(t_r,e_r,'.','MarkerSize',10,'Color',dist_cmp(d,:)), hold on
                    plot(t_r,e_r_fit,'Color',dist_cmp(d,:))
                end
            end
    %         ylim([0 50])


            % Linear Regression of Log Data

            % Nucleoplasm Viscosity (Stress)
            x_lr = log((1:reg_lim_bin).*dist_bin);
            y_lr=log(mean(t_r_dist(:,1:reg_lim_bin),1));

            [tr_fit,tr_b]=lin_reg(x_lr', y_lr'); % linear regression
            tr_cc=corrcoef(x_lr, y_lr);

            % Nucleoplasm Viscosity (Relax)
            y_lr=log(mean(np_r_dist(:,1:reg_lim_bin),1));

            [eta_r_fit,eta_r_b]=lin_reg(x_lr', y_lr'); % linear regression
            eta_r_cc=corrcoef(x_lr, y_lr);


            % PLOT
            figure('name','Parameter over Distance')
            subplot(1,2,1)
            plot((1:dist_num)*dist_bin,mean(t_r_dist,1),'k.-')
            xlabel('Distance from Border [pix]')
            ylabel('Relaxation Time (sec)')

            subplot(1,2,2), title('Relax Viscosity')
            plot((1:dist_num)*dist_bin,mean(np_r_dist,1),'r.-')
            xlabel('Distance from Border [pix]')
            ylabel('Recovery Loss')


            figure('name','Parameter over Distance LOG')
            subplot(1,2,1) % E_m
            plot(log((1:reg_lim_bin)*dist_bin),log(mean(t_r_dist(:,1:reg_lim_bin),1)),'bo','markerface','blue'), hold on
            plot(log((1:reg_lim_bin)*dist_bin), tr_fit, 'k--')

            xlabel('LOG (Border Distance)')
            ylabel('LOG (Relaxation Time)')
            legend(['slope=' num2str(tr_b(2)) ', R^2=' num2str(abs(tr_cc(1,2))) ], 'FontSize', 14)

            subplot(1,2,2) % eta
            plot(log((1:reg_lim_bin)*dist_bin),log(mean(np_r_dist(:,1:reg_lim_bin),1)),'ro','markerface','red'), hold on
            plot(log((1:reg_lim_bin)*dist_bin), eta_r_fit, 'k--')

            xlabel('LOG (Border Distance)')
            ylabel('LOG (Relax Viscosity)')
            legend(['slope=' num2str(eta_r_b(2)) ', R^2=' num2str(abs(eta_r_cc(1,2))) ], 'FontSize', 14)




            % FIT MATERIAL PROPERTIES - KV STRESS
            figure('name','KV Distance - STRESS')
    %         dist_cmp=hsv(reg_lim_bin+2);
    %         dist_cmp=lines(reg_lim_bin)
    %         dist_cmp=hot((reg_lim_bin+3)*10);
    %         dist_cmp=flipud(dist_cmp);
    %         dist_cmp(1:30,:)=[];

            dist_cmp=cool(reg_lim_bin);
            dist_cmp=flipud(dist_cmp);

            for d=1:dist_num
                % strain and time tracks
                e_s = e_dist(start(1):stop(1),d)';  % strain [rel.]
                t_s = (0:length(e_s)-1) ./fps; % time vector [sec]

                e_s = e_s - e_s(1); % baseline correction (important for very small initial changes)

                % Strain Correction from background movement
                if strain_baseline_correction==1
                    e_s= e_s - t_s .*base_slope_j;
                end        


                % for small initial strains, it is better to fit Em and eta
                % first over a small time window (2 seconds)

                if pre_fit==1 && d<=2 % first two bins

                    % first fit Em and eta over the first 2 seconds
                    P0=[1,1,10]; % initial parameters: E, lambda
    %                 P0=[1,1]; % initial parameters: E, lambda
                    P = fminsearch(@FIT_KVModel_Stress_NP,P0,[],t_s(1:2*fps+1),e_s(1:2*fps+1),sigma);
                    P(3);
    %                 e_s_fit= sigma./P(1) .* (1-exp(-t_s./P(2)));


    %                 P0 = [10]; % initial parameters: E, lambda
                    P0=[P(3)];
                    P(3) = fminsearch(@FIT_KVModel_Stress_NP_only,P0,[],t_s,e_s,sigma,P(1),P(2));

                elseif pre_fit==1 && d<=4 % next two bins                            

                    P0=[1,1,10]; % initial parameters: E, lambda
                    P = fminsearch(@FIT_KVModel_Stress_NP,P0,[],t_s(1:4*fps+1),e_s(1:4*fps+1),sigma);
                    P(3);
    %                 e_s_fit= sigma./P(1) .* (1-exp(-t_s./P(2)));

                    P0=P(3);
                    P(3) = fminsearch(@FIT_KVModel_Stress_NP_only,P0,[],t_s,e_s,sigma,P(1),P(2));

                else % normal fittig for the rest

                    P0=[1,1,10]; % initial parameters: E, lambda
                    P = fminsearch(@FIT_KVModel_Stress_NP,P0,[],t_s,e_s,sigma);

                end

                e_s_fit= sigma./P(1) .* (1-exp(-t_s./P(2))) + t_s./P(3); % calculate fitted y-values


                Em_dist(d) = P(1);       % elastic modulus [Pa]
                eta_dist(d)= P(1)*P(2);  % dynamic viscosity [Pa * sec]
                t_s_dist(d)= P(2);       % retardation time [sec]
                np_s_dist(d)=P(3);       % additional dashpot (nucleoplasm viscosity)

    %             e_t1(d)=e_s_fit(end); % save last value as start value for relaxation

                % PLOT FITs

                if d<=reg_lim_bin
    %                 cmp_ind=[3,13,28,35,42,50];
                    plot(t_s,e_s,'.','MarkerSize',10,'color',dist_cmp(d,:)), hold on
                    plot(t_s,e_s_fit,'Color',dist_cmp(d,:))
                end
            end
            xlabel('Time [s]'), ylabel('Compliance (strain/stress)')
    %         ylim([0 50])


            % Linear Regression of Log Data

            % Elastic Modulus
            x_lr = log((1:reg_lim_bin)*dist_bin);
            y_lr=log(mean(Em_dist(:,1:reg_lim_bin),1));

            [Em_fit,Em_b]=lin_reg(x_lr', y_lr'); % linear regression
            Em_cc=corrcoef(x_lr, y_lr);

            % Viscosity
            y_lr=log(mean(eta_dist(:,1:reg_lim_bin),1));

            [eta_fit,eta_b]=lin_reg(x_lr', y_lr'); % linear regression
            eta_cc=corrcoef(x_lr, y_lr);


            % PLOT
            figure('name','Parameter over Distance')
            subplot(1,2,1)
            plot((1:dist_num)*dist_bin*res,mean(Em_dist,1),'k.-','Markersize',10)
    %         plot((1:dist_num)*dist_bin*res,mean(Em_dist,1),'k.','Markersize',5)
            xlabel('Distance from Border [um]')
            ylabel('Elastic modulus [Pa]')

            subplot(1,2,2)
            plot((1:dist_num)*dist_bin*res,mean(eta_dist,1),'k.-','Markersize',10)
            xlabel('Distance from Border [um]')
            ylabel('Dynamic viscosity [Pa*s]')


            figure('name','Parameter over Distance LOG')
            subplot(1,2,1) % E_m
            plot(x_lr,log(mean(Em_dist(:,1:reg_lim_bin),1)),'ko','markerface','black'), hold on
            plot(x_lr, Em_fit, 'k--')

            xlabel('LOG (Border Distance)')
            ylabel('LOG (E_m)')
            legend(['slope=' num2str(Em_b(2)) ', R^2=' num2str(abs(Em_cc(1,2))) ], 'FontSize', 14)

            subplot(1,2,2) % eta
            plot(x_lr,log(mean(eta_dist(:,1:reg_lim_bin),1)),'ko','markerface','black'), hold on
            plot(x_lr, eta_fit, 'k--')

            xlabel('LOG (Border Distance)')
            ylabel('LOG (eta)')
            legend(['slope=' num2str(eta_b(2)) ', R^2=' num2str(abs(eta_cc(1,2))) ], 'FontSize', 14)



            % COMMAN OUTPUT
            Em_l =  Em_dist(1)
            eta_l=  eta_dist(1)

            np_s_l= np_s_dist(1)
            np_r_l= np_r_dist(1)

            t_s_l= t_s_dist(1)
            t_r_l=t_r_dist(1)

            % values taken from y-intersection point of fit
            Em_l_intersect = exp(Em_b(1)) 
            eta_l_intersect = exp(eta_b(1))




            % Distance Correction (not finished)

    %         if lamina_correction==1
    %             
    %             % Lamina Properties (from Fit)
    %     %         if it==1 % keep initial lamina material properties
    %                 Em_l =  Em_dist(2)
    %                 eta_l=  t_s_dist(2)*Em_l
    % 
    %                 eta_l_r=t_r_dist(2)*Em_l
    %     %         end
    %             alpha=Em_b(2) % update exponent
    % 
    %             Jl=zeros(size(J)); % Compliance for Lamina
    %             for s=1:seg_num % segment
    % 
    %                 for f=start(s):stop(s)-1 % STRESS PHASE Correction
    %                     t= (f-start(s)+1)/fps; % time %%% CHECK +1
    %                     j=zeros(stack_sz,stack_sz); % temp. transfer matrix
    %                     for d=1:max_dist
    %                         j(Radial_MASK(:,:,f)==d) = (1./Em_l + t./eta_l) .* (1./d^alpha); % Maxwell Model
    % 
    %     %                     j(Radial_MASK(:,:,f)==d) = ( 1-exp(-t./ t_s_dist(2) .* d^alpha) ) ./ (Em_l .* d^alpha); % KV Model
    % 
    %                         Jl(:,:,f)=j;
    %                     end
    %                 end
    % 
    %                 if s==seg_num; relax_end= frame_num ; else, relax_end=start(s+1)-1; end
    % 
    %                 for f=stop(s):relax_end % RELAX PHASE Correction
    %                     t= (stop(s)-start(s))/fps; % time at stop (t1)
    %                     j=zeros(stack_sz,stack_sz); % temp. transfer matrix
    %                     for d=1:max_dist
    %                         j(Radial_MASK(:,:,f)==d) = t./(eta_l .* d^alpha); % Maxwell Model
    %                         Jl(:,:,f)=j;
    %                     end
    %                 end
    %             end
    %         
    % 
    %             for f=1:frame_num
    % 
    %                 jl=1./Jl(:,:,f); % lamina compliance
    %                 jt=1./abs(J(:,:,f)); % total compliance
    %                 jc= 1./ (jt-jl);
    %         %         jc(isinf(jc))=0;
    %                 Jc(:,:,f)=jc; % chromatin compliance
    %             end
    % 
    % 
    %             % STRESS PHASE
    %             frame=stop(1)-1;
    %             figure('name',['Lamina Correction - f' num2str(frame)])
    %             subplot(1,3,1)
    %             j=J(:,:,frame); j(j==0)=NaN;
    %             pcolor(abs(j)), shading flat, axis square, caxis auto
    %             title('Total Compliance')
    % 
    %             subplot(1,3,2)
    %             jl=Jl(:,:,frame); jl(jl==0)=NaN;
    %             pcolor(jl), shading flat, axis square, caxis auto
    %             title('Lamina Compliance')
    % 
    %             subplot(1,3,3)
    %             pcolor(Jc(:,:,frame)), shading flat, axis square, caxis ([0,0.04]), hold on
    %             title('Chromatin Compliance')
    % 
    %             % RELAX PHASE
    %             frame=stop(1);
    %             figure('name',['Lamina Correction - f' num2str(frame)])
    %             subplot(1,3,1)
    %             j=J(:,:,frame); j(j==0)=NaN;
    %             pcolor(abs(j)), shading flat, axis square, caxis auto
    %             title('Total Compliance')
    % 
    %             subplot(1,3,2)
    %             jl=Jl(:,:,frame); jl(jl==0)=NaN;
    %             pcolor(jl), shading flat, axis square, caxis auto
    %             title('Lamina Compliance')
    % 
    %             subplot(1,3,3)
    %             pcolor(Jc(:,:,frame)), shading flat, axis square, caxis ([0,0.04]), hold on
    %             title('Chromatin Compliance')
    % 
    %             if it==1
    %                 J=Jc; % save chromatin compliance as overall compliance
    %                 
    %                 % generate new average track (not finished)
    %                 for f=1:frame_num
    %                     j=J(:,:,f);
    %                     j_avg(f)=mean(mean(abs(j(~isnan(j)))));
    %                 end
    %                 
    %                 if strain_baseline_correction==1
    %                     j_avg = j_avg - time.* base_slope_j;
    %                 end
    %                 
    %             end
    %         end

        end
    end



    
    % fit ALL chromatin
    
    for s=1:seg_num % fit each segment

        % STRESS PHASE

    %     e_KV=e_avg(:,e_model)'; % choose strain type for analysis
        e_s = j_avg(start(s):stop(s));  % strain [rel.]
        t_s = (0:length(e_s)-1) ./fps; % time vector [sec]

        % Strain Correction from background movement
        if strain_baseline_correction==1
            e_s= e_s - t_s .*base_slope_j;
        end


        P0=[0.1,1,2]; % initial parameters: E, lambda

        % FIT MATERIAL PROPERTIES - KV STRESS
        P = fminsearch(@FIT_KVModel_Stress_NP,P0,[],t_s,e_s,sigma);

        Em_all(s)=P(1);         % elastic modulus [Pa]
        eta_all(s)=P(1)*P(2);   % dynamic viscosity [Pa * s]
        t_s_all(s)=P(2);      % retardation time [s]
        np_s_all(s)=P(3);      % retardation time [s]

        % calculate fitted y-values
        e_s_fit= sigma./P(1) .* (1-exp(-t_s./P(2))) + t_s./P(3);
        e_s_fit_kv= sigma./P(1) .* (1-exp(-t_s./P(2)));

        e_t1=e_s_fit(end); % use fitted value for start of relaxation segment


        % RELAX PHASE

        % determine last frame of relaxation phase
        if s==seg_num; relax_end=stop(s) + 10*fps ; else, relax_end=start(s+1); end

        e_r = j_avg(stop(s):relax_end); % strain segment [rel.]
        t_r = (0:length(e_r)-1) ./fps; % time segment [sec]

        % Strain Correction from background movement
        if strain_baseline_correction==1
            e_r= e_r - (t_s(end)+t_r) .*base_slope_j;
        end


        P0=[t_s_all(s),e_r(end)]; % initial parameters: lambda
    %     P0=[1/t_s_all(s)]; % initial parameters: lambda

        % FIT MATERIAL PROPERTIES - KV RELAX
        P = fminsearch(@FIT_KVModel_Relax_RL,P0,[],t_r,e_r,e_t1);
        if P(2)<0 % repeat without P(2) if baseline off_set is negative
            P = fminsearch(@FIT_KVModel_Relax,P0,[],t_r,e_r,e_t1);
            P(2)=0;
        end

        t_r_all(s)= P(1);  % relaxation time [s]
        np_r_all(s)= t_s(end) / P(2); % viscosity loss recovery loss

        % calculate fitted y-values
        e_r_fit= (e_t1 - P(2)) .* exp(-t_r./P(1)) + P(2);


        % PLOT Fits
        figure('name',['Fitted Segment - ' num2str(s)])
        subplot(1,2,1)
        plot(t_s,e_s,'ko'), hold on
        plot(t_s,e_s_fit,'r')
        plot(t_s,e_s_fit_kv,'r--')
    %     ylim([0 55])

        subplot(1,2,2)
        plot(t_r,e_r,'ko'), hold on
        plot(t_r,e_r_fit,'r')
    %     ylim([0 55])
    end

    % OUTPUT

    Em_all
    eta_all
    np_s_all
    np_r_all

    t_s_all
    t_r_all



    % PLOT Trend over Repeats
    if seg_num>1
        figure('name','Parameter over Time')
        subplot(1,3,1)
        plot(Em_all,'b.-')
        xlabel('Repeat')
        ylabel('Elastic modulus (Pa)')

        subplot(1,3,2)
        plot(eta_all,'r.-')
        xlabel('Repeat')
        ylabel('Dynamic viscosity (Pa*sec)')

        subplot(1,3,3)
        plot(t_s_all,'k.-')
        xlabel('Repeat')
        ylabel('Retardation time (sec)')
    end



    % fit SEGMENTED chromatin
    if chromatin_analysis==1

        
        if nucleoli_detect_mode>0, nucleoli_num=1; else, nucleoli_num=0; end
        
        e_chrom=zeros(frame_num,chrom_bins+marker_num+nucleoli_num); % init. matrix for average strains

        % Collect Strain for Chromatin Bins
        for f=1:frame_num
            j=J(:,:,f); % get current compliance map
            for c=1:chrom_bins + marker_num % mean all strain in chromatin bin of current frame
                e_chrom(f,c)=mean(mean(abs(j(Chro_MASK(:,:,1)==c*10))));
            end
            % collect strains for nucleoli
            if nucleoli_num==1
                e_chrom(f,c+1)=mean(mean(abs(j(nucleoli_mask(:,:,1)==1))));
            end
        end


        % display RESULTS in command
        e_chrom;
        e_chrom_mean=mean(e_chrom,1)'; % mean over all frames
        e_chrom_std=std(e_chrom,1)';   % STD over all frames

        e_s_fit=[]; e_r_fit=[];

        compart_num=chrom_bins + marker_num + nucleoli_num; % total number of compartments for displaying final data
        
        for s=1:seg_num
            for c=1:compart_num

                % STRESS PHASE
                e_s = e_chrom(start(s):stop(s),c)';  % strain [rel.]
                t_s = (0:length(e_s)-1) ./fps; % time vector [sec]


                % strain baseline correction
                if strain_baseline_correction==1
                    e_s= e_s - t_s .*base_slope_j;  
                end         


                P0=[Em_all(s),t_s_all(s),np_s_all(s)]; % initial parameters: E, lambda

                % FIT MATERIAL PROPERTIES - KV STRESS
                P = fminsearch(@FIT_KVModel_Stress_NP,P0,[],t_s,e_s,sigma);

                Em_chr(c,s) = P(1);         % elastic modulus [Pa]
                eta_chr(c,s)= P(1)*P(2);    % dynamic viscosity [Pa * s]
                t_s_chr(c,s)= P(2);         % retardation time [s]
                np_s_chr(c,s)= P(3);        % addition dash pot (nucleoplasm viscosity) [Pa * s]

                % calculate fitted y-values
                e_s_fit(:,c)= sigma./P(1) .* (1-exp(-t_s./P(2))) + t_s./P(3);
                e_s_plot(:,c)= e_s;
                e_t1(c)=e_s_fit(end); % use last fitted value for start of relaxation


                % RELAX PHASE

                % determine last frame of relaxation phase
                if s==seg_num; relax_end=stop(s) + seg_time*fps ; else, relax_end=start(s+1); end

                e_r = e_chrom(stop(s):relax_end,c)'; % strain segment [rel.]
                t_r = (0:length(e_r)-1) ./fps; % time segment [sec]


                % Strain Correction from background movement
                if strain_baseline_correction==1
                    e_r= e_r - (t_s(end)+t_r) .*base_slope_j;
                end


                P0=[t_s_all(s),e_r(end)]; % initial parameters: lambda

                % FIT MATERIAL PROPERTIES - KV RELAX
                P = fminsearch(@FIT_KVModel_Relax_RL,P0,[],t_r,e_r,e_t1(c));
                if P(2)<0 % repeat without P(2) if baseline off_set is negative
                    disp('negative offset - repeat')
                    P = fminsearch(@FIT_KVModel_Relax,P0,[],t_r,e_r,e_t1(c));
                    P(2)=0;
                end

                t_r_chr(c,s)= P(1); % relxation time [s]
                np_r_chr(c,s)= t_s(end) / P(2); % viscosity from recovery loss (nucleoplasm)

                % calculate fitted y-values
                e_r_fit(:,c)= (e_t1(c) - P(2)) .* exp(-t_r./P(1)) + P(2);
                e_r_plot(:,c)= e_r;
            end


            % PLOT (first repeat only)
            if s==1
                chrom_cmp=jet((compart_num)*10);

                figure('name','KV Fit - Chrom Density')
                subplot(1,2,1), hold on
                for c=1:compart_num
                    cmp=chrom_cmp(c*10,:);
                    plot(t_s,e_s_plot(:,c),'.','MarkerSize',15,'Color',cmp)
                    plot(t_s,e_s_fit(:,c),'color',cmp)
                end

                subplot(1,2,2), hold on
                for c=1:compart_num
                    cmp=chrom_cmp(c*10,:);
                    plot(t_r,e_r_plot(:,c),'.','MarkerSize',15,'Color',cmp)
                    plot(t_r,e_r_fit(:,c),'color',cmp)
                end

                figure('name','Parameter over Density')
                subplot(1,4,1)
                plot(1:compart_num,Em_chr(:,s),'b.-')
                xlabel('chrom. density')
                ylabel('Elastic modulus (Pa)')

                subplot(1,4,2), title('Dynamics Viscosity')
                plot(1:compart_num,eta_chr(:,s),'r.-')
                xlabel('chrom. density')
                ylabel('Dynamic viscosity (Pa*s)')

                subplot(1,4,3), title('Retardation Time')
                plot(1:compart_num,t_s_chr(:,s),'k.-')
                xlabel('chrom. density')
                ylabel('Retardation time (s)')

                subplot(1,4,4), title('Relaxation Time')
                plot(1:compart_num,t_r_chr(:,s),'k.--')
                xlabel('chrom. density')
                ylabel('Relaxation time (s)')
            end
        end

        % Command OUTPUT

        Em_chr
        eta_chr
        np_s_chr
        np_r_chr

        t_s_chr
        t_r_chr


        % Visual Summary

%         e_all_crp=j_avg(crp_id)';
%         e_marker_crp=e_chrom(crp_id,chrom_bins + marker_num);
% 
%         time_crp=(0:length(crp_id)-1)./fps;
% 
%         figure('name','Marker Compare')
%         plot(time_crp,e_all_crp,'k'), hold on
%         plot(time_crp,e_marker_crp)
% 
%         OUT=[e_all_crp, e_marker_crp]

    end
end

%% Command





%% VIDEOS
% save videos of the whole stack of quiver-plots

if save_vid==1    
    cd(path)

    % QUIVER OVERLAY
    vid_QO = VideoWriter([file_name(1:end-4) '-QO.avi'],'Uncompressed AVI');
    open(vid_QO);

    figure('name','Video-Quiver Overlay')
    for f = 1:frame_num
        
        w=W{f}; x=X{f}; y=Y{f}; u=U{f}; v=V{f}; % get from cell
        
        
        if quiver_half_video==1
            x(:,2:2:end)=[]; x(2:2:end,:)=[];
            u(:,2:2:end)=[]; u(2:2:end,:)=[];

            y(2:2:end,:)=[]; y(:,2:2:end)=[];
            v(2:2:end,:)=[]; v(:,2:2:end)=[];
        end
        
       
        frame=STACK(:,:,f);
        frame(Nuc_MASK(:,:,f)==0)=NaN;
        
        pcolor(frame), shading flat, axis square, caxis auto, hold on
        set(gca,'Color','k')
        quiver(x,y,u./1000./res,v./1000./res,0,'color','w') % to scale
        xlim([0 stack_sz]), ylim([0 stack_sz]), hold off
       
        frame = getframe(gcf);
        writeVideo(vid_QO,frame);
    end
    
    close(vid_QO);
    
    
    % SEGMENTED CHROMATIN VIDEO
%     vid_SC = VideoWriter([file_name(1:end-4) '-SC.avi'],'Uncompressed AVI');
%     open(vid_SC);
%     
%     figure('name','Video-Quiver Overlay')
%     for f = 1:frame_num
%         
%         imshow(Chro_MASK(:,:,f),[0,0,0;jet((chrom_bins+marker_num)*10)])
%        
%         frame = getframe(gcf);
%         writeVideo(vid_SC,frame);
%     end
%     
%     close(vid_SC);
    

    
    % STRAIN MAP VIDEO
    vid_SM = VideoWriter([file_name(1:end-4) '-SM.avi'],'Uncompressed AVI');
    open(vid_SM);
    
    figure('name','Video-Strain Map')
    
    % generate strain plot with same resolution as image
    switch e_model % retrive selected strain mode
        case 1
            strain=STR;
        case 2
            strain=Ehy;
        case 3
            strain=Exy;
        case 4
            strain=Edx;
        case 5
            strain=Edy;
    end
    
    for f = 1:frame_num
        
        str=strain(:,:,f); % get current strain map
        str(str==0)=NaN;
        
        pcolor(str), shading flat, axis square, colormap jet, caxis(strain_axis), colorbar
        
        if quiver_overlay==1
            
            x=X{f}; y=Y{f}; u=U{f}; v=V{f}; % get from cell
            
            if quiver_half_video==1
                x(:,2:2:end)=[]; x(2:2:end,:)=[];
                u(:,2:2:end)=[]; u(2:2:end,:)=[];

                y(2:2:end,:)=[]; y(:,2:2:end)=[];
                v(2:2:end,:)=[]; v(:,2:2:end)=[];
            end

            hold on

            quiver(x,y,u./1000./res,v./1000./res,0,'color','k') % to scale
            xlim([0 stack_sz]), ylim([0 stack_sz])

            hold off
        end
       
        frame = getframe(gcf);
        writeVideo(vid_SM,frame);
    end
    
    close(vid_SM);
    
    disp('VIDEOS DONE')
end


%% SWEEP OUTPUT

% all strain data
sweep_out.time=time_crp;
sweep_out.compl=j_avg;
sweep_out.strain=e_avg;

% cropped time window
sweep_out.time_crp=time_crp;
% sweep_out.compl_crp=j_avg_crp;
sweep_out.strain_crp=e_avg_crp;


% KV analysis:
if KV_analysis==1

    if distance_analysis==1
        sweep_out.alpha=[Em_b(2),eta_b(2),Em_cc(1,2),eta_cc(1,2)]; % lamina fit exponents (alpha) and correl. coeff.
        sweep_out.lamina=[Em_l,eta_l,np_s_l,np_r_l,t_s_l,t_r_l]; % lamina properties
    end

    sweep_out.chrom_all=[Em_all,eta_all,np_s_all,np_r_all,t_s_all,t_r_all]; % all chromatin

    if chromatin_analysis==1
        sweep_out.chrom=[Em_chr,eta_chr,np_s_chr,np_r_chr,t_s_chr,t_r_chr]; % chromatin compartments
    end
end


if marker_num>0
    sweep_out.chrom_tot=[CD_tot,MA_tot,NC_tot];
    sweep_out.chrom_abs=[CD_abs,MA_abs,NC_abs];
else
    sweep_out.chrom_tot=[CD_tot,NC_tot];
    sweep_out.chrom_abs=[CD_abs,NC_abs];
end

sweep_out.peak_strain_tot=peak_strain_tot;
sweep_out.peak_strain_abs=peak_strain_abs;

sweep_out.area_low_strain = area_low_strain;
sweep_out.area_high_strain= area_high_strain;



end % END OF MAIN FUNTCION




function seg_id = histogram_segmentation(cum,int,chrom_bins,mode)

% uses histogram to create equal sized binnes of intensities values 
% chrom_bins = number of total bins

seg_id=zeros(chrom_bins,1); % init. array for segmentation indecies

if mode==1
    for b=1:chrom_bins
        seg_id(b)=knnsearch(cum',b/chrom_bins); % search for nearest bin edge intensity
    end
end

if mode==2
    
    int_rel=int-min(int); int_rel=int_rel./max(int_rel);
    
    for b=1:chrom_bins
        seg_id(b)=knnsearch(int_rel',b/chrom_bins); % search for nearest bin edge intensity
    end
end

end




% Linear Regression
function [Yr,b]=lin_reg(x,y)
% insert x and y as column vecotrs, returns parameters b1=n, b2=m and
% regression vector Yr
X=[ones(length(x),1) x]; % Matrix mit dummy-Spalte
Y=y;
b=(inv(X'*X))*(X'*Y); % parameter determination
Yr=b(1)+b(2)*x; % get regression Y values
end