%As described in 
%"Immobilization of proteins for single-molecule fluorescence resonance energy transfer measurements of conformation and dynamics.
%Choi UB, Weninger KR, Bowen ME. 
%Methods Mol Biol. 2012;896:3-20

%Operates on MULTI IMAGE TIFF movie files.
%MUST ENTER CHANNEL OFFSET MANUALLY ON LINES 23 & 24!!! 

%select Background subtraction method: local median, SPARTAN, local minimum
%select Intensity calculation method: 4 brightest pixels, Sum 3x3 pixels about maxima 
%Output is signal to noise (S2N) and Intensity for Donor and Acceptor, raw FRET, coordinate of found spot,
%offset deviation for other side.


%Can also display all the peak finding images
%if breakpoint enabled.

function BowenLabMovieProcessor
close all;
warning('off','images:imshow:magnificationMustBeFitForDockedFigure')
warning('off','MATLAB:imagesci:tiffmexutils:libtiffErrorAsWarning')

set(0,'DefaultFigureWindowStyle','docked')

%MUST GET OFF SET FROM BEAD MOVIE AND ENTER!!!!!!
xoffset=0; %Set manually
yoffset=-1; %Set manually

%Adjustable parameters
MaxOFF=3; %Setting width of Peak Window for offset spot finding.  OG is one.
doubler=0; %set to 1 for double precision 2 for int32(i.e. allow negative numbers)
distanceAwayEdges=25;
averagesize=10;  %This sets how many frames are averaged to generate search_image
startframe=0; %Increase to skip frames
averagewindow=10; %This sets number of points for calculating Intenisty and FRET averages
nn=5; %acceptable pixel separation between nearest neighbors to avoid overlap.
THRESHx=1; %scale spot finding threshold


%Opening datafile via user prompt
[datafile,datapath]=uigetfile('*.tif','Choose a tif file');
if datafile==0, return; end
mov=1;
cd(datapath);
fid=fopen(datafile,'r');
file=datafile;
fileinfo = imfinfo(datafile); 
imagesizex=fileinfo.Width;  %take pixel resolution from the tif file info
imagesizey=fileinfo.Height;
NumberFrames = numel(fileinfo);
disp('working on');
disp(file);

%Take in laser switch points to make useable for all data sequences
prompstr={'Length of first red laser phase','Length of green laser phase','Analysis: 1-left, 2-right'};
initstr={'10','500','2'};
titlestr='DO NOT PRESS CANCEL!';
nlines=1;
result=inputdlg(prompstr,titlestr,nlines,initstr);
red1length=str2num(result{1});
greenlength=str2num(result{2});

%This can adjust for timing
greenON=red1length; %Give a few frames after switch point
greenOFF=red1length + greenlength;
side=str2double(result{3});

%Generation of Green Search Image
z=0;
fseek(fid,greenON,-1);
for k=greenON:greenON+averagesize
    z =z + imread(file,k);
end
green_image=z/averagesize;

%Generategreen Green BG from last 10 frames of green illumination
fseek(fid ,(greenOFF-averagesize),-1);
z=0;
for k=greenOFF-averagesize-1:greenOFF-1
    z =z + imread(file,k);
end
green_last=z/(averagesize);


% %Generate Red BG from last 10 frames of movie. Assuming second red!
z=0;
fseek(fid,(NumberFrames-averagesize),-1);
for k=NumberFrames-averagesize:NumberFrames
    z =z + imread(file,k);
end
red_last=z/averagesize;


%Generate Red image from first 10 frames of movie.
z=0;
fseek(fid,(1),-1);
for k=1:averagesize
    z =z + imread(file,k);
end
red_image=z/averagesize;

prompstr={'Select Background Subtraction method? press 1 for MED, 2 for SP, 3 for MIN'};
initstr={'3'};
titlestr='Processor';
nlines=1;
result=inputdlg(prompstr,titlestr,nlines,initstr);
BGPick=str2double(result);

%Select BG subtraction method
switch BGPick
    case 1  %Local Median (Bowen Lab Standard) 
        tempG = blkproc(green_last, [16 16],'median(x(:))');
        bg_image_green = imresize(tempG,[imagesizex imagesizey],'bicubic');
        tempR = blkproc(red_image, [16 16],'median(x(:))');
        bg_image_red =  imresize(tempR,[imagesizex imagesizey],'bicubic');
        tempRL = blkproc(red_last, [16 16],'median(x(:))');
        bg_image_lastred =  imresize(tempRL,[imagesizex imagesizey],'bicubic');
       
        
    case 2 %Blanchard SPARTAN BG
        
        %Juette, M. F.*, Terry, D. S.*, Wasserman, M. R., Altman, R. B., Zhou, Z., Zhao, H. & Blanchard, S. C. 
        %Single-molecule imaging of non-equilibrium molecular ensembles on the millisecond scale. 
        %Nature Methods 13, p. 341 (2016).
        
        %Left
        den=6;
        SBtemp = zeros( floor(imagesizey/den), floor(imagesizex/den) );
        
        for i=1:size(SBtemp,1)
            for j=1:size(SBtemp,2)
                sort_temp =  green_last(den*(i-1)+1:den*i,den*(j-1)+1:den*j);
                sort_temp = sort( sort_temp(:) );
                
                SBtemp(i,j) = sort_temp( den );  % get the 1/den % smallest value
            end
        end
        SBtemp=uint16(SBtemp);
        bg_image_green=imresize(SBtemp,[imagesizey imagesizex],'bicubic');
        
        %For Right Side
        SBtemp2 = zeros( floor(imagesizey/den), floor(imagesizex/den) );
        
        for i=1:size(SBtemp2,1)
            for j=1:size(SBtemp2,2)
                sort_temp =  red_image(den*(i-1)+1:den*i,den*(j-1)+1:den*j);
                sort_temp = sort( sort_temp(:) );
                
                SBtemp2(i,j) = sort_temp( den );  % get the 1/den % smallest value
            end
        end
        SBtemp2=uint16(SBtemp2);
        bg_image_red=imresize(SBtemp2,[imagesizex imagesizey],'bicubic');
        bg_image_lastred= bg_image_red;
        
    case 3 %Local Minimum
        %Left side
        SIMPLtempG = blkproc(green_last, [16 16],'min(x(:))');
        bg_image_green = imresize(SIMPLtempG,[imagesizex imagesizey],'bicubic');
        %Right side
        SIMPLtempR = blkproc(red_image, [16 16],'min(x(:))');
        bg_image_red =  imresize(SIMPLtempR,[imagesizex imagesizey],'bicubic');
        
        SIMPLtempRL = blkproc(red_last, [16 16],'min(x(:))');
        bg_image_lastred =  imresize(SIMPLtempRL,[imagesizex imagesizey],'bicubic');
        
        
end

%To accumulate negative numbers
if doubler==1
green_image=double(green_image);
green_last=double(green_last); %#ok<*NASGU>
red_last=double(red_last);
red_image=double(red_image);
bg_image_green=double(bg_image_green);
bg_image_red=double(bg_image_red);
bg_image_lastred=double(bg_image_lastred);
 

elseif doubler ==2
green_image=int32(green_image);
green_last=int32(green_last);
red_last=int32(red_last);
red_image=int32(red_image);
bg_image_green=int32(bg_image_green);
bg_image_red=int32(bg_image_red);
bg_image_lastred=int32(bg_image_lastred);
end

%Finding Peaks as local maxima above threshold
m_number=0;
switch side
    case 1  %Leftside
        disp('Left Side Peak Finding');
        
        search_image=green_image-bg_image_green;
        low=ceil(max(min(search_image)));
        high=ceil(mean(max(search_image)));
        imshow(search_image, [low high]);
        
        threshold=1.5*std2(search_image(distanceAwayEdges:imagesizey-distanceAwayEdges,distanceAwayEdges:imagesizex/2-distanceAwayEdges));
        disp('threshold');disp(threshold);
        
        for i=1+distanceAwayEdges:imagesizey-distanceAwayEdges %Y coord of region to look for peaks
            for j=1+distanceAwayEdges:imagesizex/2-distanceAwayEdges %x coord of region to look for peaks
                %point above threshold value and larger than neighbors
                if search_image(i,j)>threshold && search_image(i,j) == max(max(search_image(i-1:i+1,j-1:j+1)))
                    %count peaks found
                    m_number=m_number+1;
                    %store (x,y) of peak in list
                    tempx(m_number)=j;
                    tempy(m_number)=i;
                end
            end
        end
        
        %Tag Overlapping Peaks using Variable overlap 0/1
        overlap=zeros(1,m_number);
        for i=1:m_number
            for j=i+1:m_number
                if tempy(j)-tempy(i) > nn   %nn set Line 36 gives # pixels between maxima to avoid overlap
                    break;                  
                end
                %calculate hypotenuse between points in different columns.
                if (tempx(i)-tempx(j))^2+(tempy(i)-tempy(j))^2 < nn^2
                    overlap(i)=1;
                    overlap(j)=1;
                end
            end
        end
        
        %Finding Associated Rightside Peak
        j=0;
        for i=1:m_number          %loop on all molecues
            if overlap(i)==0    %if not overlapping
                j=j+1;          %j counts good molecules
                x(2*j-1)=tempx(i);         %odd members are original leftside spots
                y(2*j-1)=tempy(i);
                
                %apply offset to look for maxima in right side
                x(2*j)=tempx(i)+imagesizex/2+xoffset;
                y(2*j)=tempy(i)+yoffset;
                
                %select 3x3 region around point and find true maxima
                temp=search_image(y(2*j)-MaxOFF:y(2*j)+MaxOFF,x(2*j)-MaxOFF:x(2*j)+MaxOFF);
                [maxy, maxx]=find(temp==max(max(temp)));
                
                peakcheck=size(maxx);
                if peakcheck(1)~=1 %prevent catch if no peak on other side
                    maxx=2;
                    maxy=2;
                end
                IndvOffstX(i)=maxx-2;
                IndvOffstY(i)=maxy-2;
                %even members are emperically found maxima on right side
                x(2*j)=x(2*j)+maxx-2;
                y(2*j)=y(2*j)+maxy-2;
                
                line(x(2*j-1),y(2*j-1),'linestyle','none','marker','o','color','w');
                line(x(2*j),y(2*j),'linestyle','none','marker','o','color','y');
                
                donor_points(j,:)=[x(2*j-1) y(2*j-1)];
                acceptor_points(j,:)=[x(2*j) y(2*j)];
            end
            GoodOnes=j;
        end
        
    case 2 %Rightside
        disp('Right Side Peak Finding');
        search_image=red_image-bg_image_red;
        
        threshold=THRESHx*std2(search_image(distanceAwayEdges:imagesizey-distanceAwayEdges,imagesizex/2+distanceAwayEdges:imagesizex-distanceAwayEdges));
                
        disp('threshold');disp(threshold);
        for i=1+distanceAwayEdges:imagesizey-distanceAwayEdges                     %Y coord of region to look for peaks
            for j=imagesizex/2+1+distanceAwayEdges:imagesizex-distanceAwayEdges    %X coord of region to look for peaks
                %Pixel above threshold value and Highest in surrounding 3x3
                if search_image(i,j)>threshold && search_image(i,j) == max(max(search_image(i-1:i+1,j-1:j+1)))
                    %count peaks found
                    m_number=m_number+1;
                    %store (x,y) of peak in list
                    tempx(m_number)=j; %#ok<*AGROW>
                    tempy(m_number)=i;
                end
            end
        end
        
        
        %Tag Overlapping Peaks using Variable overlap 0/1
        overlap=zeros(1,m_number);
        for i=1:m_number
            for j=i+1:m_number
                if tempy(j)-tempy(i) > nn   %nn set Line 36 gives # pixels between maxima to avoid overlap
                    break;                  
                end
                %calculate hypotenuse between points in different columns.
                if (tempx(i)-tempx(j))^2+(tempy(i)-tempy(j))^2 < nn^2
                    overlap(i)=1;
                    overlap(j)=1;
                end
            end
        end
        
        
        %Find Associated Left Side Peak
        j=0;
        %NOT USING ACCEPTOR IMAGE AS NO EMISSION IN DONOR CHANNEL
        %using leftside search under green
        Lsearch_image=green_image-bg_image_green;
        
        for i=1:m_number
            
            if overlap(i)==0
                j=j+1;
                
                %Use Manually Entered offset and find local maximum on other side
                x(2*j-1)=tempx(i)-imagesizex/2-xoffset;
                y(2*j-1)=tempy(i)-yoffset;
                x(2*j)=tempx(i);
                y(2*j)=tempy(i);
                
                %Search Corresponding Region for Maxima
                temp=Lsearch_image(y(2*j-1)-MaxOFF:y(2*j-1)+MaxOFF,x(2*j-1)-MaxOFF:x(2*j-1)+MaxOFF);
                subplot(1,2,1);
                imagesc(temp); colormap(gray);
                title('Intial');
                
                [maxy, maxx]=find(temp==max(max(temp)));
                peakcheck=size(maxx);
                if peakcheck(1)~=1 %prevent catch if no peak on other side
                    disp('peakcheck fail');
                    maxx=MaxOFF+1;
                    maxy=MaxOFF+1;
                end
                x(2*j-1)=x(2*j-1)+maxx-MaxOFF-1;
                y(2*j-1)=y(2*j-1)+maxy-MaxOFF-1;
                
                temp2=Lsearch_image(y(2*j-1)-MaxOFF:y(2*j-1)+MaxOFF,x(2*j-1)-MaxOFF:x(2*j-1)+MaxOFF);
                subplot(1,2,2);
                imagesc(temp2); colormap(gray);
                title('Final');
               
                %Writing out where empirically found letside molecules are
                %located
                IndvOffstX(i)=maxx-2;
                IndvOffstY(i)=maxy-2;
                
                donor_points(j,:)=[x(2*j-1) y(2*j-1)];
                acceptor_points(j,:)=[x(2*j) y(2*j)];
                
            end
            GoodOnes=j;
        end
        
        display_image(1:imagesizey,imagesizex/2:imagesizex)=search_image(1:imagesizey,imagesizex/2:imagesizex);
        display_image(1:imagesizey,1:imagesizex/2)=Lsearch_image(1:imagesizey,1:imagesizex/2);
        low=ceil(max(min(display_image)));
        high=600;%ceil(mean(max(display_image)));
        figure; imshow(display_image, [low high]);
        for z=1:GoodOnes
            gcf;
            line(x(2*z-1),y(2*z-1),'linestyle','none','marker','o','color','w');
            line(x(2*z),y(2*z),'linestyle','none','marker','o','color','y');
        end
end
gcf;
disp('there are '); disp(num2str(GoodOnes)); disp(' molecules in this movie');

%Save traces of accepted maxima
squarewidth=3;      %width of pixel area containing maxima
NumPixels=4;        %number of pixels used to calculate intenisty based on the top brightest pixels. Standard is 4 

%Select Intensity Calculation
prompstr={'Select Intensity Calculation method? press 1 for top4pix or 2 for 3x3'};
initstr={'1'};
titlestr='Processor';
nlines=1;
result=inputdlg(prompstr,titlestr,nlines,initstr);
IntensityPick=str2double(result);


%WRITING TRACES
regions=zeros(NumPixels,2,2*GoodOnes);   %stores location of top 4 BRIGHTEST pixels 
centroid=zeros([squarewidth,squarewidth]);  %centroid stores 3x3 image of peak
maximaoutname=strrep(file, '.tif','.maxima');
peaklocate=zeros(GoodOnes,4);
counter=0;
for g=1:2*GoodOnes
    if mod(g,2)==1
        counter=counter+1;
        peaklocate(counter,1)=x(g);peaklocate(counter,2)=y(g); peaklocate(counter,3)=x(g+1);peaklocate(counter,4)=y(g+1);
    end
end
save(maximaoutname, 'peaklocate')


%Finding top 4 pixels associated with each identified peak.
for m=1:2*GoodOnes                                              %for each molecule found by peakfind alternating channels green/red
    peak=search_image(y(m)-floor(squarewidth/2):y(m)+floor(squarewidth/2),...
        x(m)-floor(squarewidth/2):x(m)+floor(squarewidth/2));       %read in 3x3 image surrounding top pixel
    peak=double(peak);                                              %convert peak to double precision to allow centroid to accumulate
    center=reshape(peak,squarewidth*squarewidth,1);                 %rearrange peak to single column
    center=sort(center);                                            %sort from low to high
    [I,J]=find(peak>=center(squarewidth*squarewidth-NumPixels+1));  %find the 4th highest pixel
    A=I(1:NumPixels);                                               %locate top 4 pixels and store as (x,y) = (A,B)
    B=J(1:NumPixels);
    regions(:,:,m)=[A+y(m)-floor(squarewidth/2)-1,B+x(m)-floor(squarewidth/2)-1]; %find top 4 pixels in whole image
    if mod(m,2)==1
        centroid=centroid+peak; %running sum of selected side peak image to get avg view.  uint16 is insufficent to capture this large number
        avgpeakview=centroid/m;
    end
    
end
disp('Done finding Pixels');
traceoutname=strrep(file, '.tif','TST.traces');   %get traces filename from tif file.
fid2 = fopen(traceoutname,'w');                             %creates .traces file
fwrite(fid2,NumberFrames-startframe,'int32');                 %write length of movie file. why int32?
fwrite(fid2,2*GoodOnes,'int16');                     %writes number of traces (2 for each molecule)
traces=zeros(2*GoodOnes+1,NumberFrames-startframe);

%Goes through each frame of the movie sequentially and records intensity for all
%molecules in the frame.
for k=1+startframe:NumberFrames          %Iterate on frames
    traces(1,k)=k;                  %first row is frame numbers
    rawimage=imread(file,k);%read in movie frame
    
    %For double precision set doubler to 1 at top
    if doubler==1
   rawimage=double(rawimage);
    elseif doubler==2
   rawimage=int32(rawimage);
    end
    
    %Check which color illumination is used to select BG for subtraction
    if k<red1length
        corrimage=rawimage-bg_image_lastred;
    elseif k>red1length && k<red1length+greenlength
        corrimage=rawimage-bg_image_green;
        
    elseif k>red1length+greenlength
        corrimage=rawimage-bg_image_lastred;
    end
    
    
    %Iterate on molecules within each frame
    for m=1:2*GoodOnes
        switch IntensityPick
            case 1
                %USES TOP 4 PIXELS AS INTENSITY
                traces(m+1,k)=trace(corrimage(regions(:,1,m),regions(:,2,m)));
            case 2
                %USES SUM OF ENTIRE 3x3 CENTROID AS INTENSITY
                traces(m+1,k)=sum(sum(corrimage(y(m)-floor(squarewidth/2):y(m)+floor(squarewidth/2),...
                    x(m)-floor(squarewidth/2):x(m)+floor(squarewidth/2))));
        end
    end
    fwrite(fid2,traces(1:1+2*GoodOnes,k),'int16');
    
end

fclose(fid2);

%%%%%Asking whether to continue to view all the traces
button = questdlg('Do you want to continue viewing all the traces?', 'Continue to TraceView', 'Yes', 'No', 'Yes')
if strcmp(button, 'No')
    return
else
%%%%%End question dialogue


%TraceViewing.
%USING TRACES IN MEMORY RATHER THAN READ IN OUTPUT FILE


disp('done saving traces '); 
disp(num2str(GoodOnes)); disp(' molecules in this movie');

%Select Trace View for FRET or Binding.  Changes Graphing
prompstr={'Select Trace View?  1 for FRET - 2 for Separate Intensity'};
initstr={'1'};
titlestr='Processor';
nlines=1;
result=inputdlg(prompstr,titlestr,nlines,initstr);
TraceView=str2double(result);


time1 = zeros(1,NumberFrames);
donor=zeros(GoodOnes,NumberFrames);
acceptor=zeros(GoodOnes,NumberFrames);
fretE=zeros(GoodOnes,NumberFrames);
time_1 = zeros(NumberFrames,1);
donor_1 = zeros(NumberFrames,GoodOnes);
acceptor_1 = zeros(NumberFrames,GoodOnes);

%Creating time for graph axis
z=0;
for z=1:NumberFrames
    time1(1,z)=z;
end

%Separating donor and acceptor traces into two separate matrices.
%Each row is a molecule
i=0;
for i=1:GoodOnes,
    donor(i,:)=traces(i*2,:);
    acceptor(i,:)=traces(i*2+1,:);
end

figure;
axis_x = NumberFrames*1;	%for time lapse need to multiply by spacing (usu 350)--for regular *1
m1=1;
%m1 = input('number of starting trace = ');
leakage = 0.08; %for Alexa555, 0.09 for cy3, set to zero for calcein

%sets up go back during trace viewing to allow user to go back 2 traces
AnotherPrevLastGood=m1;
prevlastgood=m1;
lastgood=m1;
m=m1;
saved=0;
while m < (GoodOnes + 1); %for each Movie
    
    %Creating 3 separate 1 column matrices for the frame# (time_1).  The
    %donor & accptr intensity (donor_1, acceptor_1)
    time_1 = time1';
    donor_1 = donor(m,:)';
    acceptor_1 = acceptor(m,:)'-leakage*donor_1;
    
    %       YFPmax=-100;
    
    
    
    for i = 1:NumberFrames %for each frame of a movie
        
        %       if acceptor_1(i,1)>YFPmax
        %          YFPmax = acceptor_1(i,1); %track highest acceptor emission value
        %       end
        
        %Calculate FRET for points with non-zero intensity
        total = donor_1(i,1)+acceptor_1(i,1);
        if total <= 0
            fret_1(i,1) = 0.01; %negative points called 0.01
            %disp('negative FRET');
        else
            fret_1(i,1) = acceptor_1(i,1)/(donor_1(i,1)+acceptor_1(i,1));
        end
        %iterate on each frame
    end
   
    
    switch TraceView
            case 1 %FRET Plot
    %initalize plot viewer w/ emission in top subplot @ calc FRET in second subplot
    gcf;
    subplot (2,1,1)
    plot(time_1(1:NumberFrames-5),donor_1(1:NumberFrames-5),'g',time_1(1:NumberFrames-5),acceptor_1(1:NumberFrames-5),'r');
    whitebg('k')
    grid on;
    temp=axis;
    temp(1)=0;
    temp(2)=axis_x;
    axis(temp);
    title(['Molecule ' num2str(m) '/' num2str(GoodOnes)]);
    zoom on;
    subplot (2,1,2)
    plot(time_1(1:NumberFrames-5),fret_1(1:NumberFrames-5),'m');
    grid on;
    axis([0 axis_x -0.1 1.1])
    zoom on;
        case 2 %Separate Intensity
    %initalize plot viewer w/ acceptor emission in top subplot @ donor in second subplot  
   subplot (2,1,1) 
   plot(time_1(1:NumberFrames-5),acceptor_1(1:NumberFrames-5),'r'); %to plot acceptor only
   grid on;
   temp=axis;
   temp(1)=0;
   temp(2)=axis_x;
   axis(temp);
   title(['Molecule ' num2str(m) ' ' file]);
   zoom on;
   subplot (2,1,2)
   plot(time_1(1:NumberFrames-5),donor_1(1:NumberFrames-5),'g'); %to plot donor only
   temp2=axis;
   temp2(1)=0;
   temp2(2)=axis_x;
   axis(temp2);
   zoom on;
    end
    outfile=file(1:numel(file)-4);
    
    what2do = input('press s to save OR b to back up a trace OR other to pass === ','s');
    
    
    %if trace not saved advance to next trace.  Adjust backup function
    if isempty(what2do)== 1 
        what2do=' ';
        AnotherPrevLastGood=prevlastgood;
        prevlastgood=lastgood;
        lastgood = m;
        m=m+1;
    end
    
    %play movie of spots
    if what2do=='m'
        willplay=0;
        if mov~=0
            while willplay==0
                prompstr={'Start Frame','Frames to Play','Pause'};
                initstr={'1','50','0.05'};
                titlestr='Movie Player';
                nlines=1;
                result=inputdlg(prompstr,titlestr,nlines,initstr);
                frame1=str2num(result{1});
                frames2play=str2num(result{2});
                pausetime=str2num(result{3});
                if frame1+frames2play<NumberFrames
                    willplay=1;
                else
                end
                continue
            end
            
            showwidth=7;
            dlow=0.1*donor_1(11)+1;
            dhigh=max([1000 1.25*donor_1(11)+1]);
            alow=min([dlow 0.1*acceptor_1(11)+1]);
            ahigh=max([1.25*(acceptor_1(11)+1) dhigh]);
            fseek(fid,0,-1);
            
            for d=frame1:frame1+frames2play
                rawimage=imread(file,d);
                subplot(1,2,1)
                imshow(rawimage (peaklocate(m,2)-floor(showwidth/2):peaklocate(m,2)+floor(showwidth/2),...
                    peaklocate(m,1)-floor(showwidth/2):peaklocate(m,1)+floor(showwidth/2)), [dlow dhigh]);
                title(['frame ' num2str(d) ' of ' num2str(frames2play)]);
                subplot(1,2,2)
                imshow(rawimage (peaklocate(m,4)-floor(showwidth/2):peaklocate(m,4)+floor(showwidth/2),...
                    peaklocate(m,3)-floor(showwidth/2):peaklocate(m,3)+floor(showwidth/2)), [alow ahigh]);
                title(['D=' num2str(donor_1(d)) ' A=' num2str(acceptor_1(d))]);
                drawnow;
                pause(pausetime);                               
            end
            
        else
            disp('no movie for these traces')
        end

    end
    %Quit if 'q' is pressed
    if what2do== 'q'
        m=GoodOnes*2;
    end
    %backup trace viewing
    if what2do== 'b'
        m=lastgood;
        lastgood=prevlastgood;
        prevlastgood=AnotherPrevLastGood;
    end
    %write saved traces to file
    if what2do == 's'
        saved = saved+1;
        DONOR_temp=donor_1(greenON:greenON+averagewindow,1);
        ACCEPTOR_temp=acceptor_1(greenON:greenON+averagewindow,1);
        FRET_temp=fret_1(greenON:greenON+averagewindow,1);
        avg_FRET(saved,1)=mean(FRET_temp);
        avg_D(saved,1)=mean(DONOR_temp);
        avg_A(saved,1)=mean(ACCEPTOR_temp);
        s2n_D(saved,1) = (mean(DONOR_temp))/std(DONOR_temp);
        s2n_A(saved,1) = (mean(ACCEPTOR_temp))/std(ACCEPTOR_temp);
        Xcoord(saved, 1)=x(m);
        Ycoord(saved,1)=y(m);
        disp('Now saving trace number  ');disp(saved);
        fname=[outfile 'trace' num2str(m) '.itx']; %setting file extension
        igorname=[outfile 'trace' num2str(m)];
        [fid4,message] = fopen(fname,'w');
        
        %This sets up the file header specific to Igor
        fprintf(fid4,'IGOR\nWAVES/D ');		
        fprintf(fid4,'donor');
        fprintf(fid4,igorname);
        fprintf(fid4,' ');
        fprintf(fid4,'acceptor');
        fprintf(fid4,igorname);
        fprintf(fid4,'\n');
        fprintf(fid4,'BEGIN\n');
        
        %Writing actual data point by point File structure is:
        %donor-space-acceptor-\n last is Igor specific
        for i = 1:NumberFrames
            fprintf(fid4, num2str(donor_1(i,1)));
            fprintf(fid4, '	');
            fprintf(fid4, num2str(acceptor_1(i,1)));
            fprintf(fid4, '\n');
        end
        
        fprintf(fid4,'\nEND\n');
        fclose(fid4);
        
        %Tracking trace # for backup function
        AnotherPrevLastGood=prevlastgood;
        prevlastgood=lastgood;
        lastgood = m;
        m=m+1;
        
    end
end %end of plot viewing for a trace
disp('done');


save
if saved>0
    %Generate text files for output
    
    %Location of maxima
    dlmwrite('SelectXcoord.txt',Xcoord,'Delimiter','\t'); 
    dlmwrite('SelectYcoord.txt',Ycoord,'Delimiter','\t');
    
    %Channel offset of individual maxima in pixels
    dlmwrite('IndvOffstX.txt',IndvOffstX,'Delimiter','\t'); 
    dlmwrite('IndvOffstY.txt',IndvOffstY,'Delimiter','\t'); 
    
    %Raw FRET from first frames (Set on Line 33) 
    dlmwrite('avg_FRET.txt',avg_FRET,'Delimiter','\t');
    
    %Signal Properties for molecules from each channel
    dlmwrite('s2n_D.txt',s2n_D,'Delimiter','\t'); 
    dlmwrite('s2n_A.txt',s2n_A,'Delimiter','\t'); 
    dlmwrite('avg_D.txt',avg_D,'Delimiter','\t'); 
    dlmwrite('avg_A.txt',avg_A,'Delimiter','\t'); 
    
    %Generate Summary Figure
    figure;
    subplot(4,1,1)
    imagesc(avgpeakview);
    title('Avg Peak Selected Side');
    subplot(4,1,2)
    hist(avg_FRET,20);
    title('FRET');
    subplot(4,2,1)
    hist(s2n_D,20);
    title('S2N D');
    subplot(4,2,2)
    hist(s2n_A,20);
    title('S2N A');
end
set(0,'DefaultFigureWindowStyle','normal')

end


