//                        # Pre-Processing for Brightfield
// 						Created by Julian Stopp on 05.12.2018
//								Edited by Saren Tasciyan
//								Last updated: 01.08.2019

/*
 * What does this macro do? 
 *  a) It enables setting image properties dynamically
 *  b) Potential of fitting the histogram to obtain an equalized illumination of your sample
 *  c) It further allowes a stabilization of your video to remove drifts
 *  d) It also allowes the substraction of the median to remove background signals
 *
 * What shall I use that macro for?
 *  For pre-processing of images for Trackmate or Roberts script (but beware of the issues, see below)
 *
 * Requirements:
 *  Images need NOT to be RGB files as registration can not cope with it! 
 *
 * Needed Plugins:
 *  fit polynomial: http://imagejdocu.tudor.lu/doku.php?id=plugin:filter:fit_polynomial:start
 *  -> download, use Plugins-> Install and click okay (will also install Java compiler that way that it can run) 
 *
 * Known issues:
 *  1. Median substraction can cause issues when your cells are stopping at some point as then the cells are removed from the images after some points (again an issue for Robert's script)
 * 
 * Issues which are auto-solved with the script
 *  1. FitPol is reducing the efficiency of "stack - median"; to bypass this you need to create a 32 bit image and then create "stack - median" 
(this is done autmatically)
 *  2. Stabilization adds emtpy pixels at border regions to correct for drifts. These are removed via cropping the image. Hence, the final output will be a smaller image than the original. 
 *  
*/

// Funtion declarations
function getImageFileExtension(fileName){
	recognizedFileTypes = newArray(".png", ".tif", ".tiff", ".jpg", ".jpeg", ".bmp");
	lowerCaseFileName = toLowerCase(fileName);
	for(i = 0; i < lengthOf(recognizedFileTypes); i++){
		if(endsWith(lowerCaseFileName, recognizedFileTypes[i])){
			return recognizedFileTypes[i];
		}
	}
	return "";
}

// Closes all images
Dialog.create("Warning");
Dialog.addMessage("Script will close ALL open images!")
items = newArray("Yes", "No");
Dialog.addRadioButtonGroup("Do you want to continue?",items, 1, 2, "Yes");
Dialog.show;
Continue=(Dialog.getRadioButton=="Yes");
if(Continue){
	close("*");	
}else{
	exit();
}

// Create Dialog
Dialog.create("Adjust parameters")

items = newArray("Entire folder             ", "Single File");
Dialog.addRadioButtonGroup(" ", items, 1, 2, "Entire folder             ");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Do you want to set image properties (pixel size, etc.)?",items, 1, 2, "Yes       ");

Dialog.addMessage("                           Please set the image properties:")
Dialog.addNumber("                                            Pixel size:", 2.301);
Dialog.addString("Scale in:", "µm")
Dialog.addNumber("Framerate [x]:", 0.5);
Dialog.addString("Framerate in [x]:", "min.");
Dialog.addNumber("Voxeldepth:", 1);

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Do you want to equalize the illumination of the video?",items, 1, 2, "Yes       ");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Do you want to stabilize the video?",items, 1, 2, "Yes       ");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Do you want to select a ROI for stabilization?",items, 1, 2, "No");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Do you want to substract the median of the stack?",items, 1, 2, "Yes       ");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("Save file(s) at the end? (highly suggested for folder processing, otherwise errors may occur)",items, 1, 2, "Yes       ");

items = newArray("Yes       ", "No");
Dialog.addRadioButtonGroup("If save option selected, close the file(s) after save? (highly suggested for folder processing, otherwise errors may occur)",items, 1, 2, "Yes       ");

Dialog.show;

// Save variables (not needed as they are stored anyhow, but easier to track)
isFolder=(Dialog.getRadioButton=="Entire folder             ");
isProperties=(Dialog.getRadioButton=="Yes       ");
ps = Dialog.getNumber();
sc = Dialog.getString();
fr = Dialog.getNumber();
fr2 = Dialog.getString();
vd = Dialog.getNumber();
isEqualize=(Dialog.getRadioButton=="Yes       ");
isStabilize=(Dialog.getRadioButton=="Yes       ");
isROI=(Dialog.getRadioButton=="Yes       ");
isMedian=(Dialog.getRadioButton=="Yes       ");
isSave=(Dialog.getRadioButton=="Yes       ");
isClose=(Dialog.getRadioButton=="Yes       ");

// Load files
if (isFolder) {
	dir1 = getDirectory("Choose a Directory ");
	list = getFileList(dir1);
} else {
	path = File.openDialog("Select file");

	list=newArray(1);
	list[0]=path;
	dir1 = File.getParent(path) + File.separator;
}

// Batch mode saves time as it prevents images from beeing displayed
setBatchMode("show");

// Loop through all files
for (ii=0; ii<list.length; ii++) {
	fileSuffix = "";
	fileExt = getImageFileExtension(list[ii]);
	
	if (lengthOf(fileExt) > 0) {
		// Open image & rename
		print(list[ii]);
		open(list[ii]);
		name=getTitle();
		mainImage=getImageID();
		//rename("WorkStack.tif");
		
		// Get image details
		Stack.getDimensions(width, height, channels, slices, frames);
		
		// Set Image Properties
		if (slices>frames) {
			stack_size=slices;
			run("Properties...", "slices=1 frames="+stack_size);
		}
		
		if (isProperties) {
			run("Properties...", "unit="+sc+" pixel_width="+ps+" pixel_height="+ps+" voxel_depth="+vd+" frame=["+fr+" "+fr2+"]");
		}
		mainImage=getImageID();
		rename("Result.tif");
		
		// Fit polynomial to equalize histogram an illumination (does change histrogram!)
		if (isEqualize){
			run("fit polynomial", "x=2 y=2 mixed=0 stack");
			mainImage=getImageID();
			rename("Result.tif");
			fileSuffix += "_FitPoly";
		}
		
		// Correct drift and crop images after correction; also limits crop
		if (isStabilize){

			print("\\Clear"); // Clear the log
			
			selectImage(mainImage);
			Stack.getDimensions(width, height, channels, slices, frames);
			ori_width=width;
			ori_height=height;
			
			if (isROI){
					waitForUser("Generate ROI", "Generate an ROI witht the respective tool! \n \nPress OK to continue.");
					run("Correct 3D drift", "channel=1 multi_time_scale sub_pixel edge_enhance only=0 lowest=1 highest=1");
			}	else {
					run("Correct 3D drift", "channel=1 multi_time_scale sub_pixel edge_enhance only=0 lowest=1 highest=1");
			}
									
			driftCorrectedImage=getImageID();
			
			selectImage(driftCorrectedImage);
			Stack.getDimensions(width, height, channels, slices, frames);
			new_width=width;
			new_height=height;
			
			// check if image became to big - indicator for wrongly corrected drift
			if (((new_width-ori_width)/ori_width)<0.03 || ((new_height-ori_height)/ori_height)<0.03){
				selectImage(mainImage);
				close();
				mainImage=driftCorrectedImage;
				selectImage(mainImage);
				
				// Drift crop 
				regs = getInfo("log");
				regsA = split(regs, "\n");
				shiftsX = newArray(frames);
				shiftsY = newArray(frames);
				shiftsZ = newArray(frames);
				for(i = 0; i < lengthOf(regsA); i++){
					if(startsWith(regsA[i], "    frame")){
						params = split(regsA[i], " ");
						coordinates = split(params[4], ",");
						
						shiftsX[parseInt(params[1]) - 1] = parseFloat(coordinates[0]);
						shiftsY[parseInt(params[1]) - 1] = parseFloat(coordinates[1]);
						shiftsZ[parseInt(params[1]) - 1] = parseFloat(coordinates[2]);
					}
				}
				Array.getStatistics(shiftsX, minX, maxX, meanX, stdDevX);
				Array.getStatistics(shiftsY, minY, maxY, meanY, stdDevY);
				Array.getStatistics(shiftsZ, minZ, maxZ, meanZ, stdDevZ);
				rangeX = maxX - minX;
				rangeY = maxY - minY;
				makeRectangle(rangeX, rangeY, width - rangeX * 2, height - rangeY * 2);
				run("Crop");
				
				// remove 1 pixel from all sides (after drift correction there are weird pixels at the border)
				Stack.getDimensions(width, height, channels, slices, frames);
				makeRectangle (1, 1,width-2, height-2);
				run("Crop");
				mainImage=getImageID();
				rename("Result.tif");
			} else {
				selectImage(driftCorrectedImage);
				close();
				Dialog.create("ERROR");
				Dialog.addMessage("Drift cannot be corrected with current settings! \n \nPress OK to continue.")
				Dialog.show;
				mainImage=getImageID();
				rename("Result.tif");
			}
			fileSuffix += "_Stab";
		}
		
		// Calculate median of stack and substract it from all
		if (isMedian){
			selectImage(mainImage);
			run("32-bit");
			run("Z Project...", "projection=Median");
			medianImage=getImageID();
			imageCalculator("Subtract create stack", mainImage, medianImage);
			medianSubstractedImage=getImageID();
			mainImage=medianSubstractedImage;
			selectImage(mainImage);
			run("8-bit");
			selectWindow("Result.tif");
			close();
			mainImage=getImageID();
			rename("Result.tif");
			fileSuffix += "_MedSub";
		}
		
		// Rename modified image & save
		//waitForUser;
		if(isSave){
			selectWindow("Result.tif");
			nameCore = substring(name, 0, lengthOf(name) - lengthOf(fileExt));
			newName = nameCore + fileSuffix + ".tif";
			newPath = dir1 + newName;
			save(newPath);
			
			if(isClose){
				close("*");
			}
		}
	}
}

write("done");
setBatchMode(false);