import pandas as pd
import matplotlib.pyplot as plt
import numpy as np
from scipy.signal import find_peaks

# Import the data file
df = pd.read_csv('', encoding='shift_jis')

# Define range of x-axis
start = 2.8  # Minimum x value
end = 3.5   # Maximum x value

# Generate list of time (1 row represents 1/66.7 seconds)
time = np.arange(0, len(df) / 66.7, 1/66.7)

def moving_average(data, window_size):
    window = np.ones(window_size) / window_size
    return np.convolve(data, window, mode='valid')

# Get column names (data from the first row)
column_names = df.columns.values.tolist()
colors = ['blue', 'magenta'] 


for idx, col in enumerate(column_names[:2]):
    window_size = 5
    smoothed_wave = moving_average(df[col], window_size) # Find peaks

    # Adjust time stamp
    time_adjusted = time[: len(smoothed_wave)]

    # Get the index within the selected range
    start_index = np.where(time_adjusted >= start)[0][0]
    end_index = np.where(time_adjusted <= end)[0][-1]

    # Obtain and normalize the data within the selected range
    smoothed_wave = smoothed_wave[start_index: end_index]
    smoothed_wave = (smoothed_wave - smoothed_wave.min()) / (smoothed_wave.max() - smoothed_wave.min()) * 100

    peaks, _ = find_peaks(smoothed_wave, prominence=2.8)

    # Draw the smoothed waveform
    plt.plot(time_adjusted[start_index: end_index], smoothed_wave, label='Smoothed ' + col, color=colors[idx],linewidth=5)

    # Draw peaks with red 'x' markers
    plt.plot(time_adjusted[start_index: end_index][peaks], smoothed_wave[peaks], 'kx',markersize=1)

ax = plt.gca()
for axis in ['bottom','left']:
    ax.spines[axis].set_linewidth(2)

for axis in ['top','right']:
    ax.spines[axis].set_linewidth(0)
# Hide the y-axis numbers
ax.yaxis.set_ticklabels([])

# Change the thickness of the axis ticks
ax.xaxis.set_tick_params(width=2)
ax.yaxis.set_tick_params(width=0)
# Legend setting
# plt.legend()

# Restrict the range of the x-axis from start to end
plt.xlim(start, end)

# Set the range of the y-axis from 0 to 100
plt.ylim(-10, 110)

# Set x-axis to time notation
plt.xlabel('Time (s)')

# Display graph
plt.show()
