abstract type cell end
@with_kw mutable struct stem <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.35
    birthrad::Float64 = 0.35
    rstar::Float64 = rand(Distributions.Normal(0.35,0.35/stdev))

    phase::String = "G1"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*21.5/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = SVector(0.009940445423126221, 0.1668413728475571, 0.07760512828826904, 0.6716265678405762, 0.0, 0.6605867147445679, 0.01855352707207203, 0.9992357492446899, 0.2954076826572418, 1.0, 0.009814870543777943, 0.0, 0.01715379953384399, 0.3117263317108154,1.0,.0,1.0)

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = false
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
end

# ABSORPTIVE CELLS

@with_kw mutable struct absprog <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0
    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G1"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = SVector(0.009940445423126221, 0.1668413728475571, 0.07760512828826904, 0.6716265678405762, 0.0, 0.6605867147445679, 0.01855352707207203, 0.9992357492446899, 0.2954076826572418, 1.0, 0.009814870543777943, 0.0, 0.01715379953384399, 0.3117263317108154,1.0,.0,1.0)

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0
    timesincedivide::Float64 = 0.0

    notchemit::Bool = false
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
    nsplits::Int = 0
end

@with_kw mutable struct entero <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G0"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,1.0,.0,1.0]

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = false
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
    nsplits::Int = 0
end

# SECRETORY CELLS

@with_kw mutable struct secprog <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G0"
    status::String = "dunno"
    fate::String = rand(["goblet","goblet","goblet","goblet","goblet","goblet","goblet","goblet","goblet","entendo"])
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,1.0,.0,1.0]

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = true
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
end

@with_kw mutable struct goblet <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G0"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,1.0,.0,1.0]

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = true
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
end

@with_kw mutable struct entendo <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 4
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G0"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,1.0,.0,1.0]

    lifespan::Float64 = rand(Gamma(100,(1/100)*5))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = true
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = false
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
end

@with_kw mutable struct paneth <: cell
    pos::SVector{3,Float64}
    vel::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    force::SVector{3,Float64} = SVector(0.0,0.0,0.0)
    pressure::Float64 = 0.0

    youngmod::Float64 = 25
    pratio::Float64 = 0.4

    cellrad::Float64 = 0.5
    birthrad::Float64 = 0.3
    rstar::Float64 = rand(Distributions.Normal(0.5,0.5/stdev))

    phase::String = "G0"
    status::String = "dunno"
    fate::String = "undecided"
    splittime::Float64 = rand(Gamma(100,(1/100)*10/24))
    growrate::Float64 = 0
    timeinstate::Float64 = 0.0
    cyclevec::SVector{17,Float64} = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,1.0,.0,1.0]

    lifespan::Float64 = rand(Gamma(100,(1/100)*pannydeff))
    talive::Float64 = 0.0

    timesincedivide::Float64 = 0.0

    notchemit::Bool = true
    notch::Float64 = 0.0
    notchgiving::Int = 0

    wntemit::Bool = true
    boundwnt::Float64 = 0.0

    ID::Vector{Int} = []
end

mutable struct consts
    mintop::Float64
    cellpos::Float64
    zent::Float64
    nent::Float64
    p::Float64
    deff::Float64
    enterodeff::Float64
    secdeff::Float64
    villuslength::Int
end

mutable struct infopack
    ki67status::Bool
    brdustatus::Bool
    type::Int
    pos::SVector{3,Float64}
    origpos::SVector{3,Float64}
    cellrad::Float64
end

function setup(crypttype)
    crypty = rand(1:10)
    villusy = rand(1:10)

    choice, consts = subset(crypttype)

    cells = choice[crypty,1]
    villus = choice[villusy,2]

    if crypttype == "ablation"
        fixdeffs(cells,consts)
    end

    return cells, villus, consts
end

function subset(crypttype)
    if crypttype == "jeju"
        return load("mousejeju.jld","data"), consts(18.0,21,14.5*18.0/21,940,8.5,6,6,11,1000)
    elseif crypttype == "ileum"
        return load("mouseileum.jld","data"), consts(16.0,19,13*16.0/19,920,4,6.5,6.5,11.5,1000)
    elseif crypttype == "ablation"
        return load("mousebarker.jld","data"), consts(16.0,19,13*16.0/19,920*0.7,4,6.5,4,11.5,700)
    end
end

function sumbrdu(m)
    w,x,y,z = size(m)
    result = zeros(eltype(m),x,y,z)
    for i in 1:w
        for j in 1:x
            for k in 1:y
                for l in 1:z
                    result[j,k,l] += m[i,j,k,l]
                end
            end
        end
    end
    return result
end

function avgcount(m)
    x,y,z = size(m)
    result = zeros(Float64,y,z)
    for j in 1:y
        for k in 1:z
           result[j,k] = mean(m[:,j,k])
        end
    end
    return result
end


function stemcellfromsphericalcoords(rad, theta, phi)
    a = stem(pos = [rad * sin(theta) * cos(phi), rad * sin(theta) * sin(phi), rad * cos(theta)], notch = rand(Uniform(100,200)))
    return a
end

function panethcellfromsphericalcoords(rad, theta, phi)
    a = paneth(pos = [rad * sin(theta) * cos(phi), rad * sin(theta) * sin(phi), rad * cos(theta)], boundwnt = 128, notch = rand(Uniform(0,200)), talive = rand(Uniform(0,50)))
    return a
end

function convert2stem(a::cell,deff)
    newlifespan = rand(Gamma(100,(1/100)*deff))
    newrstar = rand(Distributions.Normal(0.35,0.35/stdev))
    if cbrt(2)*newrstar < a.cellrad
        newrstar = cbrt(1/2)*a.cellrad
    end
    newa = stem(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, rstar = newrstar, boundwnt = a.boundwnt, splittime = a.splittime, cyclevec = a.cyclevec, notch = a.notch,  ID = a.ID, lifespan = newlifespan, talive = (newlifespan/a.lifespan)*a.talive)
    return newa
end

function convert2absprog(a::cell,deff)
    newrstar = rand(Distributions.Normal(0.5,0.5/stdev))
    if cbrt(2)*newrstar < a.cellrad
        newrstar = cbrt(1/2)*a.cellrad
    end
    newa = absprog(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, rstar = newrstar, boundwnt = a.boundwnt, splittime = a.splittime, cyclevec = a.cyclevec, notch = a.notch,  ID = a.ID, talive = a.talive,lifespan=rand(Gamma(100,(1/100)*deff)))
    return newa
end

function convert2secprog(a::cell,secdeff)
    newa = secprog(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, boundwnt = a.boundwnt, splittime = a.splittime, notch = a.notch,  ID = a.ID,lifespan=rand(Gamma(100,(1/100)*secdeff)))
    return newa
end

function convert2entero(a::cell,enterodeff)
    newa = entero(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, rstar = a.cellrad, boundwnt = a.boundwnt, splittime = a.splittime, notch = a.notch,  ID = a.ID, nsplits = a.nsplits,lifespan=rand(Gamma(100,(1/100)*enterodeff)))
    return newa
end

function convert2goblet(a::cell)
    newa = goblet(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, rstar = a.cellrad, boundwnt = a.boundwnt, splittime = a.splittime, notch = a.notch,  ID = a.ID, lifespan = a.lifespan, talive = a.talive)
    return newa
end

function convert2entendo(a::cell)
    newa = entendo(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, rstar = a.cellrad, boundwnt = a.boundwnt, splittime = a.splittime, notch = a.notch,  ID = a.ID, lifespan = a.lifespan, talive = a.talive)
    return newa
end

function convert2paneth(a::cell)
    newlifespan = rand(Gamma(100,(1/100)*pannydeff))
    newa = paneth(pos = a.pos, force = a.force, cellrad = a.cellrad, birthrad = a.cellrad, boundwnt = a.boundwnt, splittime = a.splittime, notch = a.notch,  ID = a.ID, lifespan = newlifespan, talive = (newlifespan/a.lifespan)*a.talive)
    return newa
end

function agecells(a::cell)
    a.talive += dt
    a.timeinstate += dt
    a.timesincedivide += dt
    return nothing
end


function forcesandsig(grid,cells)
    @inbounds for i in eachindex(grid)
        cellcellinteractions(grid[i],cells)
        for j in offsets
            if haskey(grid, i+j)
                cellcellinteractions(grid[i],grid[i+j],cells)
            end
        end
    end
end

function cellcellinteractions(inds,cells)
    for i in 1:(length(inds)-1)
        for j in (i+1):length(inds)
            interactions(cells[inds[i]],cells[inds[j]])
        end
    end
end

function cellcellinteractions(inds,otherinds,cells)
    for i in inds
        for j in otherinds
            interactions(cells[i],cells[j])
        end
    end
end

function interactions(a,b)
    restlen = a.cellrad + b.cellrad
    diff = b.pos - a.pos
    dist = mag(diff)

    sig(restlen,dist,a,b)
    forces(restlen,dist,diff,a,b)

    return nothing
end


function villusloop(villus,BMP,top,zent,deff,secdeff,enterodeff)
    @inbounds for i in eachindex(villus)
        villusloopcontent(villus[i],BMP,top,zent)
        differentiate(villus,i,deff,secdeff,enterodeff)
    end
    return nothing
end

function villusloopcontent(a::cell,BMP,top,zent)
    villusfate(a,BMP,top,zent)

    if a.fate == "divide" && a.phase != "arrest"
        a.phase = "splitting"
    end

    agecells(a)
    begindeathshrink(a)
    setgrowrate(a)
    growcells(a,1)

    return nothing
end

function cellremoval(cells,villus,top,villuslength)
    filter!(x -> x.cellrad > 0.1 || x.phase != "dying", cells)
    filter!(x -> x.phase != "popping", cells)
    filter!(x -> x.cellrad > 0.1 || x.phase != "dying", villus)

    append!(villus,filter(x->x.pos[3]>top, cells))
    filter!(x->x.pos[3] <= top , cells)

    if length(villus) > villuslength
        diff = length(villus) - villuslength
        deleteat!(villus,1:diff)
    end

    return nothing
end

function fixdeffs(cells,consts)
    for i in eachindex(cells)
        if typeof(cells[i]) == absprog || typeof(cells[i]) == stem
            cells[i].lifespan = rand(Gamma(100,(1/100)*consts.deff))
        elseif typeof(cells[i]) == entero
            cells[i].lifespan = rand(Gamma(100,(1/100)*consts.enterodeff))
        elseif typeof(cells[i]) == secprog || typeof(cells[i]) == entendo || typeof(cells[i]) == goblet
            cells[i].lifespan = rand(Gamma(100,(1/100)*consts.secdeff))
        elseif typeof(cells[i]) == paneth
            cells[i].lifespan = rand(Gamma(100,(1/100)*pannydeff))
        else
            cells[i].lifespan = rand(Gamma(100,(1/100)*consts.deff))
        end
    end
end

function fatedecider(a::stem,BMP,top,zent)
    if a.boundwnt < wntbound
        if a.notch <= notchold*0.99
            a.fate = "secprog"
            a.phase = "G0"
            a.timeinstate = 0
        else
            a.fate = "absprog"
            a.phase = "G0"
            a.timeinstate = 0
        end
    elseif a.boundwnt >= wntbound && a.notch <= pannynotchold*0.99
        a.fate = "paneth"
        a.phase = "G0"
        a.timeinstate = 0
    else
        a.fate = "divide"
        a.phase = "S"
        if a.cyclevec[16] <= .0
            a.cyclevec = @SVector [a.cyclevec[1],a.cyclevec[2],a.cyclevec[3],a.cyclevec[4],a.cyclevec[5],a.cyclevec[6],a.cyclevec[7],a.cyclevec[8],a.cyclevec[9],a.cyclevec[10],a.cyclevec[11],a.cyclevec[12],a.cyclevec[13],a.cyclevec[14],a.cyclevec[15]/2,a.cyclevec[15]/2,a.cyclevec[17]]
        end
        a.timeinstate = 0
    end

    return nothing
end

function fatedecider(a::secprog,BMP,top,zent)
    if a.boundwnt >= wntbound
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    end

    return nothing
end

function fatedecider(a::paneth,BMP,top,zent)
    if a.notch > overnotchold
        a.fate = "notchdiffprep"
        a.phase = "G0"
        a.timeinstate = 0.0
    elseif a.boundwnt < wntbound
        a.fate = "wntdiffprep"
        a.phase = "G0"
        a.timeinstate = 0.0
    end

    return nothing
end

function fatedecider(a::goblet,BMP,top,zent)
    if a.boundwnt >= wntbound
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    end

    return nothing
end
function fatedecider(a::entendo,BMP,top,zent)
    if a.boundwnt >= wntbound
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    end

    return nothing
end

function fatedecider(a::absprog,BMP,top,zent)
    BMPdetected = BMP * exp(-log(8)/(top - zent)*(top - a.pos[3]))
    if a.boundwnt >= wntbound
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    elseif a.boundwnt <= BMPdetected
        a.fate = "entero"
        a.phase = "G0"
        a.timeinstate = 0.0
    else
        a.fate = "divide"
        a.phase = "S"
        if a.cyclevec[16] <= .0
            a.cyclevec = @SVector [a.cyclevec[1],a.cyclevec[2],a.cyclevec[3],a.cyclevec[4],a.cyclevec[5],a.cyclevec[6],a.cyclevec[7],a.cyclevec[8],a.cyclevec[9],a.cyclevec[10],a.cyclevec[11],a.cyclevec[12],a.cyclevec[13],a.cyclevec[14],a.cyclevec[15]/2,a.cyclevec[15]/2,a.cyclevec[17]]
        end
        a.timeinstate = 0.0
    end

    return nothing
end

function fatedecider(a::entero,BMP,top,zent)
    if a.boundwnt >= wntbound
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    end
    return nothing
end

function differentiate(cells,i,deff,secdeff,enterodeff)
    if cells[i].fate == "stem" && cells[i].phase == "G0"
        if typeof(cells[i]) == paneth && cells[i].timeinstate > 1.5
            cells[i] = convert2stem(cells[i],deff)
        elseif typeof(cells[i]) == paneth
            return nothing
        elseif cells[i].timeinstate > 4/24
            cells[i] = convert2stem(cells[i],deff)
        end
    elseif cells[i].fate == "secprog" && cells[i].phase == "G0" && cells[i].timeinstate > 4/24
        cells[i] = convert2secprog(cells[i],secdeff)
    elseif cells[i].fate == "absprog" && cells[i].phase == "G0" && cells[i].timeinstate > 4/24
        cells[i] = convert2absprog(cells[i],deff)
    elseif cells[i].fate == "paneth" && cells[i].phase == "G0" && cells[i].timeinstate > 4/24
        cells[i] = convert2paneth(cells[i])
    elseif cells[i].fate == "goblet" && cells[i].phase == "G0" && cells[i].timeinstate > 4/24
        cells[i] = convert2goblet(cells[i])
    elseif cells[i].fate == "entendo" && cells[i].phase == "G0" && cells[i].timeinstate > 4/24
        cells[i] = convert2entendo(cells[i])
    elseif cells[i].fate == "entero" && cells[i].phase == "G0"
        cells[i] = convert2entero(cells[i],enterodeff)
    end
    return nothing
end

function villusfate(a,BMP,top,zent)
    if a.fate == "undecided" && a.phase != "arrest"
        fatedecider(a,BMP,top,zent)
    end

    return nothing
end

function setgrowrate(a::paneth)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/2
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif a.cellrad < a.rstar
        rate = log(a.rstar/a.birthrad)* a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::secprog)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/(4/24)
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif a.cellrad < a.rstar
        rate = log(cbrt(2)*a.rstar/a.birthrad)/(a.splittime) * a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::goblet)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/(4/24)
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif a.cellrad < a.rstar
        rate = log(cbrt(2)*a.rstar/a.birthrad)/(a.splittime) * a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::entendo)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/(4/24)
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif a.cellrad < a.rstar
        rate = log(cbrt(2)*a.rstar/a.birthrad)/(a.splittime) * a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::entero)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/(4/24)
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::stem)
    if a.phase == "G0"
        rate = 0.0
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif cbrt(2)*a.rstar > a.birthrad
        rate = log(cbrt(2)*a.rstar/a.birthrad)/(a.splittime) * a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function setgrowrate(a::absprog)
    if a.phase == "G0" && a.fate == "stem" && a.cellrad > 0.35
        rate = (0.35 - 0.55)/(4/24)
    elseif a.phase == "G0"
        rate = 0.0
    elseif a.phase == "dying"
        rate = - a.rstar/apoptime
    elseif a.phase == "arrest"
        rate = 0.0
    elseif cbrt(2)*a.rstar > a.birthrad
        rate = log(cbrt(2)*a.rstar/a.birthrad)/(a.splittime) * a.cellrad
    else
        rate = 0.0
    end

    a.growrate = rate
    return nothing
end

function growcells(a::cell,scale)
    if a.phase == "dying"
        a.cellrad += a.growrate*dt/scale
    else
        pee = 2
        a.cellrad += 2*(a.cyclevec[17]^pee/(a.cyclevec[17]^pee + 1))*a.growrate*dt/scale
    end
    if a.cellrad < 0
        a.cellrad = 0
    end
    return nothing
end

function getposse(a::cell)
    if a.pos[3] > .0
        cyl1 = convertcart2cyl(a.pos)
        deltaang = rand(Uniform(0,2*pi))
        delta2D = @SVector [cbrt(1/2)*a.cellrad * cos(deltaang),cbrt(1/2)*a.cellrad * sin(deltaang)]
        cyl2 = cyl1 - delta2D
        cyl1 += delta2D
        pos1 = convertcyl2cart(cyl1)
        pos2 = convertcyl2cart(cyl2)
    else
        sphere1 = rad * convertcart2spher(a.pos)
        deltaang = rand(Uniform(0,2*pi))
        delta2D = @SVector [cbrt(1/2)*a.cellrad * cos(deltaang),cbrt(1/2)*a.cellrad * sin(deltaang)]
        sphere2 = sphere1 - delta2D
        sphere1 += delta2D
        pos1 = convertspher2cart(sphere1/rad)
        pos2 = convertspher2cart(sphere2/rad)
    end
    return pos1, pos2
end

function birthcells(cells,deff)
    for i in eachindex(cells)
        if cells[i].phase == "splitting"
            pos1, pos2 = getposse(cells[i])

            cells[i].pos = pos1
            cells[i].phase = "G1"
            cells[i].fate = "undecided"
            cells[i].timeinstate = 0.0
            cells[i].timesincedivide = 0.0
            cells[i].boundwnt /= 2
            cells[i].birthrad = cbrt(1/2) * cells[i].cellrad
            cells[i].cellrad = cbrt(1/2) * cells[i].cellrad
            cells[i].talive = 0.0
            cells[i].status = "dunno"
            newcyclevec = @SVector [cells[i].cyclevec[1],cells[i].cyclevec[2],cells[i].cyclevec[3],cells[i].cyclevec[4],cells[i].cyclevec[5],cells[i].cyclevec[6],cells[i].cyclevec[7],cells[i].cyclevec[8],cells[i].cyclevec[9],cells[i].cyclevec[10],cells[i].cyclevec[11],cells[i].cyclevec[12],cells[i].cyclevec[13],cells[i].cyclevec[14],cells[i].cyclevec[16],.0,cells[i].cyclevec[17]]
            cells[i].cyclevec = @SVector [cells[i].cyclevec[1],cells[i].cyclevec[2],cells[i].cyclevec[3],cells[i].cyclevec[4],cells[i].cyclevec[5],cells[i].cyclevec[6],cells[i].cyclevec[7],cells[i].cyclevec[8],cells[i].cyclevec[9],cells[i].cyclevec[10],cells[i].cyclevec[11],cells[i].cyclevec[12],cells[i].cyclevec[13],cells[i].cyclevec[14],cells[i].cyclevec[15],.0,cells[i].cyclevec[17]]
            cells[i].ID = cells[i].ID .- 1
            if typeof(cells[i]) == stem
                rstar1 = rand(Distributions.Normal(0.35,0.35/stdev))
                rstar2 = rand(Distributions.Normal(0.35,0.35/stdev))

                if cbrt(2)*rstar1 <= cells[i].birthrad
                    cells[i].rstar = cbrt(1/2)*cells[i].birthrad
                else
                    cells[i].rstar = rstar1
                end

                if cbrt(2)*rstar2 <= cells[i].birthrad
                    rstar2 = cbrt(1/2)*cells[i].birthrad
                end

                push!(cells, stem(pos = pos2, rstar = rstar2, cellrad = cells[i].cellrad, birthrad = cells[i].cellrad, boundwnt = cells[i].boundwnt, splittime = cells[i].splittime, cyclevec = newcyclevec, notch = cells[i].notch, ID = cells[i].ID))
            end
            if typeof(cells[i]) == absprog
                rstar1 = rand(Distributions.Normal(0.5,0.5/stdev))
                rstar2 = rand(Distributions.Normal(0.5,0.5/stdev))

                if cbrt(2)*rstar1 <= cells[i].birthrad
                    cells[i].rstar = cbrt(1/2)*cells[i].birthrad
                else
                    cells[i].rstar = rstar1
                end

                if cbrt(2)*rstar2 <= cells[i].birthrad
                    rstar2 = cbrt(1/2)*cells[i].birthrad
                end

                cells[i].nsplits += 1
                push!(cells, absprog(pos = pos2, rstar = rstar2, cellrad = cells[i].cellrad, birthrad = cells[i].cellrad, boundwnt = cells[i].boundwnt, splittime = cells[i].splittime, cyclevec = newcyclevec, notch = cells[i].notch, ID = cells[i].ID, nsplits = cells[i].nsplits))
            end

            projectback(cells[i])
            projectback(cells[end])
        end
    end
    return nothing
end

function villusbirthcells(cells,deff)
    for i in eachindex(cells)
        if cells[i].phase == "splitting"
            deltas = @SVector [0.1*randn(),0.1*randn(),0.1*randn()]
            deltas = cbrt(1/2)*cells[i].cellrad/mag(deltas)*deltas

            cells[i].pos += deltas
            cells[i].phase = "G1"
            cells[i].fate = "undecided"
            cells[i].timeinstate = 0
            cells[i].timesincedivide = 0.0
            cells[i].boundwnt = cells[i].boundwnt/2
            cells[i].birthrad = cbrt(1/2) * cells[i].cellrad
            cells[i].cellrad = cbrt(1/2) * cells[i].cellrad

            if typeof(cells[i]) == stem
                cells[i].rstar = rand(Distributions.Normal(0.35,0.35/stdev))
                pushfirst!(cells, stem(pos = cells[i].pos - deltas, cellrad = cells[i].cellrad, birthrad = cells[i].cellrad, boundwnt = cells[i].boundwnt, splittime = cells[i].splittime, cyclevec = cells[i].cyclevec, notch = cells[i].notch, ID = cells[i].ID))
            end
            if typeof(cells[i]) == absprog
                cells[i].rstar = rand(Distributions.Normal(0.5,0.5/stdev))
                cells[i].nsplits += 1
                pushfirst!(cells, absprog(pos = cells[i].pos - deltas, cellrad = cells[i].cellrad, birthrad = cells[i].cellrad, boundwnt = cells[i].boundwnt, splittime = cells[i].splittime, cyclevec = cells[i].cyclevec, notch = cells[i].notch, ID = cells[i].ID, nsplits = cells[i].nsplits))
            end
        end
    end
    return nothing
end

function begindeathshrink(a::cell)
    if a.talive > a.lifespan && a.phase != "dying"
        a.rstar = a.cellrad
        a.phase = "dying"
        a.cyclevec = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,a.cyclevec[15],a.cyclevec[16],a.cyclevec[17]]
    end
    return nothing
end

function popcell(a::cell)
    if (a.cellrad/a.rstar)^3 > 4 && a.phase != "arrest" && a.phase != "dying"
        a.phase = "arrest"
    end
    return nothing
end

function sig(restlen,dist,a,b)
    if (dist - restlen) <= Notchrange
        if b.notchemit == true
            a.notch += knotch * dt
            b.notchgiving += 1
        end
        if a.notchemit == true
            b.notch += knotch * dt
            a.notchgiving += 1
        end
    end

    if (dist - restlen) <= Wntrange
        if b.wntemit == true
            a.boundwnt += kwnt * dt
        end
        if a.wntemit == true
            b.boundwnt += kwnt * dt
        end
    end

    return nothing
end

function emergencynotchscan(a,cells,grid,ind)
    if a.notchemit == true && a.notchgiving <= 3
        gridid = CartesianIndex(Int(fld(a.pos[1],spacing)),Int(fld(a.pos[2],spacing)),Int(fld(a.pos[3],spacing)))

        for i in grid[gridid]
            givenotch(a,cells[i])
        end

        for j in bigoffsets
            if haskey(grid, gridid+j)
                for i in grid[gridid+j]
                    givenotch(a,cells[i])
                end
            end
        end
    end
    a.notchgiving = 0
    return nothing
end

function givenotch(a,b)
    checkme = mag(b.pos - a.pos) - (a.cellrad + b.cellrad)
    if checkme > Notchrange
        b.notch += knotch * dt
    end
    return nothing
end

function nichewntacc(a,numniche)
    if a.pos[3] < .0
        a.boundwnt += kwnt * nummy * dt / numniche
    end

    if a.boundwnt > 128
        a.boundwnt = 128
    end
    return nothing
end

function emergencyfatedecisions(a::stem,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.fate == "absprog" || a.fate == "secprog"
        if a.boundwnt > wntbound
            a.fate = "divide"
            a.phase = "S"
            if a.cyclevec[16] <= .0
                a.cyclevec = @SVector [a.cyclevec[1],a.cyclevec[2],a.cyclevec[3],a.cyclevec[4],a.cyclevec[5],a.cyclevec[6],a.cyclevec[7],a.cyclevec[8],a.cyclevec[9],a.cyclevec[10],a.cyclevec[11],a.cyclevec[12],a.cyclevec[13],a.cyclevec[14],a.cyclevec[15]/2,a.cyclevec[15]/2,a.cyclevec[17]]
            end
            a.timeinstate = 0
        end
    elseif a.fate == "paneth"
        if a.boundwnt < wntbound || a.notch >= 0.99*pannynotchold
            fatedecider(a,BMP,top,zent)
        end
    end
    return nothing
end

function emergencyfatedecisions(a::absprog,BMP,top,zent)
    return nothing
end

function emergencyfatedecisions(a::secprog,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.boundwnt > wntbound && a.fate != "stem"
        a.fate = "stem"
        a.phase = "G0"
        a.timeinstate = 0.0
    end

    return nothing
end

function emergencyfatedecisions(a::paneth,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.fate == "undecided"
        fatedecider(a,BMP,top,zent)
    elseif a.fate == "wntdiffprep" && a.boundwnt >= wntbound
        a.phase = "G0"
        a.fate = "undecided"
    elseif a.fate == "notchdiffprep" && a.notch < overnotchold
        a.phase = "G0"
        a.fate = "undecided"
    elseif a.fate == "wntdiffprep" || a.fate == "notchdiffprep"
        if a.timeinstate >= 12/24
            a.fate = "stem"
        end
    end
    return nothing
end

function emergencyfatedecisions(a::goblet,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.fate == "undecided"
        fatedecider(a,BMP,top,zent)
    end
    return nothing
end

function emergencyfatedecisions(a::entendo,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.fate == "undecided"
        fatedecider(a,BMP,top,zent)
    end
    return nothing
end

function emergencyfatedecisions(a::entero,BMP,top,zent)
    if a.phase == "dying"
        return nothing
    elseif a.fate == "undecided"
        fatedecider(a,BMP,top,zent)
    end
    return nothing
end

function setsplittime(a::cell)
    OGsplit = a.splittime
    NEWsplit = 1*(dee/(1 + exp(2*(-a.pressure + p0))) + absdiv)/24
    ratio = NEWsplit/OGsplit

    a.splittime = NEWsplit
    a.timeinstate = ratio*a.timeinstate
    return nothing
end

function znrf3calc(cells)
    stemz = filter(x->typeof(x)==stem && x.boundwnt>=64.0 && x.phase != "dying",cells)
    znrf3vec = Vector{SVector{3,Float64}}(undef,length(stemz))
    for i in eachindex(stemz)
        znrf3vec[i] = stemz[i].pos
    end
    return znrf3vec
end

function znrf3field(a,znrf3vec)
    stronk = .0
    for b in znrf3vec
        stronk += feld(a.pos,b)::Float64
    end

    return stronk
end

function feld(a,b)
    return exp(-(mag(b-a)::Float64))
end

function notchdecay(a)
    a.notch -= a.notch * knotch * dt

    return nothing
end

function wntdecay(a,znrf3vec)
    sigstronk::Float64 = znrf3field(a,znrf3vec)::Float64

    dwnt = 2*dzero/((stemmylim/sigstronk)^cue + 1)

    a.boundwnt -= dwnt * a.boundwnt * dt

    return nothing
end

function Bee(A,B,C,D)
    return B - A + C * B + D * A
end

function Gee(A,B,C,D)
    if A == 0.0
        return 0.0
    else
        ans = (2 * D * A) / (Bee(A,B,C,D) + sqrt((Bee(A,B,C,D))^2 - 4 * (B - A) * D * A))
        return ans
    end
end

function Michaelis(M,J,k,S)
    return (k*S*M)/(J+S)
end

function setphase(a,initX1,initX2,initX3,prevWee1,BMP,top,zent)
    OGphase = a.phase
    OGfate = a.fate

    if a.cyclevec[3] >= 0.55 && a.fate == "undecided" && a.cyclevec[15] >= DNAdamagethreshold && a.cyclevec[17] >= RNAdamagethreshold
        fatedecider(a,BMP,top,zent)
    elseif a.cyclevec[3] >= 0.55 && a.fate == "undecided"
        a.timeinstate = 0
        a.rstar = a.cellrad
        a.phase = "dying"
    end

    if prevWee1 > 0.6 && a.cyclevec[10] <= 0.6 && a.phase != "M"
        a.timeinstate = 0
        a.phase = "G2"
    end

    if initX1 > 0.99 && a.cyclevec[1] <= 0.99 && a.cyclevec[15] >= DNAdamagethreshold && a.cyclevec[16] >= DNAdamagethreshold && a.cyclevec[17] >= RNAdamagethreshold
        a.timeinstate = 0
        a.phase = "M"
    elseif initX1 > 0.99 && a.cyclevec[1] <= 0.99
        a.timeinstate = 0
        a.rstar = a.cellrad
        a.phase = "dying"
    end

    if initX2 > 0.2 && a.cyclevec[2] <= 0.2 && a.status == "ready"
        a.timeinstate = 0
        a.phase = "splitting"
    elseif initX2 > 0.2 && a.cyclevec[2] <= 0.2
        a.timeinstate = 0
        a.rstar = a.cellrad
        a.phase = "dying"
    end

    return nothing
end


function mitosiscomplete(a::cell)
    if a.phase == "M" && a.timeinstate > 0.5/24
        a.status = "ready"
    end
    return nothing
end

function BMPcalc(enteronum,nent,p)
    return 128/((nent/enteronum)^p + 1)
end

function topcalc(enteronum,mintop,nent)
    return max(1.2 * mintop / ((1.2 - 1)*(enteronum/nent)^3.5 + 1),mintop)
end

function getvsifac(splut)
    Vsifac = (splut - absdiv/24)*(Vsifacslow-Vsifacfast)/(stemdiv/24-absdiv/24) + Vsifacfast
    if Vsifac > Vsifacslow
        Vsifac = Vsifacslow
    elseif Vsifac < Vsifacfast
        Vsifac = Vsifacfast
    end
    return Vsifac
end

function getkaweep(splut)
    kawee_p = (splut - absdiv/24)*(kawee_pslow-kawee_pfast)/(stemdiv/24-absdiv/24) + kawee_pfast
    if kawee_p > kawee_pfast
        kawee_p = kawee_pfast
    elseif kawee_p < kawee_pslow
        kawee_p = kawee_pslow
    end
    return kawee_p
end

function scaleyconsts(splut)
    kawee_p = getkaweep(splut)

    Vsifac = getvsifac(splut)

    periodchange = 140.02683333333334 / splut

    return kawee_p, Vsifac, periodchange
end

function scaleymass(splut)
    massconst = (splut - absdiv/24)*(massconstslow - massconstfast)/(stemdiv/24-absdiv/24) + massconstfast
    if massconst > massconstslow
        massconst = massconstslow
    elseif massconst < massconstfast
        massconst = massconstfast
    end

    return massconst
end

function cellwiper(cells,celltype,prob)
    for i in eachindex(cells)
        if typeof(cells[i]) == celltype
            if rand(Distributions.Bernoulli(prob))
                cells[i].phase = "dying"
            end
        end
    end
    return nothing
end

function cellcycle5FU(a::cell,BMP,top,drug,zent)
    if a.phase == "arrest" || a.phase == "dying" || a.phase == "popping"
        setgrowrate(a)
        growcells(a,1)
        return nothing
    end

    kawee_p,Vsifac,periodchange = scaleyconsts(a.splittime)

    for j in 1:cyclescale
        mass = scaleymass(a.splittime) * (a.cellrad/a.rstar)^3
        dX = getdX(a.cyclevec,kawee_p,Vsifac,mass,periodchange,a.phase)
        drugeff = fiveFUdrugeff(a.cyclevec,drug)

        initX1 = a.cyclevec[1]
        initX2 = a.cyclevec[2]
        initX3 = a.cyclevec[3]
        prevWee1 = a.cyclevec[10]

        a.cyclevec += (dX + drugeff) * (dt/cyclescale)

        setgrowrate(a)
        growcells(a,cyclescale)

        a.cyclevec = correctovershoots(a.cyclevec)

        setphase(a,initX1,initX2,initX3,prevWee1,BMP,top,zent)
        if a.phase == "splitting" || a.phase == "dying" || a.phase == "arrest"
            break
        end
    end
    return nothing
end

function cellcycleCDK(a::cell,BMP,top,drug,zent)
    if a.phase == "arrest" || a.phase == "dying" || a.phase == "popping"
        setgrowrate(a)
        growcells(a,1)
        return nothing
    end

    kawee_p,Vsifac,periodchange = scaleyconsts(a.splittime)

    for j in 1:cyclescale
        mass = scaleymass(a.splittime) * (a.cellrad/a.rstar)^3
        dX = getdX(a.cyclevec,kawee_p,Vsifac,mass,periodchange,a.phase)

        drugeff = cdkdrugeff(a.cyclevec,drug)

        initX1 = a.cyclevec[1]
        initX2 = a.cyclevec[2]
        initX3 = a.cyclevec[3]
        prevWee1 = a.cyclevec[10]

        a.cyclevec += (dX + drugeff) * (dt/cyclescale)

        setgrowrate(a)
        growcells(a,cyclescale)

        a.cyclevec = correctovershoots(a.cyclevec)

        setphase(a,initX1,initX2,initX3,prevWee1,BMP,top,zent)
        if a.phase == "splitting" || a.phase == "dying" || a.phase == "arrest"
            break
        end
    end
    return nothing
end

function cellcycle(a::cell,BMP,top,zent)
    if a.phase == "arrest" || a.phase == "dying" || a.phase == "popping"
        setgrowrate(a)
        growcells(a,1)
        return nothing
    end

    kawee_p,Vsifac,periodchange = scaleyconsts(a.splittime)

    for j in 1:cyclescale
        mass = scaleymass(a.splittime) * (a.cellrad/a.rstar)^3
        dX = getdX(a.cyclevec,kawee_p,Vsifac,mass,periodchange,a.phase)

        initX1 = a.cyclevec[1]
        initX2 = a.cyclevec[2]
        initX3 = a.cyclevec[3]
        prevWee1 = a.cyclevec[10]

        a.cyclevec += dX * (dt/cyclescale)

        setgrowrate(a)
        growcells(a,cyclescale)

        a.cyclevec = correctovershoots(a.cyclevec)

        setphase(a,initX1,initX2,initX3,prevWee1,BMP,top,zent)
        if a.phase == "splitting" || a.phase == "dying" || a.phase == "arrest"
            break
        end
    end
    return nothing
end

function getdX(X,kawee_p,Vsifac,mass,periodchange,phase)
    #CycD
    CycD0=0.05

    #CycE
    kde_p=0.01
    kdea_pp=0.5
    kdeb_pp=0.5
    kdee_pp=0.1
    kasse = 50.0
    kdisse=1.0
    kse_p = 0.01
    kse_pp = 0.18

    #TFE
    katf_p=0.0
    katfa_pp=0.2
    katfd_pp=3.0
    katfe_pp=0.5
    kitf_p=0.25
    kitfa_pp=0.1
    kitfb_pp=0.1
    Jatf=0.01
    Jitf=0.01

    #CycB (not complexed with p27)
    ksb_p=0.01
    ksb_pp=0.03
    kdb_p=0.0050
    kdbh_pp=2.0
    kdbc_pp=0.1
    kassb=0.0
    kdissb=0.0
    kwee_p=0.02
    kwee_pp=0.2
    k25_p=0.01
    k25_pp=5

    #TFB=MCM
    kafb=1.0
    kifb=0.1
    Jafb=0.1
    Jifb=0.1

    #CycA
    ksa_p=0.0
    ksa_pp=0.025
    kda_p=0.020
    kda_pp=2.0
    kda_ppp=0.0
    kassa=25.0
    kdissa=1.0

    #CKI=p27
    ksi_p=1.8
    ksi_pp=0.0
    kdia_pp=5.0
    kdib_pp=5.0
    kdid_pp=0.0
    kdie_pp=5.0
    k14di=0.0

    #TFI=none
    kafi=88.0
    kifi_p=88.0
    kifib_pp=88.0
    Jafi=88.0
    Jifi=88.0

    #APC activation
    kaie=0.07
    kiie=0.18
    Jaie=0.01
    Jiie=0.01

    #Cdc20
    ks20_p=0.0
    ks20_pp=0.15
    n20=1.0
    J20=1
    kd20=0.15
    ka20=0.5
    ki20=0.25
    Ja20=0.0050
    Ji20=0.0050

    #Cdh1
    kah1_p=0.18
    kah1_pp=3.5
    kih1_p=0.0
    kih1a_pp=0.2
    kih1b_pp=1.0
    kih1d_pp=0.0
    kih1e_pp=0.1
    Jah1=0.01
    Jih1=0.01

    #Wee1-Cdc25
    kawee_pp=0
    kiwee_p=0.0
    kiwee_pp= 1.0
    Jawee=0.05
    Jiwee=0.05
    ka25_p=0
    ka25_pp=1
    ki25_p=0.3
    ki25_pp=0
    Ja25=0.1
    Ji25=0.1

    APCT=1.0
    Cdh1T=1.0

    # some functions to shorten ODES
    preMPF=X[11]+X[12]

    TriB=X[5]+X[12]

    CycBT=X[2]+X[11]+X[5]+X[12]

    CycAT=X[1]+X[13]

    CycET=X[3]+X[14]

    CKIT=X[9]+X[5]+X[12]+X[13]+X[14]

    Cdc20T=X[7]+X[6]

    Cdc14=X[6]

    Cdc25=Gee(ka25_p+ka25_pp*X[2],ki25_p+ki25_pp*Cdc14,Ja25,Ji25)

    V25=k25_p+k25_pp*Cdc25

    TFB=Gee(kafb*X[2],kifb,Jafb,Jifb)

    CycD=CycD0*mass

    Vatf=Vatffac*(katf_p+katfa_pp*X[1]+katfe_pp*X[3]+katfd_pp*CycD)
    Vitf=kitf_p+kitfa_pp*X[1]+kitfb_pp*X[2]

    TFE=Gee(Vatf,Vitf,Jatf,Jitf)
    TFI=Gee(kafi*Cdc14,kifi_p+kifib_pp*X[2],Jafi,Jifi)

    tweaky = min(1,X[15]+X[16])

    kay = 5
    powah = 3
    Vsi = (ksi_p+ksi_pp*TFI) * kay/ (1 + (kay/Vsifac-1)*tweaky^powah)

    if phase == "G0"
        Vsa= 0.0
        Vsb= 0.0
        Vse= 0.0
        Vah1= 0.0
        ks20_p= 0.0
        ks20_pp= 0.0
        Wee1 = 1.0
    else
        Vsa=(ksa_p+ksa_pp*TFE)*mass
        Vsb=(ksb_p+ksb_pp*TFB)*mass
        Vse=(kse_p+kse_pp*TFE)*mass
        Vah1=(kah1_p+kah1_pp*Cdc14)
        Wee1=Gee(kawee_p+kawee_pp*Cdc14,kiwee_p+kiwee_pp*X[2],Jawee,Jiwee)
    end

    ks20_p=0.0
    ks20_pp=0.15

    Vwee=kwee_p+kwee_pp*Wee1

    Vih1=kih1_p+kih1a_pp*X[1]+kih1b_pp*X[2]+kih1e_pp*X[3]+kih1d_pp*CycD
    Vdb=kdb_p+kdbh_pp*X[8]+kdbc_pp*X[6]
    Vda=kda_p+(kda_pp+kda_ppp)*X[6]+kda_ppp*X[7]
    Vde=kde_p+kdee_pp*X[3]+kdea_pp*X[1]+kdeb_pp*X[2]
    Vdi=(kdi_p+kdia_pp*X[1]+kdib_pp*X[2]+kdie_pp*X[3]+kdid_pp*CycD)/(1.0+k14di*Cdc14)

    # Velocities and Dependent Variables
    APC=(-(X[4]-APCT)/1)
    Cdh1i=(-(X[8]-Cdh1T)/1)

    # DNA and drug
    if X[15] > 0 && X[15] < 1
        if phase == "S"
            kDNA1 = 16/sphasedur
        else
            kDNA1 = 8/sphasedur
        end
    else
        kDNA1 = 0.0
    end

    if X[16] > 0 && X[16] < 1
        if phase == "S"
            kDNA2 = 16/sphasedur
        else
            kDNA2 = 8/sphasedur
        end
    else
        kDNA2 = 0.0
    end

    if X[17] > 0 && X[17] < 1
        kRNA = 5
    else
        kRNA = 0
    end

    # ODEs
    dX1 = - (kassa*X[9]*X[1]) + (kdissa*X[13]) + (Vdi*X[13]) + (Vsa) - (Vda*X[1])

    dX2 = (Vsb) - (Vdb*X[2]) + (V25*X[11]) - (Vwee*X[2]) - (kassb*X[2]*X[9]) + (kdissb*X[5]) + (Vdi*X[5])

    dX3 = - (kasse*X[9]*X[3]) + (kdisse*X[14]) + (Vdi*X[14]) + (Vse) - (Vde*X[3])

    dX4 = (Michaelis(X[2],Jaie,kaie,APC)) - (Michaelis(1.0,Jiie,kiie,X[4]))

    dX5 = (kassb*X[2]*X[9]) - (kdissb*X[5]) + (V25*X[12]) - (Vwee*X[5]) - (Vdb*X[5]) - (Vdi*X[5])

    dX6 = (Michaelis(X[4],Ja20,ka20,X[7])) - (Michaelis(1.0,Ji20,ki20,X[6])) - (kd20*X[6])

    dX7 = (ks20_p+ks20_pp*X[2]^n20/(J20^n20+X[2]^n20)) - (kd20*X[7]) - (Michaelis(X[4],Ja20,ka20,X[7])) + (Michaelis(1.0,Ji20,ki20,X[6]))

    dX8 = (Michaelis(Vah1,Jah1,1.0,Cdh1i)) - (Michaelis(Vih1,Jih1,1.0,X[8]))

    dX9 = - (kassb*X[2]*X[9]) + (kdissb*X[5]) - (kassb*X[11]*X[9]) + (kdissb*X[12]) + (Vdb*X[5]) + (Vdb*X[12]) + (Vsi) - (Vdi*X[9]) - (kassa*X[9]*X[1]) + (kdissa*X[13]) + (Vda*X[13]) - (kasse*X[9]*X[3]) + (kdisse*X[14]) + (Vde*X[14])

    dX11 = - (V25*X[11]) + (Vwee*X[2]) - (Vdb*X[11]) - (kassb*X[11]*X[9]) + (kdissb*X[12]) + (Vdi*X[12])

    dX12 = (kassb*X[11]*X[9]) - (kdissb*X[12]) - (V25*X[12]) + (Vwee*X[5]) - (Vdb*X[12]) - (Vdi*X[12])

    dX13 = (kassa*X[9]*X[1]) - (kdissa*X[13]) - (Vdi*X[13]) - (Vda*X[13])

    dX14 = (kasse*X[9]*X[3]) - (kdisse*X[14]) - (Vdi*X[14]) - (Vde*X[14])

    dX10 = (Wee1 - X[10]) / ((dt/cyclescale))

    # DNA1
    dX15 = kDNA1
    # DNA2
    dX16 = kDNA2
    # RNA
    dX17 = kRNA

    dX = @SVector[periodchange*dX1, periodchange*dX2, periodchange*dX3, periodchange*dX4, periodchange*dX5,periodchange*dX6, periodchange*dX7, periodchange*dX8, periodchange*dX9, dX10, periodchange*dX11,periodchange*dX12, periodchange*dX13, periodchange*dX14, dX15, dX16, dX17]
    return dX
end

function correctovershoots(X)
    return @SVector[cizzle(X[1]),cizzle(X[2]),cizzle(X[3]),cizzle(X[4]),cizzle(X[5]),cizzle(X[6]),cizzle(X[7]),cizzle(X[8]),cizzle(X[9]),pcizzle(X[10]),cizzle(X[11]),cizzle(X[12]),cizzle(X[13]),cizzle(X[14]),pcizzle(X[15]),pcizzle(X[16]),pcizzle(X[17])]
end

function cizzle(ex)
    if ex < .0
        return .0
    else
        return ex
    end
end

function pcizzle(ex)
    if ex < .0
        return .0
    elseif ex > 1.0
        return 1.0
    else
        return ex
    end
end

function convertcart2spher(vec)
    theta = acos(vec[3]/rad)
    phi = atan(vec[2],vec[1])
    return SVector(theta, phi)
end

function convertcart2cyl(vec)
    X = rad * atan(vec[2],vec[1])
    Y = vec[3]
    return SVector(X, Y)
end

function convertspher2cart(angs)
    x = rad * sin(angs[1]) * cos(angs[2])
    y = rad * sin(angs[1]) * sin(angs[2])
    z =  rad * cos(angs[1])
    return SVector(x,y,z)
end

function convertcyl2cart(vec)
    phi = vec[1] /rad
    x = rad * cos(phi)
    y = rad * sin(phi)
    z = vec[2]
    return SVector(x,y,z)
end

function dot(a,b)
    dot = 0.0
    if length(a) == length(b)
        for i in 1:length(a)
            dot += a[i] * b[i]
        end
    end
    return dot
end

function mag(vec)
    return sqrt(dot(vec,vec))
end

function findsigma(a::cell,b::cell)
    anga = convertcart2spher(a.pos)
    angb = convertcart2spher(b.pos)

    num = sqrt(abs2(cos(pi/2 - angb[1]) * sin(abs(angb[2] - anga[2]))) + abs2(cos(pi/2 - anga[1]) * sin(pi/2 - angb[1]) - sin(pi/2 - anga[1]) * cos(pi/2 - angb[1]) * cos(abs(angb[2] - anga[2]))))
    denom = sin(pi/2 - anga[1]) * sin(pi/2 - angb[1]) + cos(pi/2 - anga[1]) * cos(pi/2 - angb[1]) * cos(abs(angb[2] - anga[2]))
    sigma = atan(num,denom)

    return sigma
end

function sphergeotan(a::cell,b::cell, sigma::Float64)
    dirF = (-sigma / tan(sigma)) * a.pos + (sigma / sin(sigma)) * b.pos
    unitdirF = dirF / mag(dirF)
    return unitdirF
end

function projectback(a::cell)
    if a.pos[3] < 0
        a.pos = rad/mag(a.pos) * a.pos
    elseif a.pos[3] >= 0
        proj = @SVector [a.pos[1], a.pos[2]]
        a.pos = @SVector [rad/mag(proj) * proj[1], rad/mag(proj) * proj[2], a.pos[3]]
    end
    return nothing
end

function makegrid(cells)
    grid = Dict{CartesianIndex{3}, Vector{Int}}()
    @inbounds for i in eachindex(cells)
        fillgrid(grid,cells[i],i)
    end
    return grid
end

function fillgrid(grid,a,i)
    gridid = getgridid(a)
    if haskey(grid, gridid)
        push!(grid[gridid], i)
    else
        grid[gridid] = [i]
    end
end

function getgridid(a)
    x::Int = floor(Int,a.pos[1]/spacing)
    y::Int = floor(Int,a.pos[2]/spacing)
    z::Int = floor(Int,a.pos[3]/spacing)
    gridid = CartesianIndex(x,y,z)

    return gridid
end

function heaviside(t)
   return 0.5 * (sign(t) + 1)
end

function interval(t, a, b)
   return heaviside(t-a) - heaviside(t-b)
end

function forces(restlen,dist,diff,a,b)
    tanveca = diff / dist
    tanvecb = - diff / dist

    Estar = 1 / ((1-a.pratio^2)/a.youngmod + (1-b.pratio^2)/b.youngmod)
    Rstar = 1 / (1/a.cellrad + 1/b.cellrad)

    if dist < restlen
        x = (a.cellrad^2 - b.cellrad^2 + dist^2)/(2 * dist)

        fricA = friction(x,a,b)
        fricB = friction(x,b,a)

        if typeof(a) == paneth && typeof(b) == paneth
            epsilon = 0.111
        else
            epsilon = 0.0111
        end

        d = restlen - dist
        F = - (4/3) * Estar * sqrt(Rstar) * sqrt(d)^3
        F += area(dist,x,epsilon)
    else
        F = 0.0
        fricA = @SVector [0.0,0.0,0.0]
        fricB = @SVector [0.0,0.0,0.0]
    end

    a.pressure += - F / (pi * abs2(a.cellrad))
    b.pressure += - F / (pi * abs2(b.cellrad))

    a.force += F * tanveca + fricA
    b.force += F * tanvecb + fricB

    return nothing
end

function area(dist,x,epsilon)
    F = 2 * epsilon * pi * x * (1 - x/dist)
    return F
end

function friction(x,a,b)
    numu = (1/10000) * pi * (a.cellrad^2 - x^2)

    F = - numu * (a.vel - b.vel)

    return F
end

function movecells(a::cell)
    mu = (0.5 + pi * abs2(a.cellrad))/1500

    a.vel = a.force/mu

    a.pos += a.vel * dt

    projectback(a)

    a.force = @SVector [0.0,0.0,0.0]
    a.pressure = .0
    return nothing
end

function movecells(a::stem)
    mu = (0.5 + pi * abs2(a.cellrad))/1500

    if a.fate == "paneth"
        aincrease = (1000000-1)*a.timeinstate / (4/24) + 1
    else
        aincrease = 1.0
    end

    mu *= aincrease

    a.vel = a.force/mu

    a.pos += a.vel * dt

    projectback(a)

    a.force = @SVector [0.0,0.0,0.0]
    a.pressure = .0
    return nothing
end

function movecells(a::paneth)
    mu = (0.5 + pi * abs2(a.cellrad))/1500

    if a.fate == "stem"
        areduce = (1.5 - a.timeinstate) / 1.5
    else
        areduce = 1.0
    end

    mu *= areduce*100000

    a.vel = a.force/mu

    a.pos += a.vel * dt

    projectback(a)

    a.force = @SVector [0.0,0.0,0.0]
    a.pressure = .0
    return nothing
end

function scalepressure(a::cell)
    a.pressure /= (pi * abs2(a.cellrad))
    return nothing
end

function plotstuff(cells)
    positions = Vector{Point3f}(undef,length(cells))
    markers = Vector{Float64}(undef,length(cells))
    colours = Vector{Symbol}(undef,length(cells))

    for i in eachindex(cells)
        positions[i] = cells[i].pos
        colours[i] = colour(cells[i])
        markers[i] = cells[i].cellrad
    end

    return positions,markers,colours
end

function plot3d(cells = finalcells)
    positions,markers,colours = plotstuff(cells)
    figure = Figure()
    ax1 = Axis3(figure[1,1],aspect=:data,viewmode=:fit)
    meshscatter!(ax1,positions,markersize=markers,color=colours)
    hidedecorations!(ax1)
    hidespines!(ax1)
    return figure
end

function ki67plot3d(cells = finalcells)
    positions,markers,colours = plotstuff(cells)
    for i in eachindex(cells)
        if ki67positive(cells[i])
            colours[i] = :red
        elseif cells[i].phase == "dying"
            colours[i] = :brown4
        else
            colours[i] = :white
        end
    end
    figure = Figure()
    ax1 = Axis3(figure[1,1],aspect=:data,viewmode=:fit)
    meshscatter!(ax1,positions,markersize=markers,color=colours)
    hidedecorations!(ax1)
    hidespines!(ax1)
    return figure
end

function updates(cells,villus,cellcounts,villuscounts,fiveFUtrack,ki67,brdu,celldeets,inc,tmax,top; fiveFU = zeros(7))
    if inc % round(Int,tmax/(dt*(size(cellcounts,2)-1))) == 0
        updatedacounts(cells,villus,cellcounts,villuscounts,inc,tmax,top)
        fiveFUtrack[:,ceil(Int,inc/round(Int,tmax/(dt*(size(cellcounts,2)-1))) + 1)] = fiveFU[:]
    end

    if inc % 139 == 0
        stripscount(cells,ki67,brdu,celldeets,Int(inc/139))
    end

    return nothing
end

function brdupositive(a::cell)
    if a.ID[1] > 0
        return true
    else
        return false
    end
end

function ki67positive(a::cell)
    if a.talive <= 6/24
        return true
    elseif a.phase == "dying" && a.timeinstate <= 6/24
        return true
    elseif a.phase == "G0" && a.timeinstate <= 6/24
        return true
    elseif a.phase == "G1" 
        if a.splittime < 15/24 && a.timeinstate <= 8/24
            return true
        elseif a.splittime >= 15/24 && a.timeinstate <= 8/24
            return true
        end
    elseif a.phase == "S" && a.timeinstate <= 16/24
        return true
    elseif a.phase == "G2" 
        if a.splittime < 15/24 && a.timeinstate <= 16/24
            return true
        elseif a.splittime >= 15/24 && a.timeinstate <= 16/24
            return true
        end
    elseif a.phase == "M" && a.timeinstate <= 18/24
        return true
    end

    return false
end

function stripscount(cells,ki67,brdu,celldeets,indy)
    thetas = rand(Float64,500) .* 2*pi
    col = makecol(cells)
    for theta in thetas
        arrangecol(col,theta)
        enoomerate(col,ki67,brdu,celldeets,indy)
    end

    return nothing
end

function intercept(pos,cellrad,theta)
    dis = sqrt(0.5)
    a = 1
    b = - 2 * pos[1] * cos(theta) - 2 * pos[2] * sin(theta)
    c = pos[1]^2 + pos[2]^2 - (dis*cellrad)^2

    disc = b^2 - 4*a*c
    
    if disc < .0
        return false
    else
        return true
    end
end

function makecol(cells)
    col = Vector{infopack}(undef,length(cells))
    for i in eachindex(col)
        col[i] = infopack(ki67positive(cells[i]),brdupositive(cells[i]),gettypeind(cells[i]),cells[i].pos,cells[i].pos,cells[i].cellrad)
    end

    return col
end

function arrangecol(col,theta)
    for a in col
        if intercept(a.origpos,a.cellrad,theta)
            if a.origpos[1] >= .0
                a.pos = @SVector [1., .0, a.origpos[3]]
            else
                a.pos = @SVector [-1., .0, a.origpos[3]]
            end
        else
            a.pos = @SVector[length(col),.0,.0]
        end
    end

    sort!(col, by = v -> (v.pos[1],v.pos[3]))

    for i in eachindex(col)
        if i == 1
            col[i].pos = @SVector [col[i].pos[1],.0,1.]
        elseif col[i].pos[1] == col[i-1].pos[1]
            col[i].pos = @SVector [col[i].pos[1],.0,col[i-1].pos[3]+1]
        else
            col[i].pos = @SVector [col[i].pos[1],.0,1.]
        end
    end
end

function enoomerate(col,ki67,brdu,celldeets,indy)
    for a in col
        if abs(a.pos[1]) == 1
            index = round(Int,a.pos[3])
            
            ki67count(a,ki67,index,indy)

            brducount(a,brdu,index,indy)

            celldeets[a.type,index] += 1
        end
    end
end

function ki67count(a,ki67,index,indy)
    if a.ki67status
        ki67[1,index,indy] += 1
    end
    ki67[2,index,indy] += 1
end

function brducount(a,brdu,index,indy)
    if a.brdustatus
        brdu[1,index,indy] += 1
    end
    brdu[2,index,indy] += 1
end

function gettypeind(a)
    if typeof(a) == stem && a.boundwnt >= 64.0
        return 1
    elseif typeof(a) == stem
        return 2
    elseif typeof(a) == paneth
        return 3
    elseif typeof(a) == absprog
        return 4
    elseif typeof(a) == entero
        return 5
    elseif typeof(a) == secprog
        return 6
    elseif typeof(a) == goblet
        return 7
    elseif typeof(a) == entendo
        return 8
    end
end

function brdustain(cells,inc)
   for i in eachindex(cells)
        if cells[i].phase == "S"
            value = round(Int,min(max(8-24*cells[i].timeinstate,0),(2 - inc*dt*24))/2 * 8*5/7) # inner max is just to prevent negative values
            cells[i].ID = [max(value, cells[i].ID[1])] 
        end
    end
end

function updatedacounts(cells,villus,cellcounts,villuscounts,inc,tmax,top)
    index = ceil(Int,inc/round(Int,tmax/(dt*(size(cellcounts,2)-1))) + 1)
    cellcounts[1,index] = count(x->typeof(x)==stem && x.boundwnt >= 64.0 && x.phase != "arrest" && x.phase != "dying",cells)
    cellcounts[2,index] = count(x->typeof(x)==stem && x.boundwnt < 64.0 && x.phase != "arrest" && x.phase != "dying",cells)
    cellcounts[3,index] = count(x->typeof(x)==absprog && x.phase != "arrest" && x.phase != "dying",cells)
    cellcounts[4,index] = count(x->typeof(x)==entero && x.phase != "dying",cells)
    cellcounts[5,index] = count(x->typeof(x)==secprog && x.phase != "dying",cells)
    cellcounts[6,index] = count(x->typeof(x)==goblet && x.phase != "dying",cells)
    cellcounts[7,index] = count(x->typeof(x)==paneth && x.phase != "dying",cells)
    cellcounts[8,index] = count(x->typeof(x)==entendo && x.phase != "dying",cells)
    cellcounts[9,index] = count(x->x.phase == "arrest", cells)
    cellcounts[10,index] = count(x->x.phase == "dying", cells)
    cellcounts[11,index] = length(cells)

    villuscounts[1,index] = count(x->typeof(x)==stem && x.boundwnt > 63.9 && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[2,index] = count(x->typeof(x)==stem && x.boundwnt <= 63.9 && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[3,index] = count(x->typeof(x)==absprog && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[4,index] = count(x->typeof(x)==entero && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[5,index] = count(x->typeof(x)==secprog && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[6,index] = count(x->typeof(x)==goblet && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[7,index] = count(x->typeof(x)==paneth && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[8,index] = count(x->typeof(x)==entendo && x.phase != "arrest" && x.phase != "dying",villus)
    villuscounts[9,index] = count(x->x.phase == "arrest", villus)
    villuscounts[10,index] = count(x->x.phase == "dying", villus)
    villuscounts[11,index] = length(villus)
    return nothing
end

function countsplot(res = (900,1200), data = "on")
    fig = Figure(resolution = res)

    ax1 = fig[1, 1] = Axis(fig, xlabel = "Time / days", ylabel = "Number of cells")
    ax2 = fig[2, 1] = Axis(fig, xlabel = "Time / days", ylabel = "Number of cells")

    if length(doseschedule) > 0
        vlines!(ax1,doseschedule,color=:darkgrey,linewidth=:3,linestyle=:dot)
        vlines!(ax2,doseschedule,color=:darkgrey,linewidth=:3,linestyle=:dot)
    end

    colours = [:yellow,:yellow4,:orange,:red,:green,:purple,:blue,:turquoise,:darkgrey,:brown4,:black]
    for i in 1:length(cellcounts[:,1])
        lines!(ax1,tee,cellcounts[i,:],color=colours[i],linewidth=5)
        lines!(ax2,tee,villuscounts[i,:],color=colours[i],linewidth=5)
    end

    ylims!(ax1,(-15,400))
    if crypttype == "ablation"
        ylims!(ax2,(-50,750))
    else
        ylims!(ax2,(-50,1100))
    end

    return fig
end

function colour(a::stem)
    if a.phase == "arrest"
        return :grey
    elseif a.phase == "dying"
        return :brown4
    elseif a.boundwnt > 63.9
        return :yellow
    else
        return :yellow4
    end
end

function colour(a::absprog)
    if a.phase == "arrest"
        return :grey
    elseif a.phase == "dying"
        return :brown4
    else
        return :orange
    end
end

function colour(a::paneth)
    if a.phase == "dying"
        return :brown4
    else
        return :blue
    end
end

function colour(a::entero)
    if a.phase == "dying"
        return :brown4
    else
        return :red
    end
end

function colour(a::secprog)
    if a.phase == "dying"
        return :brown4
    else
        return :green
    end
end

function colour(a::entendo)
    if a.phase == "dying"
        return :brown4
    else
        return :turquoise
    end
end

function colour(a::goblet)
    if a.phase == "dying"
        return :brown4
    else
        return :purple
    end
end

function postprocess(cells,villus,incellcounts,invilluscounts,indose,indoseschedule,inki67,inbrdu,incelldeets,intmax)
    sort!(cells, by = v -> v.pos[3])
    global finalcells = cells
    global finalvillus = villus
    open("finalcells.txt", "w") do file
        println(file, finalcells)
    end

    global tmax = intmax
    global tee = range(0, tmax, length=size(incellcounts,2))
    global cellcounts = incellcounts
    global villuscounts = invilluscounts
    global dose = indose
    global doseschedule = indoseschedule
    global ki67 = inki67
    global brdu = inbrdu
    global celldeets = incelldeets
    global dose = indose

    global finalx = Vector{Float64}(undef,length(cells))
    global finaly = Vector{Float64}(undef,length(cells))
    global finalz = Vector{Float64}(undef,length(cells))
    global wntvec = Vector{Float64}(undef,length(cells))
    global splittimevec = Vector{Float64}(undef,length(cells))
    global notchvec = Vector{Float64}(undef,length(cells))
    global zvelvec = Vector{Float64}(undef,length(cells))
    global colours = Vector{Symbol}(undef,length(cells))
    global cellradvec = Vector{Float64}(undef,length(cells))

    for i in eachindex(cells)
        finalx[i] = cells[i].pos[1]
        finaly[i] = cells[i].pos[2]
        finalz[i] = cells[i].pos[3]
        wntvec[i] = log(2,cells[i].boundwnt)
        splittimevec[i] = cells[i].splittime*24
        notchvec[i] = cells[i].notch
        zvelvec[i] = cells[i].vel[3]/24
        cellradvec[i] = cells[i].cellrad
        colours[i] = colour(cells[i])
    end

    println("Crypt Averages")
    println("LGR5+ Stem = ", Statistics.mean(cellcounts[1,:]))
    println("Undecided prog. = ", Statistics.mean(cellcounts[2,:]))
    println("Abs. prog. = ", Statistics.mean(cellcounts[3,:]))
    println("Entero. = ", Statistics.mean(cellcounts[4,:]))
    println("Sec. prog. = ", Statistics.mean(cellcounts[5,:]))
    println("Goblet = ", Statistics.mean(cellcounts[6,:]))
    println("Paneth = ", Statistics.mean(cellcounts[7,:]))
    println("Entendo. = ", Statistics.mean(cellcounts[8,:]))
    println("Arrested = ", Statistics.mean(cellcounts[9,:]))
    println("Dying = ", Statistics.mean(cellcounts[10,:]))
    println("Total = ", Statistics.mean(cellcounts[11,:]))

    println("Villus Averages")
    println("Abs. prog. = ", Statistics.mean(villuscounts[3,:]))
    println("Entero. = ", Statistics.mean(villuscounts[4,:]))
    println("Goblet = ", Statistics.mean(villuscounts[6,:]))
    println("Entendo. = ", Statistics.mean(villuscounts[8,:]))
    println("Arrested = ", Statistics.mean(villuscounts[9,:]))
    println("Dying = ", Statistics.mean(villuscounts[10,:]))
    println("Total = ", Statistics.mean(villuscounts[11,:]))

    return nothing
end

function timereadout(inc,cells,tmax)
    if inc % Int(tmax/(10*dt)) == 0
        println(round(Int,inc*dt*100/tmax), "%, i have ", length(cells), " cells")
    end
end

function gimmeindex(a::cell,cnt,nuch,top)
    unit = 10*top/cnt
    nichelevels = 1 + floor(Int,nuch/10)
    nicheunit = circum/(4*nichelevels)
    if a.pos[3] > 0
        return floor(Int,a.pos[3]/unit) + nichelevels + 1
    else
        return floor(Int,(rad/nicheunit)*(pi - acos(a.pos[3]/rad))+1)
    end
end

function deetscount(a,celldeets,index)
    if typeof(a) == stem && a.boundwnt >= 64.0
        celldeets[1,index] += 1
    elseif typeof(a) == stem
        celldeets[2,index] += 1
    elseif typeof(a) == paneth
        celldeets[3,index] += 1
    elseif typeof(a) == absprog
        celldeets[4,index] += 1
    elseif typeof(a) == entero
        celldeets[5,index] += 1
    elseif typeof(a) == secprog
        celldeets[6,index] += 1
    elseif typeof(a) == goblet
        celldeets[7,index] += 1
    elseif typeof(a) == entendo
        celldeets[8,index] += 1
    end

    return nothing
end

function initcounts(tmax,lenny,mintop)
    cellcounts = Array{Int,2}(undef,11,ceil(Int,tmax*20)+1)
    villuscounts = Array{Int,2}(undef,11,ceil(Int,tmax*20)+1)
    fiveFUtrack = zeros(Float64,7,ceil(Int,tmax*20)+1)
    ki67 = zeros(Int,2,Int(mintop*5),ceil(Int,tmax*24*3))
    brdu = zeros(Int,2,Int(mintop*5),ceil(Int,tmax*24*3))
    celldeets = zeros(Int,8,Int(mintop*5))

    return cellcounts, villuscounts, fiveFUtrack, ki67, brdu, celldeets
end

function fancyinitcounts(tmax,lenny,mintop)
    cellcounts = Array{Int}(undef,11,ceil(Int,tmax*200)+1)
    villuscounts = Array{Int}(undef,11,ceil(Int,tmax*200)+1)
    fiveFUtrack = zeros(Float64,7,ceil(Int,tmax*200)+1)
    ki67 = zeros(Int,2,Int(mintop*5),ceil(Int,tmax*24*3))
    brdu = zeros(Int,2,Int(mintop*5),ceil(Int,tmax*24*3))
    celldeets = zeros(Int,8,Int(mintop*5))

    return cellcounts, villuscounts, fiveFUtrack, ki67, brdu, celldeets
end

function detailsplot(res = (1000,600))
    fig = Figure(resolution = res)
    ax1 = fig[1, 1] = Axis(fig, yticks = 0:0.25:1,xlabel = "Cell position", ylabel = "Proportion")
    colours = [:yellow,:yellow4,:blue,:orange,:red,:green,:purple,:turquoise]
    normcelldeets = zeros(size(celldeets))
    for i in eachindex(celldeets[1,:])
        normcelldeets[:,i] .= celldeets[:,i]./sum(celldeets[:,i])
    end
    for i in 8:-1:1
        plotmeh = normcelldeets[1,:]
        for j in 1:i
            if j > 1
                plotmeh += normcelldeets[j,:]
            end
        end
        barplot!(ax1,1:length(plotmeh),plotmeh,color = colours[i], linewidth=3)
    end
    xlims!(ax1,[0.5, 26.5])
    return fig
end

function ki67plot(res = (900,600))
    fig = Figure(resolution = res)
    ax1 = fig[1, 1] = Axis(fig, xlabel = "Cell position", ylabel = "Proportion", xticklabelsize = 24, yticklabelsize = 24)

    if dose != 0
        shift = Int(72 * first(doseschedule))
    else
        shift = 0
    end
    lenny = length(ki67[1,:,1])

    # lines!(ax1,1:lenny,(ki67[:,1,1] ./ ki67[:,2,1]),color=:yellow,linewidth=3)
    lines!(ax1,1*(1:lenny),(ki67[1,:,18+shift] ./ ki67[2,:,18+shift]),color=:blue,linewidth=3)
    lines!(ax1,1*(1:lenny),(ki67[1,:,72+shift] ./ ki67[2,:,72+shift]),color=:purple,linewidth=3)
    lines!(ax1,1*(1:lenny),(ki67[1,:,288+shift] ./ ki67[2,:,288+shift]),color=:red,linewidth=3)
    lines!(ax1,1*(1:lenny),(ki67[1,:,432+shift] ./ ki67[2,:,432+shift]),color=:orange,linewidth=3)
    
    xlims!(ax1,[0, 25])
    return fig
end

function brduplot(scale=1,res = (843,562))
    fig = Figure(resolution = res)
    ax1 = fig[1, 1] = Axis(fig, xlabel = "Cell position", ylabel = "Proportion", xticklabelsize = 22, yticklabelsize = 22)

    lines!(ax1,1:length(brdu[1,:,1]),(brdu[1,:,round(Int,6*scale)] ./ brdu[2,:,round(Int,6*scale)]),color=:blue,linewidth=3)
    lines!(ax1,1:length(brdu[1,:,1]),(brdu[1,:,round(Int,72*scale)] ./ brdu[2,:,round(Int,72*scale)]),color=:purple,linewidth=3) # 72
    lines!(ax1,1:length(brdu[1,:,1]),(brdu[1,:,round(Int,240*scale)] ./ brdu[2,:,round(Int,240*scale)]),color=:red,linewidth=3) # 240
    
    xlims!(ax1,[0,25])
    return fig
end

function FUplot(res = (1000,600), loggy = true)
    fig = Figure(resolution = res)
    teep = range(0, 10, length=length(fiveFUtrack[1,:]))
    ax1 = fig[1, 1] = Axis(fig, xlabel = "Time/days", ylabel = "log(Concentration+1)/(ng/ml)", yscale = log10)

    lines!(ax1,teep .- 1 ,1 .+fiveFUtrack[1,:],colormap=:seaborn_colorblind,linewidth=3)
    lines!(ax1,teep.- 1 ,1 .+fiveFUtrack[5,:],linewidth=3)
    lines!(ax1,teep.- 1 ,1 .+fiveFUtrack[7,:],linewidth=3)
    xlims!(ax1,[-0.5,7])
    return fig
end

function correcteddiv(x,y)
    if y == 0
        return 0
    else
        return x/y
    end
end

function ki67avgplot(res = (900,600),shadow=true)
    fig = Figure(resolution = res)
    avgki67 = zeros(length(ki67[1,:,1]))
    for i in eachindex(avgki67)
        avgki67[i] = mean(correcteddiv.(ki67[1,i,:],ki67[2,i,:]))
    end
    
    ax1 = fig[1, 1] = Axis(fig, xlabel = "Cell position", ylabel = "Proportion", xticklabelsize = 22, yticklabelsize = 22)
    lines!(ax1,1:length(avgki67),avgki67,color=:firebrick3,linewidth=5)
    xlims!(ax1,[0, 25])
    return fig
end

function drugcalc(inc,tmax,doseschedule,exposureduration)
    i = searchsortedlast(doseschedule,inc*dt)
    if i > 0
        return interval(inc*dt,doseschedule[i],doseschedule[i]+exposureduration)
    else
        return .0
    end
end

function initfiveFU(dose,OGdoseschedule)
    if in(.0,OGdoseschedule)
        k_a=1123
        return @SVector[dose * k_a / (dt * 24),.0,.0,.0,.0,.0,.0]
    else
        return @SVector zeros(7)
    end
end

function fiveFloop(fiveFU,dose,doseschedule,inc)
    mouse=0.025

    # Metabolite rates
    ke_FUTP=1.010
    k_FdUTP=3.154
    k_FUMP=1.555
    k_FUDP=0.2973
    k_FUTP=1.010
    k_FdUMP=0.1452


    # Pk 2 compartments (palsma, peridfery  from simcyp data*/
    k_a=1123
    ke_5FUplas=5.625-k_FUMP
    k_per=2.160
    k_plasfromper=1.516


    if in(inc,doseschedule) && dose > 0
        druggy = dose * k_a / (dt * 24)
    else
        druggy = .0
    end

    dFUpl=druggy-ke_5FUplas*fiveFU[1]-k_per*fiveFU[1]+k_plasfromper*fiveFU[2]
    dFUper=k_per*fiveFU[1]-k_plasfromper*fiveFU[2]

    dFUMP=k_FUMP*fiveFU[1]- k_FUDP*fiveFU[3]
    dFUDP= k_FUDP*fiveFU[3]-k_FUTP*fiveFU[4]
    dFUTP=k_FUTP*fiveFU[4]-ke_FUTP*fiveFU[5]

    dFdUMP = (k_FdUMP*fiveFU[4] - fiveFU[6])/(24*dt)
    dFdUTP = (k_FdUTP*fiveFU[4] - fiveFU[7])/(24*dt)

    dfiveFU = @SVector [dFUpl,dFUper,dFUMP,dFUDP,dFUTP,dFdUMP,dFdUTP]

    return dfiveFU
end

function fiveFUdrugeff(X,drug)
    dRNA = -30.264
    Krna = 1611
    m = 1.939

    dDNA = -50.81
    Kdna = 13090
    n = 5.8242
    return @SVector[.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0, dDNA * X[15] * (drug[7]^n/(drug[7]^n+Kdna^n)), dDNA * X[16] * (drug[7]^n/(drug[7]^n+Kdna^n)), dRNA * X[17] * (drug[5]^m/(drug[5]^m+Krna^m))]
end

function cdkdrugeff(X,drug)
    aeff = 70
    beff = 2*aeff
    return @SVector [-aeff*X[1]*drug, -beff*X[2]*drug,.0,.0,-beff*X[5]*drug,.0,.0,.0,.0,.0,-beff*X[11]*drug,-beff*X[12]*drug,-aeff*X[13]*drug,.0,.0,.0,.0]
end

function cellloopablat(cells,grid,BMP,top,zent,znrf3vec,drug,celltype,numniche,deff,secdeff,enterodeff)
    @inbounds for i in eachindex(cells)
        emergencynotchscan(cells[i],cells,grid,i)

        loopcontentablat(cells[i],BMP,top,zent,znrf3vec,drug,celltype,numniche)

        differentiate(cells,i,deff,secdeff,enterodeff)
    end
    return nothing
end

function loopcontentablat(a,BMP,top,zent,znrf3vec,drug,celltype,numniche)
    setsplittime(a)
    nichewntacc(a,numniche)

    emergencyfatedecisions(a,BMP,top,zent)

    if drug == 1 && typeof(a) == celltype && a.phase != "dying"
        a.phase = "dying"
        a.rstar = a.cellrad
        a.cyclevec = @SVector [.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,.0,a.cyclevec[15],a.cyclevec[16],a.cyclevec[17]]
    end

    cellcycle(a,BMP,top,zent)

    notchdecay(a)
    wntdecay(a,znrf3vec)

    mitosiscomplete(a)
    movecells(a)

    agecells(a)
    begindeathshrink(a)
    popcell(a)

    return nothing
end

function cellloop5FU(cells,grid,BMP,top,zent,znrf3vec,drug,numniche,deff,secdeff,enterodeff)
    @inbounds for i in eachindex(cells)
        emergencynotchscan(cells[i],cells,grid,i)

        loopcontent5FU(cells[i],BMP,top,zent,znrf3vec,drug,numniche)

        differentiate(cells,i,deff,secdeff,enterodeff)
    end
    return nothing
end

function loopcontent5FU(a,BMP,top,zent,znrf3vec,drug,numniche)
    setsplittime(a)
    nichewntacc(a,numniche)

    emergencyfatedecisions(a,BMP,top,zent)

    cellcycle5FU(a,BMP,top,drug,zent)

    notchdecay(a)
    wntdecay(a,znrf3vec)

    mitosiscomplete(a)
    movecells(a)

    agecells(a)
    begindeathshrink(a)
    popcell(a)

    return nothing
end

function cellloopCDK(cells,grid,BMP,top,zent,znrf3vec,drug,numniche,deff,secdeff,enterodeff)
    @inbounds for i in eachindex(cells)
        emergencynotchscan(cells[i],cells,grid,i)

        loopcontentCDK(cells[i],BMP,top,zent,znrf3vec,drug,numniche)

        differentiate(cells,i,deff,secdeff,enterodeff)
    end
    return nothing
end

function loopcontentCDK(a,BMP,top,zent,znrf3vec,drug,numniche)
    setsplittime(a)
    nichewntacc(a,numniche)

    emergencyfatedecisions(a,BMP,top,zent)

    cellcycleCDK(a,BMP,top,drug,zent)

    notchdecay(a)
    wntdecay(a,znrf3vec)

    mitosiscomplete(a)
    movecells(a)

    agecells(a)
    begindeathshrink(a)
    popcell(a)

    return nothing
end

function brdusuperloop(intcells,intvillus,consts,tmax)
    cells = deepcopy(intcells)
    villus = deepcopy(intvillus)
    fiveFU = zeros(1:7)
    cellcounts, villuscounts, fiveFUtrack, ki67, brdu, celldeets = initcounts(tmax,length(cells),consts.mintop)
    fiveFUtrack[:,1] = fiveFU[:]
    updatedacounts(cells,villus,cellcounts,villuscounts,0,tmax,consts.mintop)

    dose = .0
    doseschedule = [0]
    indoseschedule = [0]
    
    for i in eachindex(cells)
        cells[i].ID = [0]
    end

    for i in 1:ceil(Int,tmax/dt)+1
        if i <= (139*3)
            brdustain(cells,i)
        end
        fiveFU = mainloopbrdu(consts,cells,villus,cellcounts,villuscounts,fiveFU,fiveFUtrack,dose,indoseschedule,ki67,brdu,celldeets,tmax,i)
    end
    return brdu, cellcounts, villuscounts
end

function ki67superloop(intcells,intvillus,consts,tmax)
    dose = 50
    doseschedule = [1.0,1.5,2.0,2.5,3.0,3.5,4.0,4.5]
    cells = deepcopy(intcells)
    villus = deepcopy(intvillus)
    fiveFU = initfiveFU(dose,doseschedule)
    cellcounts, villuscounts, fiveFUtrack, ki67, brdu, celldeets = initcounts(tmax,length(cells),consts.mintop)
    fiveFUtrack[:,1] = fiveFU[:]
    updatedacounts(cells,villus,cellcounts,villuscounts,0,tmax,consts.mintop)
    indoseschedule = round.(doseschedule/dt,digits=0)
    indose = deepcopy(dose)

    for i in 1:ceil(Int,tmax/dt)+1
        fiveFU = mainloopbrdu(consts,cells,villus,cellcounts,villuscounts,fiveFU,fiveFUtrack,dose,indoseschedule,ki67,brdu,celldeets,tmax,i)
    end
    return ki67, cellcounts, villuscounts
end


function mainloopbrdu(consts,cells,villus,cellcounts,villuscounts,fiveFU,fiveFUtrack,dose,doseschedule,ki67,brdu,celldeets,tmax,inc)
    numniche = count(x->x.pos[3]<.0,cells)
    enteronum = count(x->typeof(x)==entero,villus)
    BMP = BMPcalc(enteronum,consts.nent,consts.p)
    top = topcalc(enteronum,consts.mintop,consts.nent)
    znrf3vec = znrf3calc(cells)

    dfiveFU = fiveFloop(fiveFU,dose,doseschedule,inc)
    fiveFU += dt * 24 * dfiveFU

    cellremoval(cells,villus,top,consts.villuslength)

    birthcells(cells,consts.deff)
    villusbirthcells(villus,consts.deff)

    grid = makegrid(cells)

    forcesandsig(grid,cells)

    cellloop5FU(cells,grid,BMP,top,consts.zent,znrf3vec,fiveFU,numniche,consts.deff,consts.secdeff,consts.enterodeff)

    villusloop(villus,BMP,top,consts.zent,consts.deff,consts.secdeff,consts.enterodeff)

    updates(cells,villus,cellcounts,villuscounts,fiveFUtrack,ki67,brdu,celldeets,inc,tmax,top,fiveFU=fiveFU)

    return fiveFU
end

function mainloop5FU(consts,cells,villus,cellcounts,villuscounts,fiveFU,fiveFUtrack,dose,doseschedule,ki67,brdu,celldeets,tmax,inc)
    numniche = count(x->x.pos[3]<.0,cells)
    enteronum = count(x->typeof(x)==entero,villus)
    BMP = BMPcalc(enteronum,consts.nent,consts.p)
    top = topcalc(enteronum,consts.mintop,consts.nent)
    znrf3vec = znrf3calc(cells)

    dfiveFU = fiveFloop(fiveFU,dose,doseschedule,inc)
    fiveFU += dt * 24 * dfiveFU

    cellremoval(cells,villus,top,consts.villuslength)

    birthcells(cells,consts.deff)
    villusbirthcells(villus,consts.deff)

    grid = makegrid(cells)

    forcesandsig(grid,cells)

    cellloop5FU(cells,grid,BMP,top,consts.zent,znrf3vec,fiveFU,numniche,consts.deff,consts.secdeff,consts.enterodeff)

    villusloop(villus,BMP,top,consts.zent,consts.deff,consts.secdeff,consts.enterodeff)

    timereadout(inc,cells,tmax)

    updates(cells,villus,cellcounts,villuscounts,fiveFUtrack,ki67,brdu,celldeets,inc,tmax,top,fiveFU=fiveFU)

    return fiveFU
end

function mainloopCDK(consts,cells,villus,cellcounts,villuscounts,fiveFUtrack,exposureduration,doseschedule,ki67,brdu,celldeets,tmax,inc)
    numniche = count(x->x.pos[3]<.0,cells)
    enteronum = count(x->typeof(x)==entero,villus)
    BMP = BMPcalc(enteronum,consts.nent,consts.p)
    top = topcalc(enteronum,consts.mintop,consts.nent)
    znrf3vec = znrf3calc(cells)

    drug = drugcalc(inc,tmax,doseschedule,exposureduration)

    cellremoval(cells,villus,top,consts.villuslength)

    birthcells(cells,consts.deff)
    villusbirthcells(villus,consts.deff)

    grid = makegrid(cells)

    forcesandsig(grid,cells)

    cellloopCDK(cells,grid,BMP,top,consts.zent,znrf3vec,drug,numniche,consts.deff,consts.secdeff,consts.enterodeff)

    villusloop(villus,BMP,top,consts.zent,consts.deff,consts.secdeff,consts.enterodeff)

    timereadout(inc,cells,tmax)

    updates(cells,villus,cellcounts,villuscounts,fiveFUtrack,ki67,brdu,celldeets,inc,tmax,top)
    return nothing
end

function mainloopablat(consts,cells,villus,cellcounts,villuscounts,fiveFUtrack,celltype,exposureduration,doseschedule,ki67,brdu,celldeets,tmax,inc)
    numniche = count(x->x.pos[3]<.0,cells)
    enteronum = count(x->typeof(x)==entero,villus)
    BMP = BMPcalc(enteronum,consts.nent,consts.p)
    top = topcalc(enteronum,consts.mintop,consts.nent)
    znrf3vec = znrf3calc(cells)

    drug = drugcalc(inc,tmax,doseschedule,exposureduration)

    cellremoval(cells,villus,top,consts.villuslength)

    birthcells(cells,consts.deff)
    villusbirthcells(villus,consts.deff)

    grid = makegrid(cells)

    forcesandsig(grid,cells)

    cellloopablat(cells,grid,BMP,top,consts.zent,znrf3vec,drug,celltype,numniche,consts.deff,consts.secdeff,consts.enterodeff)

    villusloop(villus,BMP,top,consts.zent,consts.deff,consts.secdeff,consts.enterodeff)

    timereadout(inc,cells,tmax)

    updates(cells,villus,cellcounts,villuscounts,fiveFUtrack,ki67,brdu,celldeets,inc,tmax,top)
    return nothing
end
