function fpar=fitzmann(v,g,i1,i2,constrain)

% INPUTS
%           v ... nx1 vector, independent variable (voltage)
%           g ... nx1 vector, dependent variable (conductance)
%           i1 ... first point to fit, [1,n-3)
%           i2 ... last point to fit, (3,n]
%           constrain ... true/false, if user wants to constrain gmin to 0

% OUTPUTS
%           fpar ... 1x5 vector of Xhalf, dx, gmin, gmax, RSQRD

% Define the fitting function
boltz = fittype( @(a, b, vhf, d, x) a+((b-a)./(1+exp((x-vhf)./d))), 'independent', 'x', 'dependent', 'y', 'coefficients', {'a', 'b', 'vhf', 'd'});

% Fitzmann will plot onto the current axes
hold on

% Plot all points in blue first
plot(v, g, '.b', 'MarkerSize', 7, 'DisplayName', 'excld');
ylim([-1 1.1*max(g)])

% Asks user to select the range of points to fit
if isempty(i1)
    disp('waiting for user input...')
    [ix, ~] = ginput(2);
    i1 = find(v > ix(1), 1, 'first');
    i2 = find(v < ix(2), 1, 'last');

end

ftv = v(i1:i2);
ftg = g(i1:i2);

% Calculating initial guesses. This *should* be irrespective of 
% inactivation or activation if similar voltage dependence trends
a1=max([g(i1),g(i2)]);
a2=min([g(i1),g(i2)]);
Vh=median(ftv);
dx=5;

% Optional user input to contrain gmin to 0
if contains(constrain,'true')
    val=0;
else
    val=400;
end


% Run the fit
ftcurve = fit(ftv, ftg, boltz, 'StartPoint', [a1,a2,Vh,dx],...
    'Upper',[Inf val 10 25],'Lower',[-Inf -val -120 0]);

%Extract the fit parameters
coeff = coeffvalues(ftcurve);
% Re-order fpar for easy copy-paste
fpar=coeff;
fpar(1) = coeff(3);
fpar(2) = coeff(4);
fpar(3) = coeff(2);
fpar(4) = coeff(1);

% Plot the fit curve
plot(ftcurve)

% Plot only the points used in the fit
plot(ftv, ftg, '.k', 'MarkerSize', 8, 'DisplayName', 'fitd')
xlabel('step voltage (mV)')
ylabel('conductance (nS)')
legend off

gmax = max(fpar(3:4)) + 1; %a or h insensitive
%ylim([-0.5 gmax]); %I removed this bcz bug

%generating fitted curve for RSQRD calculation
yy = coeff(1) + (coeff(2) - coeff(1))./(1 + exp((ftv - coeff(3))./coeff(4)));

% Calculate RSQRD
R2=1-sum((ftg-yy).^2)/sum((ftg-mean(ftg)).^2);
%R2adj=1-(1-R2)*(numel(ftg)-1)/(numel(ftg)-numel(fpar)-1);

fpar=[fpar R2]; %fpar is now a 1x5 vector




end