classdef SpcCutter < MClust.Cutter
    %
    % Select clusters from those generated by KKwik
    %
    % Version control: ADR-2013-12-12
    
    properties
        figHandle_AvgWV = [];
        figHandle_ISI = [];
        whoHasFocus = [];
        whoHasComparison = [];
        rootCluster = [];
        C = [];
        
        nextClusterButton = [];
        prevClusterButton = []
    end
    
    methods
        % -------------------------------
        % Constructor/Destructor/Import/Export
        % -------------------------------
        function self = SpcCutter()
            OK = self.LoadClusters();
        end

        
        function exportClusters(self)
            curNode = self.rootCluster;
            iC = 1;
            while ~isempty(curNode)
                tval = get(curNode, 'Value'); %get the value of this cluster
                if strcmp(tval(1), 'selected') %if this cluster is selected, keep it
                    if tval(2) > 1
                        B{iC} = MClust.ClusterTypes.SpikelistCluster; %make a new cluster
                        B{iC}.name = ['Cluster ' num2str(iC)];
                        B{iC}.AddSpikes(self.C{tval(2)-1}+1); %add the list of spikes
                        iC = iC + 1;
                    end
                end
                curNode = get(curNode, 'NextNode'); %move on to the next node in the tree
            end
            colors = hsv;
            for i = 1:length(B) %give clusters colors
                B{i}.color = colors(ceil(length(colors)/length(B)*i),:);
            end
            self.exportClusters@MClust.Cutter(B);
        end
        
        function close(self)
            MCS = MClust.GetSettings();
            if ~isvalid(self), return; end
            if ~isempty(self.figHandle_AvgWV) && ishandle(self.figHandle_AvgWV)
                MCS.StoreWindowPlace(self.figHandle_AvgWV); % ADR 2013-12-12
                delete(self.figHandle_AvgWV);
            end
            if ~isempty(self.figHandle_ISI) && ishandle(self.figHandle_ISI)
                MCS.StoreWindowPlace(self.figHandle_ISI); % ADR 2013-12-12
                delete(self.figHandle_ISI);
            end
            self.close@MClust.Cutter();
        end
        
		function ReGo(self)
            %do nothing for now
		end
        
        % ---------------------------------
        % Display and callbacks
        % ---------------------------------
        function SetFocus(self, C)
            for iC = 1:length(self.Clusters)
                if ~isequal(self.Clusters{iC}, C)
                    self.Clusters{iC}.LoseFocus(C);
                end
            end
            self.whoHasFocus = C;
        end
        
        function SetComparison(self, C)
            for iC = 1:length(self.Clusters)
                if ~isequal(self.Clusters{iC}, C)
                    self.Clusters{iC}.LoseComparison(C);
                end
            end
            self.whoHasComparison = C;
        end
        
        function iFocus = FindFocus(self)
            iFocus = [];
            for iC = 1:length(self.Clusters)
                if isequal(self.whoHasFocus, self.Clusters{iC})
                    iFocus = iC;
                end
            end
        end
        
        function RedisplayAvgWV(self)
            MCS = MClust.GetSettings();
            figure(self.figHandle_AvgWV);
            errorbar(self.whoHasFocus.xrange, self.whoHasFocus.mWV, self.whoHasFocus.sWV, 'b');
            set(gca, 'YLim',MCS.AverageWaveform_ylim);
            if ~isequal(self.whoHasComparison, self.whoHasFocus)
                hold on
                plot(self.whoHasComparison.xrange, self.whoHasComparison.mWV, 'r');
                hold off
            end
        end
        
        function RedisplayISI(self)
            MCD = MClust.GetData();
            figure(self.figHandle_ISI);
            cla
            T = MCD.FeatureTimestamps(self.whoHasFocus.GetSpikes());
            if length(T)>2 % check for 1-spike clusters ADR 2013-12-12
                MClust.HistISI(T, 'axesHandle', gca, 'myColor', 'b');
            end
            if ~isequal(self.whoHasComparison, self.whoHasFocus)
                hold on
                T = MCD.FeatureTimestamps(self.whoHasComparison.GetSpikes());
                if length(T)>2 % check for 1-spike clusters ADR 2013-12-12
                    MClust.HistISI(T, 'axesHandle', gca, 'myColor', 'r');
                end
                hold off
            end
            
        end
        
        % ---------------------------------
        % Next/Prev
        % ---------------------------------
        function NextCluster(self)
            nC = length(self.Clusters);
            iFocus = self.FindFocus+1;
            if iFocus > nC, beep; iFocus = nC; end
            self.Clusters{iFocus}.TakeFocus;
        end
        
        function PrevCluster(self)
            nC = length(self.Clusters);
            iFocus = self.FindFocus - 1;
            if iFocus < 1, beep; iFocus = 1; end
            self.Clusters{iFocus}.TakeFocus;
        end
        
        
        % ---------------------------------
        % Load/Save
        % ---------------------------------
        function OK = LoadClusters(self)
            MCD = MClust.GetData();
            
            disp('Running SPC...')
            drawnow;
            
            % Get the features of each spike from MClust
            Np = length(MCD.Features{1}.GetData); %number of points
            Nf = length(MCD.Features); %number of features
            F = zeros(Np, Nf);
            for i = 1:Nf
                F(:,i) = MCD.Features{i}.GetData;
            end
            
            % Ask min cluster size
            def_mcs = round(Np/100); %default is Npoints/100
            MCS_str = inputdlg('Enter desired minimum cluster size:', ...
                              'Min. Cluster Size', 1, {num2str(def_mcs)});
            MIN_CS = str2double(MCS_str{1});
            
            % Run SPC
            tic;
            [C, P] = spc_mex(F', MIN_CS);
            toc
            
            % Add clusters from SPC to cutter clusters to look at
            nClu = length(C);
            disp([' Found ' num2str(nClu) ' clusters'])
            self.Clusters = cell(1,nClu+1);
            for iC = 1:nClu %for each cluster
                self.Clusters{iC+1} = MClust.ClusterTypes.SpikelistCluster; %create new cluster
                self.Clusters{iC+1}.SetSpikes(C{iC}+1); %set it to contain spikes in C
                self.Clusters{iC+1}.setAssociatedCutter(@self.GetCutter);
                self.Clusters{iC+1}.hide = true;
            end

            % And make the cluster of all points
            self.Clusters{1} = MClust.ClusterTypes.SpikelistCluster; %create new cluster
            self.Clusters{1}.SetSpikes(1:size(F,1)); %set it to contain all spikes
            self.Clusters{1}.setAssociatedCutter(@self.GetCutter);
            self.Clusters{1}.hide = true;

            % Draw the clusters in the cutting control window
            self.rootCluster = self.RedrawClusterTree(C, P, F);
            self.C = C;
            
            % Ready to go!
            OK = true;
            
        end %LoadClusters
        
        function [rootNode] = RedrawClusterTree(self, C, P, F)
            import javax.swing.*
            import javax.swing.tree.*;

            % Make the icons
            [I,map] = checkedIcon;
            javaImage_checked = im2java(I,map);
            [I,map] = uncheckedIcon;
            javaImage_unchecked = im2java(I,map);
            iconWidth = javaImage_unchecked.getWidth;

            % Create nodes
            rootNode = uitreenode('v0','root', 'All Points', [], 0);
            rootNode.setIcon(javaImage_unchecked);
            a = cell(2,1); a{1} = 'unselected'; a{2} = 1;
            rootNode.setValue(a);
            E = cell(1+length(P), 1); %list of tree objects
            E{1} = rootNode;
            for i = 1:length(P) %add each recursively
                tstr = ['Cluster ' num2str(i)];
                E{i+1} = uitreenode('v0','unselected', tstr, [], 0);
                E{i+1}.setIcon(javaImage_unchecked);
                E{P(i)+2}.add(E{i+1}); %add it to the tree
                a = cell(2,1); a{1} = 'unselected'; a{2} = i+1; % first value is select/unselected, second is this node's index
                E{i+1}.setValue(a); %save values for this node

            end

            % Create the tree
            treeModel = DefaultTreeModel( rootNode );
            tree = uitree('v0');
            tree.setModel( treeModel );
            jtree = handle(tree.getTree,'CallbackProperties');
            drawnow;
            set(tree, 'Units', 'normalized', 'position', [0.4 0 0.6 1]);
            tree.setSelectedNode( E{1} );
            set(jtree, 'MousePressedCallback', @mousePressedCallback);
            
            
            function mousePressedCallback(hTree, eventData) %,additionalVar)
                % Get the clicked node
                clickX = eventData.getX;
                clickY = eventData.getY;
                treePath = jtree.getPathForLocation(clickX, clickY);
                % check if a node was clicked
                if ~isempty(treePath)
                  % check if the checkbox was clicked
                  if clickX <= (jtree.getPathBounds(treePath).x+iconWidth)
                    node = treePath.getLastPathComponent;
                    nodeValue = node.getValue;
                    % as the value field is the selected/unselected flag,
                    % we can also use it to only act on nodes with these values
                    switch nodeValue(1)
                      case 'selected'
                        node.setIcon(javaImage_unchecked);
                        jtree.treeDidChange();
                        self.Clusters{nodeValue(2)}.hide = true; %set corresponding cluster to hidden
                        a = cell(2,1); a{1} = 'unselected'; a{2} = nodeValue(2);
                        node.setValue(a);
                        self.RedrawAxes(); %and re-draw the axes
                      case 'unselected'
                        node.setIcon(javaImage_checked);
                        jtree.treeDidChange();
                        self.Clusters{nodeValue(2)}.hide = false; %set corresponding cluster to viewable
                        a = cell(2,1); a{1} = 'selected'; a{2} = nodeValue(2);
                        node.setValue(a);
                        self.RedrawAxes(); %and re-draw the axes
                    end
                  end
                end
            end % function mousePressedCallback
            
            function [I,map] = checkedIcon()
                I = uint8(...
                    [1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0;
                     2,2,2,2,2,2,2,2,2,2,2,2,2,0,0,1;
                     2,2,2,2,2,2,2,2,2,2,2,2,0,2,3,1;
                     2,2,1,1,1,1,1,1,1,1,1,0,2,2,3,1;
                     2,2,1,1,1,1,1,1,1,1,0,1,2,2,3,1;
                     2,2,1,1,1,1,1,1,1,0,1,1,2,2,3,1;
                     2,2,1,1,1,1,1,1,0,0,1,1,2,2,3,1;
                     2,2,1,0,0,1,1,0,0,1,1,1,2,2,3,1;
                     2,2,1,1,0,0,0,0,1,1,1,1,2,2,3,1;
                     2,2,1,1,0,0,0,0,1,1,1,1,2,2,3,1;
                     2,2,1,1,1,0,0,1,1,1,1,1,2,2,3,1;
                     2,2,1,1,1,0,1,1,1,1,1,1,2,2,3,1;
                     2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                     2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,1;
                     2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,1;
                     1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1]);
                 map = [0.023529,0.4902,0;
                        1,1,1;
                        0,0,0;
                        0.50196,0.50196,0.50196;
                        0.50196,0.50196,0.50196;
                        0,0,0;
                        0,0,0;
                        0,0,0];
            end

            function [I,map] = uncheckedIcon()
                 I = uint8(...
                   [1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1;
                    2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1;
                    2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,1,1,1,1,1,1,1,1,1,1,2,2,3,1;
                    2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,1;
                    2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,1;
                    1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1]);
                 map = ...
                  [0.023529,0.4902,0;
                   1,1,1;
                   0,0,0;
                   0.50196,0.50196,0.50196;
                   0.50196,0.50196,0.50196;
                   0,0,0;
                   0,0,0;
                   0,0,0];
            end
            
        end  %PlotClusterTree
        

    end %methods
    
end % class

